	page	59,132
	title	UHDD -- DOS UltraDMA Disk Caching Driver.
;
; UHDD is a minimized disk-only variant of UIDE with these changes:
;
;   1) UHDD runs 10 controllers and 26 BIOS hard-disks or diskettes.
;
;   2) UHDD usually omits user caches and ignores /X or /Y switches.
;	 Data for a user caching call goes in UHDD's "Common" cache.
;
;   3) UHDD has no CD/DVD logic, always handles BIOS disks/diskettes
;	 and ignores /D: /N1 /N2 or /UX switches.
;
;   4) UHDD now "senses" XMS automatically and ignores a /N3 switch.
;
;   5) UHDD does not accept /N4 -- Use the /Z switch when necessary.
;
; Separate "User-1" and "User-2" caches require assembling UHDD with
; a /dCACHE4 switch.   This adds 64 bytes and changes its title name
; to "UHDD-UC", to indicate User Caches are available.
;
; For an example of how UHDD can be called by a user driver to cache
; data, see the UDVD2.ASM source file, also read the UIDE.TXT file.
;
;
; General Program Equations.
;
	.386p			;Allow use of 80386 instructions.
	.lfcond			;List conditionally-unassembled code.
s	equ	<short>		;Make a conditional jump "short".
MAXBIOS	equ	26		;Maximum 26 BIOS disks or diskettes.
MXCACHE	equ	4093		;Maximum cache size in megabytes.
STACK	equ	520		;Caching driver local-stack size.
STACKSA	equ	412		;"Cache only" & "stand alone" stack.
CDUNIT	equ	40		;CD/DVD initial cache-unit number.
CDTYP	equ	07Ch		;CD/DVD device-type code.
CMDTO	equ	00Ah		;Disk 500-msec min. command timeout.
STARTTO	equ	07Fh		;Disk 7-second startup timeout.
HDWRFL	equ	00410h		;BIOS hardware-installed flag.
DKTSTAT	equ	00441h		;BIOS diskette status byte.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
HDI_OFS	equ	0048Eh-BIOSTMR	;BIOS hard-disk int. flag "offset".
MCHDWFL	equ	0048Fh		;BIOS diskette media-change flags.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver Error Return Codes.
;
MCHANGE	equ	006h		;Diskette "media change".
DMAERR	equ	00Fh		;Hard disk DMA error.
CTLRERR equ	020h		;Hard disk controller not-ready.
DISKERR equ	0AAh		;Hard disk drive not-ready.
FAULTED	equ	0CCh		;Hard disk drive FAULTED.
HARDERR equ	0E0h		;Hard disk I-O error.
XMSERR	equ	0FFh		;XMS memory error.
;
; "Legacy IDE" Controller I-O Base Addresses.
;
NPDATA	equ	001F0h		;Normal primary      base address.
NSDATA	equ	00170h		;Normal secondary    base address.
APDATA	equ	001E8h		;Alternate primary   base address.
ASDATA	equ	00168h		;Alternate secondary base address.
;
; IDE Controller Register Offsets.
;
CDATA	equ	0		;Data port offset.
CDSEL	equ	6		;Disk-select and upper LBA offset.
CCMD	equ	7		;Command-register offset.
CSTAT	equ	7		;Primary-status register offset.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
MSEL	equ	0A0h		;"Master" device-select bits.
SSEL	equ	0B0h		;"Slave"  device-select bits.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; LBA "Device Address Packet" Layout.
;
DAP	struc
DapPL	db	?		;Packet length.
	db	?		;(Reserved).
DapSC	db	?		;I-O sector count.
	db	?		;(Reserved).
DapBuf	dd	?		;I-O buffer address (segment:offset).
DapLBA	dw	?		;48-bit logical block address (LBA).
DapLBA1	dd	?
DAP	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?		;(Segment must ALWAYS be set!).
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
RqNdx	dw	(I_Stra-@)	;"Strategy" routine pointer.
				;(Request cache-index, after Init).
XMSHndl	dw	(I_Init-@)	;Device-Interrupt routine pointer.
				;(XMS "handle" number, after Init).
	db	'UHDD'		;Driver name, normally "UHDD$", also
DNam2	db	'$',0,0,0	;  "UHDD-SA$" for stand-alone driver.
;
; General Driver "Data Page" Variables.
;
VLF	db	0		;VDS "lock" flag (001h = buffer lock).
IOF	db	0		;I-O control flags --
				;  Bit 1:  "Common" cache needs flush.
				;  Bit 0:  Driver is currently "busy".
DMAAd	dw	0		;DMA status/command register address.
IdeDA	dw	0		;IDE data register address.
SCStak	equ	[$].dwd		;"Stand alone" saved stack pointer.
RqSec	db	0		;Caching request I-O sector count.
RqCmd	db	0		;Caching I-O command bits.
RqCSC	db	0		;Caching current-block sector count.
RqTyp	db	0		;Caching device-type flag.
;
; VDS Parameter Block and UltraDMA Command-List.
;
VDSLn	dd	(LMEnd-@)	;VDS block length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;DMA 32-bit buffer address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
;
; "PRD" Address and IDE Parameter Area.
;
PRDAd	dd	(IOAdr-@)	;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0,0,0		;IDE "upper" LBA bits 24-47.
SecCt	db	0		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
WrkBf	equ	[$-2].lw	;LRU "work" buffer, shares IDE cmds.
;
; BIOS "Active Unit" Table.   The beginning of this table MUST be at
;   driver address 07Fh or below, for BP-reg. addressing at "Entry".
;
Units	db	MAXBIOS dup (0)	;BIOS unit number for each device.
SALowM	equ	$		;(Start of our "stand alone" logic).
SABASE	equ	(SALowM-@)	;(Size of base "stand alone" logic).
;
; Caching Variables.
;
RqCPA	dd	0		;Request cache parameter-table addr.
RqBuf	dd	0		;Request I-O buffer address.
RqXMS	dd	128		;Request XMS buffer offset.
RqLBA	dw	0,0		;Request initial LBA address.
RqUNo	equ	[$+2].lb	;Request "cache unit" number.
CBLBA	equ	[$+4].lw	;Current-buffer initial LBA.
CBUNo	equ	[$+10].lb	;Current-buffer "cache unit" number.
CBSec	equ	[$+11].lb	;Current-buffer sector count.
CBLst	equ	[$+12].lw	;Current-buffer "last" LRU index.
CBNxt	equ	[$+14].lw	;Current-buffer "next" LRU index.
;
; UltraDMA "Mode" Table (digit count in low 4 bits), for Init only.
;
Modes	dw	01602h		;Mode 0, ATA-16.
	dw	02502h		;Mode 1, ATA-25.
	dw	03302h		;Mode 2, ATA-33.
	dw	04402h		;Mode 3, ATA-44.
	dw	06602h		;Mode 4, ATA-66.
	dw	01003h		;Mode 5, ATA-100.
	dw	01333h		;Mode 6, ATA-133.
	dw	01663h		;Mode 7, ATA-166 (obviated by SATA).
;
; "Working" Cache Parameter Table.
;
STLmt	dw	0		;Binary-search table limit index.
LUTop	dw	0FFFFh		;Least-recently-used "top" index.
LUEnd	dw	0FFFFh		;Least-recently-used "end" index.
Blocks	equ	[$].lw		;Number of cache blocks.
SecPB	equ	[$+2].lb	;512-byte sectors per cache block.
SecSC	equ	[$+3].lb	;"Granularity" sector-shift count.
MidPt	equ	[$+4].lw	;Binary-search "midpoint" index.
XBase	equ	[$+6].dwd	;32-bit cache XMS "base" address.
DTAddr	equ	[$+10].dwd	;32-bit cache data-table address.
STAddr	equ	[$+14].dwd	;32-bit binary-search table address.
STBuff	equ	[$+18].dwd	;32-bit binary-search buffer address.
;
; Init Variables and "Not UltraDMA" message, located here to reduce
;   the size of our Init logic.    After Init, this area is used as
;   our Caching Parameter Constants, described above.
;
BiosHD	db	0		;BIOS hard-disk count.
HDUnit	db	0		;Current BIOS unit number.
UEMsg	db	' is not UltraDMA!',CR,LF,'$'
;
; Caching "Calculated LBA" Buffer (values shown are for Init only).
;
LBABuf	dw	00082h,0	;"Calculated LBA" buffer (8 bytes).
PCISubC	dw	00100h		;PCI subclass/function (Init only).
PCITbl	dw	(Ctl1Tbl-@)	;PCI-test table ptr.   (Init only).
;
; Fixed Cache Parameter-Table Pointers, MUST be at our offset 009Ch!
;
CCTblP	dw	(CCTbl-@)	;"Common" parameter-table offset.
U1TblP	dw	(CCTbl-@)	;"User-1" parameter-table offset.
U2TblP	dw	(CCTbl-@)	;"User-2" parameter-table offset.
CDTblP	dw	(CCTbl-@)	;"CD/DVD" parameter-table offset.
;
; Caching Caller's Saved Stack Pointer.   UHDD does not require a
;   driver I.D. here, given its "fixed" name in the driver header.
;
CStack	dd	0		;Caller's saved stack pointer.
;
; Int 13h Entry Routine.   For a CHS request, the registers are:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number:  000h/001h diskettes, 080h+ hard-disks.
;   ES:BX   I-O buffer address.
;
; For an LBA request, the registers are:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number:  000h/001h diskettes, 080h+ hard-disks.
;   DS:SI   Pointer to Device Address Packet ("DAP") as shown above.
;
Entry:	pushf			;Save CPU flags and BP-reg.
	push	bp
	mov	bp,0		;Reset active-units table index.
@LastU	equ	[$-2].lw	;(Last-unit index, set by Init).
NextU:	dec	bp		;Any more active units to check?
	js s	QuickX		    ;No, not for us -- exit quick!
	cmp	dl,cs:[bp+Units-@]  ;Does request unit match table?
	jne s	NextU		    ;No, see if more units remain.
	cli			;Disable CPU interrupts.
	bts	cs:IOF,0	;Is this driver currently "busy"?
	jc s	InvalF		;Yes?  Set "invalid function" & exit!
	mov	cs:CStack.lw,sp	;Save caller's stack pointer.
	mov	cs:CStack.hw,ss
	push	cs		;Switch to our driver stack, to avoid
	pop	ss		;  CONFIG.SYS "STACKS=0,0" problems!!
	mov	sp,0
@Stack	equ	[$-2].lw	;(Starting stack pointer, Init set).
	sti			;Re-enable CPU interrupts.
	pushad			;Save all CPU registers.
	push	ds
	push	es
	pusha			;Issue "A20 local-enable" request.
	mov	ah,005h
	call	A20Req
	popa
	jc s	A20Err		;If "A20" error, use routine below.
	db	0EAh		;Go to main (HMA) request handler.
@HDMain	dd	(HDReq-@)	;(Main handler address, set by Init).
InvalF:	mov	ah,001h		;We are BUSY, you fool!  Set "invalid
	jmp s	GetOut		;  function" code and get out, QUICK!
A20Err:	mov	al,XMSERR	;"A20" error!  Post "XMS error" code.
	jmp s	Abandn		;Abandon this request and exit QUICK!
PassRq:	call	A20Dis		;Pass request -- Disable "A20" line.
	cli			;Disable CPU interrupts.
	dec	IOF		;Reset driver "busy" flag.
	pop	es		;Reload CPU segment regs. we used.
	pop	ds
	popad			;Reload all 32-bit CPU regs.
	lss	sp,cs:CStack	;Switch back to caller's stack.
QuickX:	pop	bp		;Reload BP-register and CPU flags.
	popf
	nop			;(Unused alignment "filler").
	db	0EAh		;Go to next routine in Int 13h chain.
@Prev13	dd	0		;(Prior Int 13h vector, set by Init).
HDExit:	call	A20Dis		;Exit -- Issue "A20 local disable".
	pop	ax		;Reload error code and error flag.
	popf
Abandn:	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+33],al	;Set error code in exiting AH-reg.
	cli			;Disable CPU interrupts.
	dec	IOF		;Reset driver "busy" flag.
	pop	es		;Reload CPU segment regs. we used.
	pop	ds
	popad			;Reload all 32-bit CPU regs.
	lss	sp,cs:CStack	;Switch back to caller's stack.
GetOut:	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1	;(One of the flags saved by Int 13h).
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-register and CPU flags.
	popf
	iret			;Exit.
;
; Subroutine to issue "A20" local-enable and local-disable requests.
;
A20Dis:	mov	ah,006h		;"A20 local disable" -- get XMS code.
A20Req:	db	09Ah		;Call XMS manager for "A20" request.
@XMgr1:	dd	0		;(XMS "entry" address, set by Init).
A20Rst:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	sub	al,1		;Zero AL-reg. if success, -1 if error.
	ret			;Exit.
	db	0,0		;(Unused alignment "filler").
LMEnd	equ	($+STACK)	;End of our driver's low-memory space.
				;  All that follows can go in the HMA.
;
; Main Hard-Disk and Diskette Request Handler.
;
HDReq:	cmp	bp,CDUNIT	;CD/DVD or user-driver request?
	jae	SetBuf		;Yes, go set I-O parameters below.
	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	cmp	dl,002h		    ;CHS or LBA read/write request?
	mov	dl,cs:[bp+TypeF-@]  ;(Set unit "type" flag anyway).
@TypeF	equ	[$-2].lw
	mov	RqTyp,dl
	je s	ChkLBA		;Yes, check for actual LBA request.
	cmp	dl,0C0h		;Not for us -- Diskette I-O request?
	jne s	Pass		;No, "flush" cache and pass request.
	cmp	ah,005h		;Diskette track-format request?
	jne s	Pass1		;No, "pass" other diskette requests.
Pass:	or	IOF,002h	;Flush cache on next caching request.
Pass1:	push	ss		;Return to request "pass" routine.
	push	(PassRq-@)
	retf
ChkLBA:	mov	di,sp		;Point DI-reg. to current stack.
	shl	ah,1		;Is this an LBA read or write request?
	jns s	ValSC		;No, go validate CHS sector count.
	push	ds		;Save this driver's DS-reg.
	mov	ds,[di+2]	;Reload "DAP" segment into DS-reg.
	cmp	[si].DapBuf,-1	;Check for 64-bit "DAP" I-O buffer.
	mov	al,[si].DapSC	;(Get "DAP" I-O sector count).
	les	dx,[si].DapLBA1	;(Get "DAP" LBA bits 16-47).
	mov	di,es
	les	cx,[si].DapBuf	;(Get "DAP" I-O buffer address).
	mov	si,[si].DapLBA	;(Get "DAP" LBA bits 0-15).
	pop	ds		;(Reload this driver's DS-reg.).
	je s	Pass		;If 64-bit buffer, pass this request!
ValSC:	dec	al		;Is sector count zero or over 128?
	js s	Pass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	xchg	ax,cx		;(Save sector count & command in CX).
	js s	ZeroBX		;Yes, go zero BX-reg. for our logic.
	mov	es,[di]		;CHS -- Reload I-O buffer segment.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx		    ;Swap cylinder & head values.
	mov	al,cs:[bp+CHSec-@]  ;Get disk CHS sectors/head value.
@CHSec	equ	[$-2].lw
	or	al,al		    ;Were disk CHS values legitimate?
	jz s	Pass		    ;No?  Let BIOS have this request!
	push	ax		    ;Save CHS sectors/head value.
	mul	ah		    ;Convert head to sectors.
	add	si,ax		    ;Add result to starting sector.
	pop	ax		    ;Reload CHS sectors/head value.
	mul	cs:[bp+CHSHd-@].lb  ;Convert cylinder to sectors.
@CHSHd	equ	[$-2].lw
	mul	dx
	add	si,ax		;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		    ;Get buffer offset in AX-reg.
ZeroBX:	xor	bx,bx		    ;Zero BX-reg. for our logic.
	mov	[bx+VDSOf-@].lw,ax  ;Set VDS I-O buffer address.
	mov	[bx+VDSSg-@],es
	and	ch,006h		    ;Mask off read/write commands.
	movzx	ax,cl		    ;Set VDS buffer length =
	shl	ax,1		    ;  sectors * (512 bytes/sector).
	mov	[bx+VDSLn+1-@],ax
	pusha			    ;Save all 16-bit CPU registers.
	push	ds
	or	[bx+IOAdr-@].dwd,-1 ;Invalidate VDS buffer address.
	mov	ax,08103h	    ;Get VDS "lock" parameters.
	mov	dx,0000Ch
	mov	di,(VDSLn-@)	    ;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		    ;Execute "VDS lock" VDS request.
	sti			    ;RESTORE all critical settings!
	cld			    ;(NEVER "trust" external code!).
	pop	ds		    ;Reload all 16-bit registers.
	popa
	jc	Pass		    ;Lock error?  Pass this request!
	mov	eax,[bx+IOAdr-@]    ;Get 32-bit VDS buffer address.
	cmp	eax,-1		    ;Is VDS address not "all-ones"?
	jb s	VDLokF		    ;Yes, set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw ;VDS logic is NOT present --
	shl	eax,4		    ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
VDLokF:	adc	[bx+VLF-@],bl	    ;Set VDS "lock" flag from carry.
	push	cs		    ;Use our "Common" cache for all
	push	CCTblP		    ;  hard-disk and diskette work.
	pop	[bx+RqCPA-@].dwd
SetBuf:	mov	[bx+RqBuf-@],eax    ;Set user I-O buffer address.
	mov	[bx+RqSec-@],cx	    ;Set sector count & I-O command.
	mov	[bx+RqLBA-@],di	    ;Set initial request LBA.
	mov	[bx+RqLBA+2-@],dx
	mov	[bx+RqLBA+4-@],si
	mov	[bx+RqUNo-@],bp	    ;Set "cache unit" number.
;
; Main Caching Routine.
;
	btr	[bx+IOF-@].lb,1	     ;"Common" cache flush request?
	jc s	Flush		     ;Yes, do "flush" before I-O.
	cmp	[bx+RqTyp-@].lb,0C0h ;Is this request for a diskette?
	jne s	SetTbl		     ;No, continue with this request.
	mov	es,bx		     ;Point ES-reg. to low memory.
	mov	al,es:[DKTSTAT]	     ;Get BIOS diskette status code.
	and	al,01Fh
	cmp	al,MCHANGE	 ;Diskette media-change detected?
	jne s	SetTbl		 ;No, continue with this request.
Flush:	mov	si,CCTblP	 ;Get "Common" cache parameters addr.
	mov	cs:[si],bx	 ;Reset binary-search table limit.
	or	cs:[si+2].dwd,-1 ;Reset LRU "top" & "end" indexes.
	nop			 ;(Unused alignment "filler").
SetTbl:	mov	cx,7		 ;Set all "working" cache parameters.
	lds	si,[bx+RqCPA-@]
	mov	di,(STLmt-@)
	push	ss
	pop	es
	rep	movsd
	push	es		;Reload this driver's DS-reg.
	pop	ds
NxtBlk:	push	ds		;Set ES-reg. same as DS-reg.
	pop	es
	mov	dx,MidPt	;Set initial binary-search offset.
	mov	bp,dx		;Set initial binary-search index.
	dec	bp
	mov	al,SecPB	;Get "granularity" (sectors/block).
	cmp	al,[bx+RqSec-@]	;Will we need multiple cache blocks?
	jbe s	SetSC		;Yes, use "full block" sector count.
	mov	al,[bx+RqSec-@]	;Use remaining request sector count.
SetSC:	mov	[bx+RqCSC-@],al	;Set maximum I-O sector count.
	mov	[bx+RqXMS-@],bx ;Reset lower XMS buffer offset.
Search:	shr	dx,1		;Divide binary-search offset by 2.
	cmp	bp,[bx+STLmt-@]	;Is our search index too high?
	jae s	SrchLo		;Yes, cut search index by offset.
	call	SIGet		;Get next cache entry into buffer.
XMErr1:	jc	XMErrF		;If any XMS error, exit immediately!
	mov	si,(RqLBA-@)	;Compare initial request & table LBA.
	call	CComp
	jae s	ChkEnd		;Request >= table:  Check for found.
SrchLo:	sub	bp,dx		;Subtract offset from search ptr.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	jmp s	NoFind		;Handle this request as "not found".
ChkEnd:	je	Found		;Request = table:  Treat as "found".
	mov	cl,[bx+CBSec-@]	;Calculate and set ending entry LBA.
	mov	si,(CBLBA-@)
	call	CalcEA
	mov	si,(RqLBA-@)	;Compare request start & entry end.
	call	CComp1
	jb s	Found		;Request < Entry:  Handle as found.
	ja s	SrchHi		;Request > Entry:  Bump search ptr.
	mov	cl,SecPB	;Get "granularity" (sectors/block).
	cmp	cl,[bx+CBSec-@]	;Is this cache entry "full"?
	ja s	Found		;No, handle this request as "found".
SrchHi:	add	bp,dx		;Add offset to search pointer.
	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	Search		;No, go compare next table entry.
	inc	bp		;Bump index to next table entry.
	nop			;(Unused alignment "filler").
NoFind:	mov	[bx+RqNdx-@],bp	;Unfound:  Save search-table offset.
	mov	bp,[bx+STLmt-@]	;Get next "free" cache-table index
	call	STGet		;  and leave index in work buffer.
XMErr2:	jc s	XMErr1		;If any XMS error, exit immediately!
	movzx	ecx,bp		    ;Get move-up word count.
	movzx	esi,[bx+RqNdx-@].lw
	sub	cx,si		    ;Any search indexes to move up?
	jz s	NFind2		    ;No, go set up new cache entry.
	shl	ecx,1		;Set move-up byte count.
	shl	esi,1		;Set 32-bit move-up addresses.
	mov	edx,STAddr
	add	esi,edx
	lea	edi,[esi+2]
	push	ecx		;Save move length & destination addr.
	push	edi
	mov	edi,STBuff	;Get & save search-table buffer addr.
	push	edi
	call	MvData		;Send needed data to search buffer.
	pop	esi		;Reload search-table buffer address.
	pop	edi		;Reload move destination & length.
	pop	ecx
	jc s	XMErr2		;If any XMS error, exit immediately!
	call	MvData		;Bring data BACK from search buffer.
	jc s	XMErr2		;If any XMS error, exit immediately!
	mov	bp,[bx+RqNdx-@]	;Set up table-index "put" to XMS.
	inc	bx
	call	STPut		;Insert "free" index in search table.
	jc s	XMErr2		;If any XMS error, exit immediately!
	nop			;(Unused alignment "filler").
NFind2:	inc	[bx+STLmt-@].lw	;Advance binary-search limit index.
	mov	si,(RqLBA-@)	;Set 48-bit LBA in new entry.
	mov	di,(CBLBA-@)
	movsd
	movsw
	movsb			   ;Set "cache unit" in new entry.
	mov	[di],bl		   ;Reset new entry's sector count.
	mov	ax,[bx+WrkBf-@]	   ;Reload "free" cache-table index.
Found:	mov	[bx+RqNdx-@],ax	   ;Post current cache-entry index.
	mov	cx,[bx+RqLBA+4-@]  ;Get starting I-O offset in block.
	sub	cx,[bx+CBLBA+4-@]
	mov	[bx+RqXMS-@],cl	   ;Set starting XMS sector offset.
	mov	[bx+LBA2+1-@],cx   ;Save starting I-O sector offset.
	cmp	[bx+CBSec-@],bl	   ;Is this a new cache-table entry?
	je s	ReLink		   ;Yes, relink entry as top-of-list.
	push	ax		   ;Unlink this entry from LRU list.
	call	UnLink
	pop	ax
ReLink:	mov	[bx+RqXMS+2-@],bx  ;Reset upper XMS buffer offset.
	movzx	edx,ax		   ;Get 32-bit cache block number.
	mov	cl,SecSC	   ;Shift number to starting sector.
	shl	edx,cl
	add	[bx+RqXMS-@],edx   ;Add to "preset" sector offset.
	shl	[bx+RqXMS-@].dwd,9 ;Convert sectors to bytes.
	mov	edx,XBase	   ;Add in XMS "base" address to 
	add	[bx+RqXMS-@],edx   ;  get cache-block XMS address.
	mov	cx,0FFFFh	   ;Make this entry "top of list".
	or	[bx+CBLst-@],cx	   ;Set this entry's "last" index.
	mov	dx,ax		   ;Swap top-of-list & entry index.
	xchg	dx,[bx+LUTop-@]
	mov	[bx+CBNxt-@],dx	;Set this entry's "next" index.
	cmp	dx,cx		;Is this the only LRU index?
	mov	cx,ax		;(Get this entry's index in CX-reg.).
	je s	ReLnk1		;Yes, make entry last on LRU list.
	push	ax		;Link entry to prior "top of list".
	call	UnLnk3
	pop	ax
	jmp s	ReLnk2		;Go deal with I-O sector count.
Update:	mov	[bx+CBSec-@],cl	;Update this entry's total sectors.
	call	CBPut		;Update this cache-table entry.
	jc s	XMErr3		;If any XMS error, exit immediately!
	movzx	cx,[bx+RqCSC-@]	;Calculate ending LBA for this I-O.
	mov	si,(RqLBA-@)
	call	CalcEA
	inc	bp		;Skip to next search index.
Ovrlap:	cmp	bp,[bx+STLmt-@]	;More cache-table entries to check?
	jae s	CachIO		;No, O.K. to handle "found" entry.
	call	SIGet		;Get next cache entry into buffer.
	jc s	XMErr3		;If any XMS error, exit immediately!
	mov	si,(LBABuf-@)	;Compare request end & entry start.
	call	CComp
	jbe s	CachIO		;Request <= entry:  O.K. to proceed.
	push	bp		;Delete this overlapping table entry.
	call	Delete
	pop	bp
	jmp s	Ovrlap		 ;Go check for more entry overlap.
ReLnk1:	mov	[bx+LUEnd-@],ax	 ;Make entry last on LRU list, too!
ReLnk2:	mov	cx,[bx+LBA2+1-@] ;Reload initial I-O sector offset.
	mov	ch,SecPB	 ;Get entry's available sectors.
	sub	ch,cl
	cmp	ch,[bx+RqCSC-@]	 ;More I-O sectors than available?
	jae s	Larger		 ;No, retain maximum sector count.
	mov	[bx+RqCSC-@],ch	 ;Reduce current I-O sector count.
Larger:	add	cl,[bx+RqCSC-@]	 ;Get ending I-O sector number.
	cmp	cl,[bx+CBSec-@]	 ;More sectors than entry has now?
	ja s	Update		 ;Yes, update entry sectors.
	inc	bx		 ;Reset Z-flag for "put" into XMS.
	call	CBPut		 ;Update this cache-table entry.
XMErr3:	jc	XMErrF		      ;If any XMS error, exit fast!
	test	[bx+RqCmd-@].lb,002h  ;Is this a read request?
	jz s	BufMov		      ;Yes, move cache data to user.
CachIO:	bts	[bx+RqCmd-@].lb,6     ;I-O done during a prior block?
	jc s	BfMore		      ;Yes, buffer more cache data.
	cmp	[bx+RqTyp-@].lb,CDTYP ;What type of unit is this?
	jae s	ExtIO		      ;External/BIOS -- go see which.
	call	UdmaIO		 ;Call "UdmaIO" for SATA/IDE disks.
	jmp s	ErrChk		 ;Go check for any I-O errors.
ExtIO:	or	[bx+RqTyp-@],bl	 ;Is this unit handled by the BIOS?
	js s	BiosIO		 ;Yes, go "Call the BIOS" below.
	call	[bx+DMAAd-@].dwd ;Call user-driver "callback" rtn.
	jmp s	ErrChk		 ;Go check for any I-O errors.
BiosIO:	lds	si,CStack	 ;BIOS I-O:  Reload caller CPU flags.
	push	[si+2].lw
	popf
	pop	es		;Reload all CPU registers.
	pop	ds
	popad
	pushf			;"Call the BIOS" to do this request.
	call	ss:@Prev13
	pushad			;Save all CPU registers again.
	push	ds
	push	es
	mov	al,ah		;Move any BIOS error code to AL-reg.
ErrChk:	sti			;Restore critical driver settings.
	cld
	push	ss		;Reload this driver's DS- & ES-regs.
	pop	ds
	push	ds
	pop	es
	mov	bx,0		;Rezero BX-reg. but save carry.
	jnc s	InCach		;If no error, go see about caching.
IOErr:	push	ax		;Error!  Save returned error code.
	mov	ax,RqNdx	;Delete cache entry for this I-O.
	call	SrchD
	pop	ax		;Reload returned error code.
	stc			;Set carry again to denote error.
EndIO:	pushf			;Exit -- save error flag and code.
	push	ax
	mov	si,(STLmt-@)	;Update current caching parameters.
	les	di,[bx+RqCPA-@]
	movsw
	movsw
	movsw
	shr	[bx+VLF-@].lb,1	;User's I-O buffer "locked" by VDS?
	jnc s	ExitIO		;No, return to hard-disk exit logic.
	mov	ax,08104h	;Set needed VDS "unlock" parameters.
	xor	dx,dx
	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		;Issue "VDS unlock" request.
ExitIO:	db	0EAh		;Return to hard-disk exit logic with
	dw	(HDExit-@)	;  a 5-byte "jmp" (permits HMA use!).
@HDXSeg	dw	0		;(Driver segment address, Init set).
InCach:	or	[bx+RqCmd-@],bl	;Hard-disk data cached during I-O?
	js s	ChkFul		;Yes, check if cache tables are full.
BfMore:	or	al,1		    ;Ensure we LOAD data into cache!
BufMov:	movzx	ecx,[bx+RqCSC-@].lb ;Set XMS move sector count.
	mov	esi,[bx+RqXMS-@]    ;Set desired XMS buffer address.
	mov	edi,[bx+RqBuf-@]    ;Set user buffer as destination.
	call	MovBuf		  ;Move data between user and XMS.
	jc s	IOErr		  ;If any XMS error, use logic above.
ChkFul:	mov	ax,Blocks	  ;Get our cache block count.
	cmp	ax,[bx+STLmt-@]	  ;Is binary-search table full?
	ja s	MoreIO		  ;No, check for more sectors to go.
	push	ax		  ;Delete least-recently-used entry.
	mov	ax,[bx+LUEnd-@]
	call	SrchD
	pop	ax
MoreIO:	xor	ax,ax		  ;Reset error code for exit above.
	movzx	cx,[bx+RqCSC-@]	  ;Get current I-O sector count.
	sub	[bx+RqSec-@],cl	  ;More data sectors left to handle?
	jz s	EndIO		  ;No, return to our "exit" routine.
	add	[bx+RqLBA+4-@],cx ;Update current LBA for next I-O.
	adc	[bx+RqLBA+2-@],bx
	adc	[bx+RqLBA-@],bx
	shl	cx,1		  ;Convert sector count to bytes.
	add	[bx+RqBuf+1-@],cl ;Update user I-O buffer address.
	adc	[bx+RqBuf+2-@],bx
	jmp	NxtBlk		  ;Go handle next cache data block.
;
; Subroutine to do 7-byte "cache unit" and LBA comparisons.
;
CComp:	mov	di,(CBLBA-@)	;Point to current-buffer LBA value.
CComp1:	mov	cl,[bx+RqUNo-@]	;Compare our "unit" & table "unit".
	cmp	cl,[bx+CBUNo-@]
	jne s	CCompX		;If units are different, exit now.
CComp2:	mov	cx,3		;Compare our LBA & cache-table LBA.
	rep	cmpsw
CCompX:	ret			;Exit -- Main routine checks results.
;
; Subroutine to "put" an LRU index into its cache-data table entry.
;
LRUPut:	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag (always 0 = "put").
	mov	edi,(WrkBf-@)	;Set our "working" buffer address.
@WrkBf1	equ	[$-4].dwd	;(Working-buffer address, Init set).
	mov	ecx,12		;Set cache-table entry size.
	mul	cx		;Multiply cache index by entry size.
	mov	cl,2		;Set LRU-index size ("put" 2 bytes).
	jmp s	CEMov1		;Go get desired cache-entry offset.
;
; Subroutine to "get" or "put" a cache-data table entry (12 bytes).
;
SIGet:	call	STGet		;Get next binary-search table index.
	jc s	CCompX		;If any XMS error, go exit above.
	mov	ax,[bx+WrkBf-@]	;Get desired cache-data table index.
CBGet:	xor	di,di		;Entry buffer:  Set Z-flag for "get".
CBPut:	mov	edi,(CBLBA-@)	;Set 32-bit "current" buffer address.
@CBAddr	equ	[$-4].dwd	;(Current-buffer address, Init set).
CEMov:	mov	esi,DTAddr	;Set cache-data table address.
	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	mov	ecx,12		;Set cache-table entry size.
	mul	cx		;Multiply cache index by entry size.
CEMov1:	push	dx		;Get cache-entry offset in EAX-reg.
	push	ax
	pop	eax
	jmp s	CeMov2		;Go set final XMS data address.
EFlag	db	0		;(Unused -- Init "emulator" flag).
;
; Subroutine to "get" or "put" a binary-search table index (2 bytes).
;
STGet:	xor	di,di		;Search index:  Set Z-flag for get.
STPut:	mov	edi,(WrkBf-@)	;Set 32-bit "working" buffer addr.
@WrkBf2	equ	[$-4].dwd	;(Working-buffer address, Init set).
	mov	esi,STAddr	;Add starting search-table address.
	push	ax		;Save all needed 16-bit regs.
	push	cx
	push	dx
	pushf			;Save CPU Z-flag from above.
	mov	ecx,2		;Set binary-search index size.
	movzx	eax,bp		;Get 32-bit offset of requested
	shl	eax,1		;  index in binary-search table.
CeMov2:	add	esi,eax		;Set XMS data address (base + offset).
	popf			;Reload CPU flags.
	push	bp		;Save BP-reg. now (helps alignment).
	jz s	CeMov3		;Are we "getting" from XMS memory?
	xchg	esi,edi		;No, "swap" source and destination.
CeMov3:	call	MvData		;"Get" or "put" desired cache entry.
	pop	bp		;Reload all 16-bit regs. we used.
	pop	dx
	pop	cx
	pop	ax
	ret			;Exit -- Caller tests for XMS errors.
;
; Global Descriptor Table, for cache XMS-memory moves.    Its "code"
;   segment descriptor is unused by the CPU since our real-mode move
;   logic is all "in line", with no "JMP" or "CALL" commands.   Many
;   Thanks to Japheth, for confirming this "poorly documented" info!
;
GDT	dd	0,0		      ;"Null" descriptor.
	dd	0,0		      ;"Code" segment descriptor.
SrcDsc	dd	00000FFFFh,000009300h ;Protected-mode source and
DstDsc	dd	00000FFFFh,000009300h ;  destination descriptors.
	dd	0,0,0,0		      ;(Reserved for use by BIOS).
GDT_DS	dd	00000FFFFh,000CF9300h ;4-GB real-mode descriptor.
GDTLen	equ	($-GDT)		      ;GDT length for real-mode.
;
; Global Descriptor Table Pointer, for our real-mode "lgdt" command.
;
GDTP	dw	GDTLen		;GDT length (always 56 bytes).
GDTPAdr	dd	(GDT-@)		;GDT 32-bit address (Set By Init).
;
; Subroutine to do "internal" XMS buffer moves.   This logic was
;   within "UdmaIO" but is now separate, so "UdmaIO"/"DoDMA" may
;   be dismissed (not loaded), when the /E switch is given.
;
MovBuf:	jz s	MovBf2		;Is this a read from XMS memory?
	xchg	esi,edi		;No, "swap" source & destination.
	nop			;(Unused alignment "filler").
MovBf2:	shl	ecx,9		;Convert sectors to byte count.
;
; Subroutine to do XMS moves.   At entry, the move parameters are:
;
;   ECX:   Number of bytes to move.   MUST be an even value as only
;	     whole 16-bit words are moved!   Up to a 4-GB value can
;	     be used as data is moved in 2K sections, which permits
;	     interrupts between sections.
;   ESI:   32-bit source data address (NOT segment/offset!).
;   EDI:   32-bit destination address (NOT segment/offset!).
;
; At exit, the carry bit is zero for no errors or is SET for a move
; error.   If so, the AL-reg. has a 0FFh "XMS error" code.   The DS
; and ES-regs. are saved/restored, and the BX-reg. is zero at exit.
; All other registers are NOT saved, for speed.   Only a "protected
; mode" move (JEMM386, etc.) can post an error.   "Real mode" moves
; (UMBPCI, etc.) have NO errors to post!   Our "stand alone" driver
; omits this subroutine and calls the XMS manager to do XMS moves!
;
MvData:	push	ds		;Save driver's segment registers.
	push	es
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands.
	mov	edx,ecx		;Save byte count in EDX-reg.
	mov	ecx,2048	;Get 2K real-mode section count.
	smsw	ax		;Get CPU "machine status word".
	shr	ax,1		;Are we running in protected-mode?
	jc s	MvProt		;Yes, go use protected-mode logic.
MvRNxt:	cmp	ecx,edx		;At least 2048 bytes left?
	cli			;(Disable CPU interrupts for move).
	jbe s	MvRGo		;Yes, use full section count.
	mov	cx,dx		;Use remaining byte count.
MvRGo:	db	02Eh,00Fh	;"lgdt cs:GDTP", coded the hard-way
	db	001h,016h	;   to avoid any annoying V5.1 MASM
	dw	(GDTP-@)	;   warning messages about it!
@GDTP	equ	[$-2].lw
	mov	eax,cr0		;Set CPU protected-mode control bit.
	or	al,001h
	mov	cr0,eax
	mov	bx,offset (GDT_DS-GDT)	;Set DS and ES "selectors".
	mov	ds,bx
	mov	es,bx
	shr	cx,2		;Convert byte count to dword count.
	db	0F3h,067h	;Move all 32-bit words using "rep"
	movsd			;  and "address-override" prefixes.
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	adc	cx,cx		;If "odd" 16-bit word, move it also,
	db	0F3h,067h	;  using "rep" & "address-override".
 	movsw
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	dec	ax		;Clear protected-mode control bit.
	mov	cr0,eax
	sti			;Allow interrupts after next command.
	mov	cx,2048		;Reload 2K move section count.
	sub	edx,ecx		;Any more data sections to move?
	ja s	MvRNxt		;Yes, go move next data section.
MvDone:	xor	ax,ax		;Success!  Reset carry and error code.
MvExit:	pop	es		;Reload driver's segment registers.
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	ret			;Exit.
MvProt:	shl	ecx,5		;Protected-mode:  Get 64K section ct.
@MvP2K	equ	[$-1].lb	;(If /N4 /Z, "shl ecx,0" = 2K count).
MvPNxt:	push	ecx		;Save move section count.
	push	edx		;Save remaining move byte count.
	push	esi		;Save move source & destination addrs.
	push	edi
	push	cs		;Point ES-reg. to move descriptors.
	pop	es
	cmp	ecx,edx		;At least one section left?
	jbe s	MvPGo		;Yes, use full section count.
	mov	ecx,edx		;Use remaining byte count instead.
MvPGo:	shr	ecx,1		;Convert byte count to word count.
	push	edi		;Set up destination descriptor.
	mov	di,offset (DstDsc+2-@)
@MvDesc	equ	[$-2].lw
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	push	esi		;Set up source descriptor ("sub"
	sub	di,11		;  zeros carry for our Int 15h).
	pop	ax
	stosw
	pop	ax
	stosb
	mov	es:[di+2],ah
	lea	si,[di-21]	;Point to start of descriptor table.
	mov	ah,087h		;Have JEMM386/BIOS move next section.
	int	015h
	pop	edi		;Reload all 32-bit move parameters.
	pop	esi
	pop	edx
	pop	ecx
	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands.
	mov	al,XMSERR	;Get our XMS error code.
	jc s	MvExit		;If any BIOS error, exit immediately.
	sub	edx,ecx		;Any more data sections to move?
	jbe s	MvDone		;No, go set "success" code and exit.
	add	esi,ecx		;Update source and dest. addresses.
	add	edi,ecx
	jmp s	MvPNxt		;Go move next section of data.
HFlag	db	0		;(Unused -- Init "Use HMA" flag).
;
; Subroutine to calculate ending request or cache-table LBA values.
;
CalcEA:	mov	di,(LBABuf-@)	;Point to address-comparison buffer.
	push	[si].dwd	;Move high-order LBA into buffer.
	pop	[di].dwd
	add	cx,[si+4]	;Calculate ending LBA.
	mov	[di+4],cx
	adc	[di+2],bx
	adc	[di],bx
	ret			;Exit.
;
; Subroutine to search for a cache-table index, and then delete it.
;   Our binary-search table has unit/LBA order, not LRU order!   To
;   delete an LRU index, we do a binary-search for it and then pass
;   the ending BP-reg. value to our "Delete" subroutine below.
;
SrchD:	call	CBGet		;Get target cache-table entry.
	jc s	XMErr4		;If XMS error, go bail out below!
	mov	si,(CBLBA-@)	;Save target LBA address & unit no.
	mov	di,(LBABuf-@)
	movsd
	movsd
	mov	dx,MidPt	;Set initial binary-search offset.
	mov	bp,dx		;Set initial binary-search index.
	dec	bp
	nop			;(Unused alignment "filler").
SrchD1:	shr	dx,1		;Divide binary-search offset by 2.
	cmp	bp,[bx+STLmt-@]	;Is our search pointer too high?
	jae s	SrchD4		;Yes, cut search pointer by offset.
	call	SIGet		;Get next cache entry into buffer.
	jc s	XMErr4		;If XMS error, go bail out below!
	mov	si,(LBABuf-@)	;Set up target v.s. work comparison.
	mov	di,(CBLBA-@)
	mov	cl,[si+6]	;Compare target unit v.s. work unit.
	cmp	cl,[di+6]
	jne s	SrchD3		;If units differ, check results now.
	call	CComp2		;Compare target LBA v.s. work LBA.
	je s	Delete		;Target = entry:  BP has our offset.
	nop			;(Unused alignment "filler").
SrchD3:	ja s	SrchD5		;Target > entry:  Adjust offset up.
SrchD4:	sub	bp,dx		;Subtract offset from search ptr.
	jmp s	SrchD6		;Go see if we did our last compare.
SrchD5:	add	bp,dx		;Add offset to search pointer.
SrchD6:	or	dx,dx		;Offset zero, i.e. last compare?
	jnz s	SrchD1		;No, go compare next table entry.
XMErr4:	jmp s	XMErr5		;Not found!  Handle as an XMS error.
;
; Subroutine to delete a cache index from our search table.
;
Delete:	mov	[bx+WrkBf-@],ax	     ;Save table index being deleted.
	movzx	ecx,[bx+STLmt-@].lw  ;Get move-down word count.
	sub	cx,bp
	dec	cx		;Any other indexes to move down?
	jz s	Delet1		;No, put our index in "free" list.
	shl	ecx,1		;Set move-down byte count.
	movzx	edi,bp		;Set 32-bit move-down addresses.
	shl	edi,1
	add	edi,STAddr
	lea	esi,[edi+2]
	call	MvData		;Move down needed search-table data.
	jc s	XMErr5		;If any XMS error, go exit below.
Delet1:	dec	[bx+STLmt-@].lw	;Decrement search-table limit index.
	mov	bp,[bx+STLmt-@]	;Set up table-index "put" to XMS.
	inc	bx
	call	STPut		;Put deleted index in "free" list.
	jc s	XMErr5		;If any XMS error, go exit below.
;
; Subroutine to unlink a cache-table entry from the LRU list.
;
UnLink:	mov	cx,[bx+CBLst-@]	;Get entry's "last"/"next" indexes.
	mov	dx,[bx+CBNxt-@]
	cmp	cx,-1		;Is this entry top-of-list?
	je s	UnLnk1		;Yes, "promote" next entry.
	mov	[bx+WrkBf-@],dx	;Save this entry's "next" index.
	mov	ax,cx		;Get "last" entry cache-table index.
	mov	esi,DTAddr	;Get cache-data table address.
	add	esi,10		;Add offset of LRU "next" index.
	call	LRUPut		;Update last entry's "next" index.
	jnc s	UnLnk2		;If no XMS error, check end-of-list.
XMErr5:	pop	cx		;XMS error!  Discard exit address &
	pop	cx		;  AX/BP parameter saved upon entry.
XMErrF:	mov	[bx+STLmt-@],bx	    ;Zero binary-search table limit.
	or	[bx+LUTop-@].dwd,-1 ;Reset LRU-list top/end indexes.
	stc			    ;Set carry again after "or" cmd.
	mov	al,XMSERR	;Ensure "XMS error" will get posted.
	jmp	EndIO		;Go update cache parameters & exit!
UnLnk1:	mov	[bx+LUTop-@],dx	;Make next entry top-of-list.
UnLnk2:	cmp	dx,-1		;Is this entry end-of-list?
	jne s	UnLnk3		;No, link next to prior entry.
	mov	[bx+LUEnd-@],cx	;Make prior entry end-of-list.
	ret			;Exit.
	db	0,0		;(Unused alignment "filler").
UnLnk3:	mov	[bx+WrkBf-@],cx	;Save this entry's "last" index.
	mov	ax,dx		;Get "next" entry cache-table index.
	mov	esi,DTAddr	;Get cache-data table address.
	add	esi,8		;Add offset of LRU "last" index.
	call	LRUPut		;Update next entry's "last" index.
	jc s	XMErr5		;If any XMS error, go exit above!
	ret			;All is well -- exit.
	db	0,0		;(Unused alignment "filler").
;
; Hard Disk and Diskette Parameter Tables.
;
TypeF	equ  [$].lb		;Device type and "sub-unit":
				;  00h-27h:  SATA/IDE UltraDMA disk.
				;  80h:      BIOS-handled disk.
				;  C0h:      BIOS-handled diskette.
CHSec	equ  [$+MAXBIOS].lb	;Sectors-per-head table.
CHSHd	equ  [$+(MAXBIOS*2)].lb	;Heads-per-cylinder table.
;
; Table of Available Cache Sizes.    This table and the "No PCI"
;   message at its end are placed here to reduce the size of our
;   Init logic.   After cache setup, this space becomes our disk
;   and diskette parameter tables, as described above.
;
Cache5	dw	(5*128)		; 5-MB "tiny" cache.
	db	16		;   8K sectors per cache block
	db	4		;      and sector-shift count.
	dw	512		;   512 binary-search midpoint.
	dw	10		;   10K cache-tables XMS memory.
	db	"   5"		;   Title message cache-size text.
Cache15	dw	(15*64)		;15-MB "small" cache.
	db	32,5		;   16K granularity values.
	dw	512		;   512 binary-search midpoint.
	dw	15		;   15K cache-tables XMS memory.
	db	"  15"		;   Title message cache-size text.
Cache25	dw	(25*64)		;25-MB "medium 1" cache.
	db	32,5		;   16K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	25		;   25K cache-tables XMS memory.
	db	"  25"		;   Title message cache-size text.
Cache40	dw	(40*32)		;40-MB "medium 2" cache.
	db	64,6		;   32K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	20		;   20K cache-tables XMS memory.
	db	"  40"		;   Title message cache-size text.
Cache50	dw	(50*32)		;50-MB "medium 3" cache.
	db	64,6		;   32K granularity values.
	dw	1024		;   1024 binary-search midpoint.
	dw	25		;   25K cache-tables XMS memory.
	db	"  50"		;   Title message cache-size text.
CacheLC	dw	0FFFFh		;80-MB to 4093-MB "large" cache.
	db	128,7		;   64K granularity values.
	dw	1024		;   1024 to 32768 search midpoint.
	dw	32		;   32K to 1152K cache-tables XMS.
	db	"  80"		;   Title message cache-size text.
	db	(TypeF+(MAXBIOS*3)-$) dup (0)  ;Unused "filler".
;
; "Common" Cache Parameter Table, always set during Init.   Values
;   below will set an 80-MB default cache size when /S is omitted.
;
CCTbl	dw	0		;Binary-search table limit index.
	dw	0FFFFh		;Least-recently-used "top" index.
	dw	0FFFFh		;Least-recently-used "end" index.
	dw	80		;Number of cache blocks.
	db	0		;512-byte sectors per cache block.
	db	0		;"Granularity" sector-shift count.
	dw	(CacheLC-@)	;Binary-search "midpoint" index.
	dd	0		;32-bit cache XMS "base" address.
	dw	(CCTblP-@),'oC'	;32-bit cache data-table address.
	db	'mmon'		;32-bit binary-search table address.
	db	"  80"		;32-bit binary-search buffer address.
PTSIZE	equ	($-CCTbl)	;(Size of a cache parameter table).
	ifdef	CACHE4
;
; "User-1" Cache Parameter Table, set during Init by the /X switch.
;
U1Tbl	dw	0		;Binary-search table limit index.
	dw	0FFFFh		;Least-recently-used "top" index.
	dw	0FFFFh		;Least-recently-used "end" index.
	dw	0		;Number of cache blocks.
	db	0		;512-byte sectors per cache block.
	db	0		;"Granularity" sector-shift count.
	dw	0		;Binary-search "midpoint" index.
	dd	0		;32-bit cache XMS "base" address.
	dw	(U1TblP-@),'sU'	;32-bit cache data-table address.
	db	'er-1'		;32-bit binary-search table address.
	dd	0		;32-bit binary-search buffer address.
;
; "User-2" Cache Parameter Table, set during Init by the /Y switch.
;
U2Tbl	dw	0		;Binary-search table limit index.
	dw	0FFFFh		;Least-recently-used "top" index.
	dw	0FFFFh		;Least-recently-used "end" index.
	dw	0		;Number of cache blocks.
	db	0		;512-byte sectors per cache block.
	db	0		;"Granularity" sector-shift count.
	dw	0		;Binary-search "midpoint" index.
	dd	0		;32-bit cache XMS "base" address.
	dw	(U2TblP-@),'sU'	;32-bit cache data-table address.
	db	'er-2'		;32-bit binary-search table address.
	dd	0		;32-bit binary-search buffer address.
	endif
;
; "CD/DVD" Cache Parameter Table, set during Init by the /C switch.
;
CDTbl	dw	0		;Binary-search table limit index.
	dw	0FFFFh		;Least-recently-used "top" index.
	dw	0FFFFh		;Least-recently-used "end" index.
	dw	0		;Number of cache blocks.
	db	0		;512-byte sectors per cache block.
	db	0		;"Granularity" sector-shift count.
	dw	0		;Binary-search "midpoint" index.
	dd	0		;32-bit cache XMS "base" address.
	dw	(CDTblP-@),'DC'	;32-bit cache data-table address.
	db	'/DVD'		;32-bit binary-search table address.
	dd	0		;32-bit binary-search buffer address.
	ifdef	CACHE4
	db	0,0,0,0,0,0,0,0	;(Unused 4-cache alignment "filler").
	endif
ESWLEN	equ	($-HDReq)	;(Length of /E HMA caching logic).
;
; Subroutine to set up and execute hard disk UltraDMA I-O requests.
;   The "DoDMA" subroutine (below) is then used to perform the I-O.
;   Both subroutines and the Controller Address table are dismissed
;   (not loaded), when the /E switch is given.
;
UdmaIO:	mov	cx,[bx+RqSec-@]	  ;Get sector count and I-O command.
	mov	si,[bx+RqLBA+4-@] ;Get 48-bit logical block address.
	les	di,[bx+RqLBA-@]
	mov	dx,es
	mov	[bx+LBA-@],si	  ;Set 48-bit LBA in IDE commands.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di  ;Set LBA bits 16 thru 47.
	mov	bp,[bx+RqTyp-@]	  ;Get this disk's "type" flags.
	mov	ax,bp		  ;Get disk's IDE controller number.
	shr	al,2
	mov	ah,6		  ;Point to disk's I-O addresses.
	mul	ah
	add	ax,offset (Ctl1Tbl-@)
@CtlTbl	equ	[$-2].lw
	xchg	ax,si
	lods	cs:[si].dwd	  ;Set controller base address and
	mov	[bx+DMAAd-@],eax  ;  primary-channel data address.
	rcr	bp,2		  ;Primary channel I-O request?
	jnc s	GetCmd		  ;Yes, get LBA28/LBA48 commands.
	add	[bx+DMAAd-@].lw,8 ;Use secondary DMA controller.
	lods	cs:[si].lw	  ;Set secondary channel data addr.
	mov	[bx+IdeDA-@],ax
GetCmd:	shr	dx,12		  ;Shift out LBA bits 16-27.
	or	di,dx		  ;Anything in LBA bits 28-47?
	mov	dl,(LBABITS/32)	  ;(Init LBA bits now for alignment).
	jnz s	LBA48		  ;Yes, use LBA48 read/write command.
	xchg	dh,[bx+LBA2-@]	  ;LBA28:  Reload & reset bits 24-27.
	or	ch,(DRCMD+1)	  ;Get LBA28 read/write command + 5.
	jmp s	SetCmd		  ;Go set LBA and IDE command bytes.
LBA48:	shl	ch,3		  ;LBA48 -- get command as 20h/30h.
SetCmd:	shl	bp,1		  ;Shift master/slave into LBA cmds.
	rcl	dl,5
	or	dl,dh		  ;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		  ;Get final IDE command byte --
	xor	dh,ch		  ;  LBA28 C8h/CAh, LBA48 25h/35h.
	mov	[bx+DSCmd-@],dx	  ;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],cl	  ;Set disk I-O sector count.
	mov	[bx+IOLen-@],bl	  ;Set 24-bit DMA buffer length.
	mov	ch,0		  ;  We include bit 16 for 65536
	shl	cx,1		  ;  bytes, as a few "odd" chips
	mov	[bx+IOLen+1-@],cx ;  may NOT run O.K. without it!
	mov	cl,[bx+RqCSC-@]	  ;Get current I-O sector count.
	cmp	cl,[bx+RqSec-@]	     ;Is all I-O in one cache block?
	jb s	GetAdr		     ;No, go do "regular" I-O below.
	or	[bx+RqCmd-@].lb,080h ;Do NOT cache data, after I-O!
	mov	eax,[bx+RqXMS-@]     ;Do DMA thru the cache buffer!
	jmp s	SetAd2		     ;Go do "buffered" I-O below.
GetAdr:	mov	eax,[bx+RqBuf-@]     ;Get user I-O buffer address.
	test	al,003h		     ;Is user buffer 32-bit aligned?
	jnz s	SetAd1		     ;No, use our XMS memory buffer.
	cmp	[bx+RqBuf-@].hw,009h ;Is DMA beyond our 640K limit?
	ja s	SetAd1		     ;Yes, use our XMS memory buffer.
	mov	cx,[bx+IOLen-@]	     ;Get (byte count -1) + I-O addr.
	dec	cx
	add	cx,ax		     ;Would I-O cross a 64K boundary?
	jnc s	SetAdr		     ;No, set final buffer address.
	nop			     ;(Unused alignment "filler").
SetAd1:	mov	eax,0		     ;Get our 32-bit XMS buffer addr.
@XBAddr	equ	[$-4].dwd	     ;(XMS buffer address, Init set).
SetAd2:	stc			     ;Set carry to say "our buffer".
	nop			     ;(Unused alignment "filler").
SetAdr:	mov	[bx+IOAdr-@],eax     ;Set final 32-bit buffer addr.
	jnc s	DoDMA		     ;If user buffer, do DMA & exit.
	test	[bx+RqCmd-@].lb,002h ;Buffered I-O:  Input request?
	jz s	UBufIn		     ;Yes, use input routine below.
	call	UBufMv		    ;Output:  Move user data to XMS.
	jnc s	DoDMA		    ;If no XMS error, do DMA & exit.
UdmaX:	ret			    ;XMS or I-O error!  Bail out now!
UBufIn:	call	DoDMA		    ;Input:  Read data to XMS buffer.
	jc s	UdmaX		    ;If any I-O errors, bail out now!
	nop			    ;(Unused alignment "filler").
UBufMv:	movzx	ecx,[bx+RqSec-@].lb ;Get number of sectors to move.
	mov	esi,[bx+IOAdr-@]    ;Set move addresses for a read.
	mov	edi,esi
	xchg	edi,[bx+RqBuf-@]
	jmp	MovBuf		    ;Move data to/from XMS and exit.
	db	0		    ;(Unused alignment "filler").
;
; UltraDMA Controller I-O Address Table.   "Legacy" addresses are
;   required in this table.   A PCI BIOS may return ZEROS for the
;   primary and secondary addresses of a "Legacy" IDE controller!
;
Ctl1Tbl	dw	0FFFFh		;Controller 1 DMA base address.
Ctl1Pri	dw	NPDATA		;   "Legacy" primary   addresses.
Ctl1Sec	dw	NSDATA		;   "Legacy" secondary addresses.
Ctl2Tbl	dw	0FFFFh		;Controller 2 DMA base address.
Ctl2Pri	dw	APDATA		;   "Legacy" primary   addresses.
Ctl2Sec	dw	ASDATA		;   "Legacy" secondary addresses.
	dw	24 dup (0FFFFh)	;Controller 3 to 10 I-O addresses.
CtlTEnd	equ	$		   ;End of controller I-O tables.
CTLTSIZ	equ	(Ctl2Tbl-Ctl1Tbl)  ;Size of a controller I-O table.
;
; Subroutine to handle hard-disk UltraDMA I-O requests for "UdmaIO".
;
DoDMA:	mov	esi,(IOAdr-@)	 ;Set command-list move parameters.
@CLAddr	equ	[$-4].dwd	 ;(Command-list address, Init set).
	mov	edi,[bx+PRDAd-@]
	mov	ecx,8
	call	MvData		;Move DMA command-list up to XMS.
DDBase:	jc s	DoDMAE		;Command-list move ERROR?  Exit NOW!
	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	di,[bx+IdeDA-@]	;Get our disk's IDE base address.
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	lea	dx,[di+CDSEL]
	out	dx,al
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,((STARTTO*256)+FLT) ;Get timeout & "fault" mask.
	add	ch,es:[si]	       ;Set timeout limit in CH-reg.
	call	ChkRdy		       ;Await controller/disk ready.
DoDMAE:	jc s	DoDMAX		       ;If any errors, exit below!
	test	[bx+IOCmd-@].lb,012h   ;Is this a write request?
	jnz s	SetDMA		       ;Yes, init DMA transfer.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA:	push	si		;Save BIOS timer pointer.
	mov	dx,[bx+DMAAd-@]	;Get DMA command-register address.
	mov	si,(PRDAd-@)	;Get Physical-Region Descriptor addr.
	out	dx,al		;Reset DMA commands and set DMA mode.
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	inc	dx		;Point to PRD address register.
	inc	dx
	outsd			;Set PRD pointer to our DMA address.
	mov	ax,001F7h	;Set IDE parameter-output flags.
NxtPar:	lea	dx,[di+CDATA+1]	;Point to IDE sector count reg. - 1.
IDEPar:	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	ax,1		;More parameters to go in this group?
	jc s	IDEPar		;Yes, loop back and output next one.
	jnz s	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	dec	dh		;"Legacy IDE" controller channel?
@DRQ:	jmp s	DMAGo		;No, forget about awaiting 1st DRQ.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ:	cmp	ch,es:[si]	;Too long without 1st data-request?
	je s	ErrDMA		;Yes?  Return carry and DMA error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz s	ChkDRQ		;No, loop back and check again.
DMAGo:	mov	es:[si+HDI_OFS],bl  ;Reset BIOS disk-interrupt flag.
	mov	dx,[bx+DMAAd-@]	    ;Point to DMA command register.
	in	al,dx		;Set DMA Start/Stop bit (starts DMA).
	inc	ax
	out	dx,al
ChkDMA:	inc	dx		;Point to DMA status register.
	inc	dx
	in	ax,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz s	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,es:[si]	;Has our DMA transfer timed out?
	je s	HltDMA		    ;Yes?  Halt DMA & post timeout!
	cmp	es:[si+HDI_OFS],bl  ;Did BIOS get a disk interrupt?
	je s	ChkDMA		    ;No, loop back & retest status.
	mov	al,DMI		;Set "simulated" interrupt flag.
HltDMA:	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne s	ErrDMA		;No?  Return carry and DMA error!
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz s	ErrDMA		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy:	lea	dx,[di+CSTAT]	;Read IDE primary status.
	in	al,dx
	cmp	ch,es:[si]	;Too long without becoming ready?
	je s	RdyErr		;Yes?  Go see what went wrong.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle s	ChkRdy		;Yes, loop back and check again.
	and	al,cl		;Disk "fault" or hard-error?
	jnz s	HdwErr		;Yes?  Go see what went wrong.
DoDMAX:	ret			;End of request -- Return to "UdmaIO".
ErrDMA:	mov	al,DMAERR	;BAAAD News!  Post DMA error code.
	stc			;Set carry flag (error!) and exit.
	ret
RdyErr:	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp s	WhichE		;Go see which error code to return.
HdwErr:	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*FAULTED)+HARDERR ;(Get hardware error codes).
WhichE:	jz s	Kaput		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
Kaput:	stc			;Set carry flag (error!) and exit.
	ret
DDLEN	equ	($-DDBase)	;(Length of regular "DoDMA" logic).
HMALEN	equ	($-HDReq)	;(Length of all HMA caching logic).
RLF	equ	($-SALowM)	;("Stand alone" relocation factor).
;
; "Stand Alone" Driver Int 13h Entry Routine.   Driver Init will
;   overlay this and all following "stand alone" routines on our
;   caching routines, when the /B switch is specified.
;
SEntry:	pushf			;Save CPU flags & BP-reg.
	push	bp
	mov	bp,0		;Reset active-units table index.
@SLastU	equ	[$-2].lw	;(Last-unit index, set by Init).
SNextU:	dec	bp		;Any more active units to check?
	js s	SQuikX		    ;No, not for us -- exit quick!
	cmp	dl,cs:[bp+Units-@]  ;Does request unit match table?
	jne s	SNextU		    ;No, see if more units remain.
	cli			;Disable CPU interrupts.
	bts	cs:IOF.lb,0	;Is this driver currently "busy"?
	jc s	SInvF		;Yes?  Set "invalid function" & exit!
	mov	cs:SCStak.lw,sp	;Save caller's stack pointer.
	mov	cs:SCStak.hw,ss
	push	cs		;Switch to our driver stack, to avoid
	pop	ss		;  CONFIG.SYS "STACKS=0,0" problems!!
	mov	sp,0		;Set starting stack address.
@SStack	equ	[$-2].lw	;(Initial stack address, Init set).
	sti			;Re-enable CPU interrupts.
	push	ds		;Save CPU segment registers.
	push	es
	pusha			;Save all 16-bit CPU registers.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	SA20Rq
	jc s	S20Err		;If "A20" failure, ABANDON request!
	popa			;Restore and restack all Int 13h I-O
	pop	es		;  parameters, after "A20 enable" --
	push	es		;  Looks wasteful, but one learns to
	pusha			;  never-NEVER "trust" external code!
	mov	dl,0BEh		;Mask out LBA & write request bits.
	and	dl,ah
	db	0EAh		;Go to main Int 13h request handler.
@SMainE	dd	(SHDReq-@-RLF)	;(Main entry address, set by Init).
SPassR:	call	SA20Ds		;Pass request -- Disable "A20" line.
	cli			;Disable CPU interrupts.
	mov	[bx+IOF-@],bx	;Reset "busy" and "VDS lock" flags.
	popa			;Reload all 16-bit CPU registers.
	pop	es		;Reload CPU segment registers.
	pop	ds
	lss	sp,cs:SCStak	;Switch back to caller's stack.
SQuikX:	pop	bp		;Reload BP-reg. from caller's stack.
	popf			;Reload CPU flags saved on entry.
	db	0EAh		;Go to next routine in Int 13h chain.
@SPrv13	dd	0		;(Prior Int 13h vector, set by Init).
SInvF:	mov	ah,001h		;We are BUSY, you fool!  Set "invalid
	jmp s	SQuit		;  function" code and get out, QUICK!
SExit:	call	SA20Ds		;Exit -- issue "A20 local disable".
	pop	ax		;Reload error code and error flag.
	popf
S20Err:	mov	bp,sp		;Point to saved registers on stack.
	mov	[bp+15],al	;Set error code in exiting AX-reg.
	cli			;Disable CPU interrupts.
	mov	[bx+IOF-@],bx	;Reset "busy" and "VDS lock" flags.
	popa			;Reload all 16-bit CPU registers.
	pop	es		;Reload CPU segment registers.
	pop	ds
	lss	sp,cs:SCStak	;Switch back to caller's stack.
SQuit:	mov	bp,sp		;Set error flag in exiting carry bit.
	rcr	[bp+8].lb,1	;(One of the flags saved by Int 13h).
	rol	[bp+8].lb,1
	pop	bp		;Reload BP-reg. from caller's stack.
	popf			;Discard CPU flags saved on entry.
	iret			;Exit -- Int 13h data reloads flags!
;
; Subroutine to issue "A20" local-enable and local-disable requests.
;
SA20Ds:	mov	ah,006h		;"A20 local disable" -- get XMS code.
SA20Rq:	db	09Ah		;Issue "A20" request to the XMS mgr.
@XMgr2:	clc			;(Without XMS, 090h above, and this
	mov	ax,00001h	; logic posts "No XMS error" status).
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	xor	bx,bx		;Rezero BX-reg. for relative logic.
	sub	al,1		;Zero AL-reg. if success, -1 if error.
	ret			  ;Exit.
SLMLEN	equ	($-SEntry)+SABASE ;("Stand alone" base logic length).
SLMEnd	equ	(SLMLEN+STACKSA)  ;(End of "stand alone" low-memory).
;
; Main hard-disk request handler.
;
SHDReq:	cmp	dl,002h		;CHS or LBA read/write request?
	jne s	SPass		;No, not for us -- pass this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns s	SValSC		;No, go validate CHS sector count.
	mov	di,sp		;Point DI-reg. to current stack.
	push	ds		;Save this driver's DS-reg.
	mov	ds,[di+18]	;Reload "DAP" segment into DS-reg.
	cmp	[si].DapBuf,-1	;Check for 64-bit "DAP" I-O buffer.
	mov	al,[si].DapSC	;(Get "DAP" I-O sector count).
	les	dx,[si].DapLBA1	;(Get "DAP" LBA bits 16-47).
	mov	di,es
	les	cx,[si].DapBuf	;(Get "DAP" I-O buffer address).
	mov	si,[si].DapLBA	;(Get "DAP" LBA bits 0-15).
	pop	ds		;(Reload this driver's DS-reg.).
	jne s	SValSC		;If no 64-bit buffer, check sector ct.
SPass:	push	ss		;Return to request "pass" routine.
	push	(SPassR-@-RLF)
	retf
SValSC:	dec	al		;Is sector count zero or over 128?
	js s	SPass		;Yes?  Let BIOS handle this "No-No!".
	inc	ax		;Restore sector count -- LBA request?
	js s	SZroBX		;Yes, go zero driver's BX-reg.
	xchg	ax,cx		;CHS -- save request code and sectors.
	xor	di,di		;Reset upper LBA address bits.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	shr	al,6		;Set AX-reg. to starting cylinder.
	xchg	al,ah
	xchg	ax,dx			  ;Swap cylinder/head values.
	mov	al,cs:[bp+(SCHSec-@-RLF)] ;Get CHS sectors per head.
@SCHSec	equ	[$-2].lw
	or	al,al		   ;Were disk CHS values legitimate?
	jz s	SPass		   ;No?  Let BIOS have this request!
	push	ax		   ;Save CHS sectors/head value.
	mul	ah		   ;Convert head to sectors.
	add	si,ax		   ;Add result to starting sector.
	pop	ax			  ;Reload sectors per head.
	mul	cs:[bp+(SCHSHd-@-RLF)].lb ;Convert cyl. to sectors.
@SCHSHd	equ	[$-2].lw
	mul	dx
	add	si,ax		   ;Add to head/sector value.
	adc	dx,di
	xchg	ax,bx		   ;Get buffer offset in AX-register.
	xchg	ax,cx		   ;Swap offset with request/sectors.
SZroBX:	xor	bx,bx		   ;Zero BX-reg. for relative logic.
	mov	[bx+VDSOf-@],cx    ;Set VDS I-O buffer address.
	mov	[bx+VDSSg-@],es
	mov	[bx+LBA-@],si	   ;Set 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+1-@],di
	mov	bp,cs:[bp+(SCtlUn-@-RLF)] ;Get disk ctlr. & unit nos.
@SCtlUn	equ	[$-2].lw
	mov	si,offset (SCtlTbl-@-RLF) ;Point to ctlr. addresses.
@SCtlTb	equ	[$-2].lw
	mov	cx,0007Ch
	and	cx,bp
	add	si,cx
	shr	cx,1
	add	si,cx
	push	cs:[si].dwd	   ;Set controller base address and
	pop	[bx+DMAAd-@].dwd   ;  primary-channel data address.
	rcr	bp,2		   ;Primary channel I-O request?
	jnc s	SGetCm		   ;Yes, get LBA28 or LBA48 commands.
	add	[bx+DMAAd-@].lw,8  ;Use secondary DMA controller.
	mov	cx,cs:[si+4]	   ;Set secondary channel data addr.
	mov	[bx+IdeDA-@],cx
	nop			   ;(Unused alignment "filler").
SGetCm:	shr	dx,12		   ;Shift out LBA bits 16-27.
	or	di,dx		   ;Anything in LBA bits 28-47?
	jz s	SLBA28		   ;No, use LBA28 read/write command.
	shl	ah,3		   ;LBA48 -- get command as 020h/030h.
	jmp s	SSetCm		   ;Go set LBA and IDE command bytes.
SLBA28:	xchg	dh,[bx+LBA2-@]	   ;LBA28:  Reload & reset bits 24-27.
	or	ah,(DRCMD+1)	   ;Get LBA28 read/write command + 5.
SSetCm:	shl	bp,1		   ;Shift master/slave into LBA cmds.
	mov	dl,(LBABITS/32)
	rcl	dl,5
	or	dl,dh		     ;Or in LBA28 bits 24-27, if any.
	mov	dh,005h		     ;Get final IDE command byte.
	xor	dh,ah		     ;(LBA28 C8h/CAh, LBA48 25h/35h).
	mov	[bx+DSCmd-@],dx	     ;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	     ;Set disk I-O sector count.
	mov	ah,0		     ;Set VDS & I-O buffer lengths.
	shl	ax,1		     ;  We set bit 16 for 64K bytes
	mov	[bx+VDSLn+1-@],ax    ;  since a few "odd" chips may
	mov	[bx+IOLen+1-@],ax    ;  NOT work O.K. without it!
	or	[bx+IOAdr-@].dwd,-1  ;Invalidate VDS buffer address.
	mov	ax,08103h	     ;Get VDS "lock" parameters.
	mov	dx,0000Ch
	mov	di,(VDSLn-@)	     ;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		     ;Issue VDS "lock" request.
	call	SXMRst		     ;Restore all critical settings.
	jc	SPass		     ;VDS error?  Pass this request!
	push	eax		     ;Save EAX-reg. for 32-bit work.
	mov	eax,[bx+IOAdr-@]     ;Get our 32-bit VDS address.
	cmp	eax,-1		     ;Is VDS address still all-ones?
	jb s	SSetVF		     ;No, go set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw  ;VDS logic is NOT present --
	shl	eax,4		     ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
	mov	[bx+IOAdr-@],eax     ;Set final 32-bit DMA address.
SSetVF:	adc	[bx+VLF-@],bl	     ;Set VDS "lock" flag from carry.
	pop	eax		     ;Reload EAX-reg. (unneeded now).
	test	[bx+IOAdr-@].lb,003h ;Is user buffer 32-bit aligned?
	jnz s	SSetBf		     ;No, use our XMS memory buffer.
	cmp	[bx+IOAdr-@].hw,009h ;Is DMA beyond our 640K limit?
	ja s	SSetBf		     ;Yes, use our XMS memory buffer.
	mov	cx,[bx+IOLen-@]	     ;Get (I-O length -)1 + I-O addr.
	dec	cx
	add	cx,[bx+IOAdr-@].lw   ;Will I-O cross a 64K boundary?
	jnc s	SDrDMA		     ;No, do direct user-buffer DMA.
	nop			     ;(Unused alignment "filler").
SSetBf:	mov	[bx+IOAdr-@].dwd,0   ;Set our 32-bit XMS buffer addr.
@SXBAdr	equ	[$-4].dwd	     ;(XMS buffer address, Init set).
	test	[bx+IOCmd-@].lb,012h ;Buffered I-O:  Write request?
	jnz s	SBfOut		     ;Yes, use output routine below.
	call	SDoDMA		;Buffered input -- read data to XMS.
	jc s	SHDone		;If error, post return code and exit.
	call	SXMMov		;Move data from XMS to user buffer.
	jmp s	SHDone		;Go post any return code and exit.
SBfOut:	call	SXMMov		;Buffered output -- move data to XMS.
	jc s	SHDone		;If error, post return code and exit.
SDrDMA:	call	SDoDMA		;Do direct DMA or buffered output.
SHDone:	pushf			;Save error flag (carry) and code.
	push	ax
	shr	[bx+VLF-@].lb,1	;User I-O buffer "locked" by VDS?
	jnc s	SHDEnd		;No, return to hard-disk exit rtn.
	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	ds
	pop	es
	int	04Bh		;Issue VDS "unlock" request.
SHDEnd:	push	ss		;Return to hard-disk exit routine.
	push	(SExit-@-RLF)
	retf
;
; Fixed Command-List XMS Block for moving our DMA command-list to XMS.
;
CLXBlk	dd	8		;Move length      (always 8 bytes).
	dw	0		;Source "handle"     (always zero).
CLXSA	dd	(IOAdr-@)	;Source address (seg. set by Init).
CLXDH	dw	0		;Destination handle  (set by Init).
CLXDA	dd	0		;Destination offset  (set by Init).
;
; User-Data XMS Block, used to move data to/from our XMS buffer.
;
XMSBlk	equ	$		;User-data XMS block (16 bytes).
;
; Initialization Variables, located here to mimimize our Init logic.
;   After Init, this area is used as our "stand alone" driver's User
;   data XMS block.
;
InitPkt	dd	0		;DOS "Init" packet address.
XMSBase	dd	000020000h	;Starting XMS "base" address.
HMASize	dw	HMALEN		;Total HMA space required.
EntAddr	dw	(Entry-@)	;Int 13h "entry address.
StakSiz	dw	STACK		;Driver local-stack size.
MvUpAdr dw      (HDReq-@)	;First byte "moved up" to the HMA.
;
; Subroutine to move user I-O data to or from our XMS buffer.
;
SXMMov:	push	[bx+VDSSg-@].lw	  ;Save user-buffer address.
	push	[bx+VDSOf-@].lw
	push	[bx+VDSLn-@].dwd  ;Get user-buffer length.
	mov	si,(XMSBlk-@-RLF) ;Point to user-data XMS block.
@XMSBlk	equ	[$-2].lw
	push	cs
	pop	ds
	pop	[si].dwd	;Set user-buffer length.
	lea	bx,[si+4]	;Set move indexes for a read.
	lea	di,[si+10]
	jz s	SXMSet		;Is this a read request?
	xchg	bx,di		;No, "swap" indexes for a write.
SXMSet:	mov	[bx].lw,0	;Set our XMS buffer "handle" & offset.
@XMHdl	equ	[$-2].lw
	mov	[bx+2].dwd,0
@XMOffs	equ	[$-4].dwd
	and	[di].lw,0	;Set user buffer "handle" to zero.
	pop	[di+2].dwd	;Set user-buffer address.
SXMGo:	mov	ah,00Bh		;Get XMS "move memory" code.
SXMSRq:	db	09Ah		;Issue XMS "move memory" request.
@XMgr3:	clc			;(Without XMS, 090h above, and this
	mov	ax,00001h	; logic posts "No XMS error" status).
	sub	al,1		;Zero AL-reg. if success, -1 if error.
SXMRst:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	ss
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry flag.
	ret			;Exit.
;
; "Stand Alone" Driver Controller Address Table and Disk Parameter
;   tables, moved-up by driver Init from the caching tables above.
;
SCtlTbl	equ	$		   ;Controller 1 to 10 addresses.
SCtlUn	equ	($+60)		   ;Disk controller & unit table.
SCHSec	equ	($+60+MAXBIOS)     ;Disk sectors-per-head table.
SCHSHd	equ	($+60+(MAXBIOS*2)) ;Disk heads-per-cylinder table.
STBLSIZ	equ	(60+(MAXBIOS*3))   ;("Stand alone" tables size).
;
; Init subroutine to set cache parameter-table block sizes.   This
;   routine is placed here to minimize the size of our Init logic.
;   After Init, this area is used as our "stand alone" driver disk
;   parameter and controller address tables, as described above.
;
I_CBlk:	mov	[di+24].dwd,"    " ;Reset "title" message cache size.
I_CB1:	mov	cx,08000h	   ;Invalidate cache-block count.
I_CB2:	inc	si		;Bump ptr. past "S" or last digit.
	movzx	ax,es:[si].lb	;Get next command-line byte.
	cmp	al,'9'		;Is byte greater than a '9'?
	ja s	I_CB3		;Yes, see what cache size is wanted.
	cmp	al,'0'		;Is byte less than a '0'?
	jb s	I_CB3		;Yes, see what cache size is wanted.
	cmp	[di+24].lb,' '	;Have we already found 4 digits?
	jne s	I_CB1		;Yes, set INVALID & keep scanning.
	push	[di+25].dwd	;Shift "title" bytes 1 place left.
	pop	[di+24].dwd
	mov	[di+27],al	;Insert next "title" message byte.
	and	al,00Fh		;Get cache digit's binary value.
	xchg	ax,cx		;Swap block size & next digit.
	mov	dx,10		;Multiply current block size by 10.
	mul	dx
	add	cx,ax		;"Add in" next cache-size digit.
	jmp s	I_CB2		;Go scan more cache-size digits.
I_CB3:	mov	bp,(Cache5-@)	;Get 5-MB cache table pointer.
	cmp	cx,15		;Does user want less than 15-MB?
	jb s	I_CB4		;Yes, give user a 5-MB cache.
	mov	bp,(Cache25-@)	;Get 25-MB cache table pointer.
	cmp	cx,25		;Does user want exactly 25-MB?
	je s	I_CB4		;Yes, give user a 25-MB cache.
	mov	bp,(Cache15-@)	;Get 15-MB cache table pointer.
	cmp	cx,40		;Does user want less than 40-MB?
	jb s	I_CB4		;Yes, give user a 15-MB cache.
	mov	bp,(Cache50-@)	;Get 50-MB cache table pointer.
	cmp	cx,50		;Does user want exactly 50-MB?
	je s	I_CB4		;Yes, give user a 50-MB cache.
	mov	bp,(Cache40-@)	;Get 40-MB cache table pointer.
	cmp	cx,80		;Does user want less than 80-MB?
	jb s	I_CB4		;Yes, give user a 40-MB cache.
	mov	bp,(CacheLC-@)	;Get "large" cache table pointer.
	cmp	cx,MXCACHE	;Is cache size invalid or too big?
	jbe s	I_CB4		;No, give user our "large" cache.
	xor	cx,cx		;Get "null" /C or /X megabytes.
	cmp	di,(CCTbl-@)	;Were we handling the /S switch?
	jne s	I_CB4		   ;No, reset "null" megabytes.
	mov	[di+24].dwd,"08  " ;Restore 80-MB "title" bytes.
	mov	cl,80		   ;Get default 80 megabytes.
I_CB4:	mov	[di+6],cx	;Set megabytes in parameter table.
	mov	[di+10],bp	;Save cache-table pointer for later.
	ret			;Exit.
SecMsg	db	'Secondary-$'	;("Secondary" IDE channel message,
				;   placed here to save Init space).
;
; Subroutine to execute "stand alone" driver UltraDMA read or write
;   requests.   But for its initial commands below, this subroutine
;   is identical to "DoDMA" above, whose logic will be moved up to
;   "SDDBase" for use by the run-time "stand alone" driver.
;
SDoDMA:	mov	si,(CLXBlk-@-RLF) ;Point to command-list XMS block.
@CLXBlk	equ	[$-2].lw
	push	cs
	pop	ds
	nop			  ;(Unused alignment "filler").
	call	SXMGo		  ;Move DMA command-list up to XMS.
SDDBase	equ	$		  ;(Remainder of "DoDMA" goes here).
SHMALEN	equ	($+DDLEN-SHDReq)  ;(Length of all HMA driver logic).
;
; Init subroutine to convert 4-digit hex values to ASCII for display.
;   At entry, the value is in the AX-reg., and the message pointer is
;   in the SI-reg.   The SI-reg. is updated.   The CX-reg. is zeroed.
;   This subroutine is located here so it can become the space needed
;   by "SDoDMA" above.
;
I_Hex:	mov	cx,4		;Set 4-digit count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Store next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
I_HexX:	ret			;Exit.
;
; Init subroutine to test for and set up PCI disk controllers.   This
;   subroutine is placed here so it can become the extra space needed
;   by "SDoDMA" above. 
;
I_PCIC:	mov	IdeDA,ax	;Save subclass & function codes.
	and	LBABuf.hw,0	   ;Reset PCI device index.
I_PCI1:	cmp	PCITbl,(CtlTEnd-@) ;More space in address tables?
 	jae s	I_HexX		   ;No, go exit above.
	mov	ax,IdeDA	;Test PCI class 1, subclass/function.
	mov	ecx,000010003h	;(Returns bus/device/function in BX).
	xchg	ax,cx
	mov	si,LBABuf.hw
	call	I_In1A
	jc s	I_HexX		;Controller not found -- exit above.
	inc	LBABuf.hw	;Bump device index for another test.
	xor	di,di		;Get controller Vendor & Device I.D.
	call	I_PCID
	push	ecx		;Save Vendor and Device I.D.
	mov	di,32		;Save DMA controller base address.
	call	I_PCID
	xchg	ax,cx
	and	al,0FCh
	mov	DMAAd,ax
	mov	si,offset (PCMsg2-@) ;Set controller address in msg.
	call	I_Hex
	mov	si,offset (PCMsg3-@) ;Set vendor I.D. in message.
	pop	ax
	call	I_Hex
	pop	ax		;Set Device I.D. in message.
	call	I_Hex
	mov	di,4		;Get low-order PCI command byte.
	call	I_PCID
	not	cl		;Get "Bus Master" & "I-O Space" bits.
	and	cl,005h		;Is controller using both BM and IOS?
	jz s	I_PCI2		     ;Yes, save all controller data.
	mov	dx,offset (BCMsg-@)  ;Display "BAD controller!", and
	call	I_Msg		     ;  hope user gets a better BIOS!
	mov	dx,offset (PCMsg1-@)
	call	I_Msg
I_PCIJ:	jmp s	I_PCI1		;Go test for more same-class ctlrs.
I_PCI2:	mov	si,PCITbl	;Get current I-O address table ptr.
	mov	ax,DMAAd	;Set controller DMA base address.
	mov	[si],ax
	test	LBABuf.lb,001h	;Is this a "Native PCI" controller?
	jz s	I_PCI3		;No, go display controller data.
	mov	di,16		;Set primary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+2],cx
	mov	di,24		;Set secondary-channel base address.
	call	I_PCID
	and	cl,0FCh
	mov	[si+4],cx
I_PCI3:	mov	dx,offset (PCMsg-@) ;Display all controller data.
	call	I_Msg
	inc	[PCMsg+3].lb	;Bump controller number in message.
	add	PCITbl,CTLTSIZ	;Bump controller address-table ptr.
	jmp s	I_PCIJ		;Go test for more same-class ctlrs.
;
; Initialization Messages.
;
TTLMsg	db	CR,LF,'UHDD'
	ifdef	CACHE4
	db	'-UC'		;Add -UC with user caches.
	endif
	db	', 1-26-2014.',CR,LF,'$'
CTSize	db	'    -MB '
CTName	db	'Common Cache.',CR,LF,'$'
NPMsg	db	'No V2.0C+ PCI!',CR,LF,'$'
DNMsg	db	' disk is '
DName	equ	$
DNEnd	equ	DName+40
BCMsg	db	'BAD$'
PCMsg	db	'IDE0'
PCMsg1	db	' Controller at I-O address '
PCMsg2	db	'PCI h, Chip I.D. '
PCMsg3	db	'        h.',CR,LF,'$'
NonUMsg	db	'Disks run by the BIOS:  '
NonUCt	db	'0.',CR,LF,'$'
DPTEMsg	db	'DPTE$'
CHSMsg	db	'CHS'
UnitMsg	db	' data BAD, unit '
UnitNo	db	'A: ',CR,LF,'$'
CtlMsg	db	'IDE'
CtlrNo	db	'0 $'
PriMsg	db	'Primary-$'
MstMsg	db	'master$'
SlvMsg	db	'slave$'
IEMsg	db	' Identify ERROR!',CR,LF,'$'
NDMsg	db	'Nothing to use$'
VEMsg	db	'VDS init error$'
CPUMsg	db	'No 386+ CPU'
Suffix	db	'; UHDD'
	ifdef	CACHE4
	db	'-UC'		;Add -UC with user caches.
	endif
	db	' not loaded!',CR,LF,'$'
;
; Initialization "Strategy" Routine.   This MUST be placed above all
;   run-time logic, to prevent CPU cache "code modification" ERRORS!
;
I_Stra:	mov	cs:InitPkt.lw,bx ;Save DOS request-packet address.
	mov	cs:InitPkt.hw,es
	retf			 ;Exit and await "Device Interrupt".
;
; Initialization "Device Interrupt" Routine.   This is the main Init
;   routine for the driver.
;
I_Init:	pushf			;Entry -- save CPU flags.
	push	ds		;Save CPU segment registers.
	push	es
	push	ax		;Save needed 16-bit CPU registers.
	push	bx
	push	dx
	xor	ax,ax		;Get a zero for following logic.
	lds	bx,cs:InitPkt	;Point to DOS "Init" packet.
	cmp	[bx].RPOp,al	;Is this really an "Init" packet?
	jne s	I_Exit			;No?  Reload regs. and exit!
	mov	[bx].RPStat,RPDON+RPERR	;Set "Init" packet defaults
	mov	[bx].RPSeg,cs		;  and "null" driver length.
	and	[bx].RPLen,ax
	push	cs		;NOW point DS-reg. to this driver!
	pop	ds
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax		;(80286+ push SP, then decrement it).
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes?  CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_Sv32		;Yes, go save 32-bit CPU registers.
I_Junk:	mov	dx,(CPUMsg-@)	;Point to "No 386+ CPU" error msg.
I_Quit:	call	I_Msg		;Display "No 386+ CPU" or msg. suffix.
I_Exit:	jmp	I_Bye		;Go reload 16-bit regs. and exit.
I_VErr:	mov	VEMsg.dwd,eax	;Set prefix in "VDS init error" msg.
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_XDis:	mov	dx,CStack.lw	;Load our XMS "handle" number.
	or	dx,dx		;Have we reserved any XMS memory?
	jz s	I_LDMP		;No, reload pointer & display msg.
	mov	ah,00Dh		;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_LDMP:	pop	dx		;Reload error message pointer.
I_EMsg:	call	I_Msg		;Display desired error message.
	popad			;Reload all 32-bit CPU registers.
	mov	dx,(Suffix-@)	;Point to "UHDD not loaded!" suffix.
	jmp s	I_Quit		;Go display message suffix and exit!
	align	4
EDDBuff	equ	$		;(Start of EDD & search-index buffer).
I_Sv32:	pushad			;Save all 32-bit CPU registers.
	les	si,es:[bx].RPCL	;Get command-line data pointer.
	xor	bx,bx		;Zero BX-reg. for relative logic.
I_NxtC:	cld			;Ensure FORWARD "string" commands!
	lods	es:[si].lb	;Get next command byte and bump ptr.
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_TrmJ		;Yes, go validate desired cache size.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_TrmJ		;Yes, go validate desired cache size.
	cmp	al,CR		;Is byte an ASCII carriage-return?
I_TrmJ:	je	I_Term		;Yes, go validate desired cache size.
	cmp	al,'/'		;Is byte a slash?
	je s	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,'-'		;Is byte a dash?
	jne s	I_NxtC		;No, check next command-line byte.
I_NxtS:	mov	ax,es:[si]	;Get next 2 command-line bytes.
	and	al,0DFh		;Mask out 1st byte's lower-case bit.
	cmp	al,'A'		;Is this byte an "A" or "a"?
	jne s	I_ChkB		   ;No, see if byte is "B" or "b".
	mov	al,(ASDATA-0100h)  ;Reverse all "Legacy IDE" addrs.
	mov	Ctl1Sec.lb,al
	mov	al,(APDATA-0100h)
	mov	Ctl1Pri.lb,al
	mov	al,(NSDATA-0100h)
	mov	Ctl2Sec.lb,al
	mov	al,(NPDATA-0100h)
	mov	Ctl2Pri.lb,al
I_ChkB:	cmp	al,'B'		;Is switch byte a "B" or "b"?
	jne s	I_ChkC		;No, see if byte is "C" or "c".
	call	I_SetB		;Set up for "stand alone" driver.
I_ChkC:	mov	di,(CDTbl-@)	;Point to CD/DVD cache parameters.
	cmp	al,'C'		;Is switch byte a "C" or "c"?
	je s	I_GetB		;Yes, get "CD/DVD" cache block count.
	cmp	al,'E'		;Is switch byte an "E" or "e"?
	jne s	I_ChkH		  ;No, see if byte is "H" or "h".
	cmp	[bx+DNam2+2-@],bx ;"Stand alone" driver requested?
	jne s	I_ChkH		  ;Yes, must reject the /E switch!
	mov	EFlag,al	  ;Set "DOS emulator active" flag.
	mov	HMASize,ESWLEN    ;Dismiss caching UltraDMA driver.
I_ChkH:	cmp	al,'H'		;Is switch byte an "H" or "h"?
	jne s	I_ChkZ		;No, see if byte is "Z" or "z".
	mov	HFlag,al	;Set "use HMA space" flag.
I_ChkZ:	cmp	al,'Z'		;Is 1st switch byte a "Z" or "z"?
	jne s	I_ChkQ		;No, see if byte is "Q" or "q".
	mov	@MvP2K,bl	;Do slower but "safe" 2K XMS moves.
I_ChkQ:	cmp	al,'Q'		;Is switch byte a "Q" or "q"?
	jne s	I_ChkR		;No, see if byte is "R" or "r".
	mov	@DRQ.lb,075h	;Await data-request before UltraDMA.
I_ChkR:	cmp	al,'R'		;Is switch byte an "R" or "r"?
	jne s	I_ChkS		;No, see if byte is "S" or "s".
	mov	ax,es:[si+1]	;Get next 2 command-line bytes.
	mov	cx,15296	;Get 15-MB XMS memory size.
	cmp	ax,"51"		;Does user want 15-MB XMS reserved?
	je s	I_CkRA		;Yes, set memory size to reserve.
	mov	ch,(64448/256)	;Get 63-MB XMS memory size.
	cmp	ax,"36"		;Does user want 63-MB XMS reserved?
	jne s	I_NxtJ		;No, continue scan for a terminator.
I_CkRA:	mov	CStack.hw,cx	;Set desired XMS memory to reserve.
I_ChkS:	mov	di,(CCTbl-@)	;Point to "common" cache parameters.
	cmp	al,'S'		;Is switch byte an "S" or "s"?
	ifdef	CACHE4
	je s	I_GetB		;Yes, get "Common" cache block count.
	mov	di,(U1Tbl-@)	;Point to "User-1" cache parameters.
	cmp	al,'X'		;Is switch byte an "X" or "x"?
	je s	I_GetB		;No, continue scan for a terminator.
	mov	di,(U2Tbl-@)	;Point to "User-2" cache parameters.
	cmp	al,'Y'		;Is switch byte a "Y" or "y"?
	endif
	jne s	I_NxtJ		;No, continue scan for a terminator.
I_GetB:	call	I_CBlk		;Get block count for this cache.
I_NxtJ:	jmp	I_NxtC		;Continue scanning for a terminator.
I_Term:	mov	ax,04300h	;Inquire if we have an XMS manager.
	call	I_In2F
	cmp	al,080h		;Is an XMS manager installed?
	je s	I_GetX		;Yes, get & save its "entry" address.
	call	I_SetB		;Default to our "stand alone" driver.
	mov	al,090h		;Disable calls to the XMS manager.
	mov	SA20Rq.lb,al
	mov	SXMSRq.lb,al
	db	066h,0B8h	;Pass "XMS buffered" I-O to the BIOS.
	db	0E9h
	dw	(SPass-(SSetBF+3))
	db	0	
	mov	SSetBf.dwd,eax
	jmp s	I_NoHM		;Go disable loading in the HMA.
I_GetX:	mov	ax,04310h	;Get and save XMS "entry" addresses.
	call	I_In2F
	push	es
	push	bx
	pop	eax
	mov	@XMgr1.dwd,eax
	mov	@XMgr2.dwd,eax
	mov	@XMgr3.dwd,eax
	shl	HFlag,1		;Will we be loading in the HMA?
	jz s	I_NoHM		;No, go set upper/DOS memory size.
	mov	ax,04A01h	;Get total "free HMA" space.
	call	I_In2F
	cmp	bx,HMASize	;Enough HMA for our driver logic?
	jae s	I_TTL		;Yes, go display driver "title" msg.
I_NoHM:	mov	HFlag,0		;Ensure NO use of HMA space!
	mov	ax,HMASize	;Set driver upper/DOS memory size.
	add	VDSLn.lw,ax
I_TTL:	mov	dx,(TTLMsg-@)	;Display driver "title" message.
	call	I_Msg
	cmp	DNam2.lb,'-'	;"Stand alone" driver requested?
	je	I_I13V		;Yes, go save current Int 13h vector.
	mov	di,(CCTbl-@)	;Initialize parameter-table pointer.
I_FCPT:	cmp	[di+6].lw,0	;Any cache blocks set in this table?
	je s	I_FCP4		;No (omitted), go check next table.
	mov	si,[di+16]	;"Activate" this parameter table.
	mov	[si],di
	mov	si,[di+10]	;Reload cache-size table pointer.
	mov	eax,[si+4]	;Set "variable" cache parameters.
	mov	[di+10],eax
	mov	ax,128		;Set initial 128-MB cache limit.
	cmp	[si+2].lb,128	;"Large" data cache requested?
	je s	I_FCP1		;Yes, adjust variable parameters.
	mov	ax,[si]		;Set desired cache-block count.
	mov	[di+6],ax
	mov	eax,[si+8]	;Set desired "title" message bytes.
	mov	[di+24],eax
	jmp s	I_FCP3		;Go set remaining cache parameters.
I_FCP1:	cmp	[di+6],ax	;User cache size < current limit?
	jb s	I_FCP2		;Yes, set user cache-block count.
	shl	ax,1		;Double current cache-size limit.
	shl	[di+10].dwd,1	;Double variable cache parameters.
	jmp s	I_FCP1		;Go check user's cache size again.
I_FCP2:	shl	[di+6].lw,4	;Cache blocks = (16 * Megabytes).
I_FCP3:	mov	eax,[di+24]	;Set cache size in display message.
	mov	CTSize.dwd,eax
	mov	ax,[di+18]	;Set cache name in display message.
	mov	CTName.lw,ax
	mov	eax,[di+20]
	mov	[CTName+2].dwd,eax
	push	di		;Display this cache size.
	mov	dx,(CTSize-@)
	call	I_Msg
	pop	di
	mov	ax,[si+2]	;Set cache "granularity":  sectors
	mov	[di+8],ax	;  per block and sector-shift count.
	mov	ah,0		;Multiply number of cache blocks
	shr	ax,1		;  times (sectors-per-block / 2).
	mul	[di+6].lw
	push	dx		;Get 32-bit cache XMS memory size.
	push	ax
	pop	eax
	mov	[di+16],eax	;Save cache XMS size for below.
	movzx	ecx,[di+12].lw	;Add in cache-tables XMS memory.
	add	eax,ecx
	add	RqXMS,eax	;Add to total required XMS memory.
I_FCP4:	add	di,PTSIZE	;Skip to next cache parameter table.
	cmp	di,(CDTbl-@)	;Any more tables left to finalize?
	jbe	I_FCPT		;Yes, loop back and do next one.
I_I13V:	mov	ax,03513h	;Get and save current Int 13h vector.
	call	I_In21
	push	es
	push	bx
	pop	eax
	mov	@Prev13,eax
	mov	@SPrv13,eax
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	bx,(MAXBIOS*3)	;Clear disk/diskette parameters.
I_ClrP:	dec	bx		;(Cache-size table no longer needed.
	mov	TypeF[bx],al	; Less "confusing" debug, if unused
	jnz s	I_ClrP		; parameter bytes have been zeroed!).
	mov	es,ax		;Point ES-reg. to low memory.
	mov	al,es:[HDISKS]	;Save our BIOS hard-disk count.
	mov	BiosHD,al
	mov	ax,cs		;Set fixed driver segment values.
	mov	VDSSg.lw,ax
	mov	@HDMain.hw,ax
	mov	@HDXSeg,ax
	mov	@SMainE.hw,ax
	mov	CLXSA.hw,ax
	shl	eax,4		;Set driver's VDS 20-bit address.
	mov	IOAdr,eax
	cli			     ;Avoid interrupts in VDS tests.
	test	es:[VDSFLAG].lb,020h ;Are "VDS services" active?
	jz s	I_REnI		     ;No, re-enable CPU interrupts.
	mov	ax,08103h	;"Lock" driver in memory forever, so
	mov	dx,0000Ch	;  EMM386 may NOT re-map this driver
	call	I_VDS		;  when doing UltraDMA -- bad CRASH!
	mov	dx,(VEMsg-@)	;Point to "VDS init error" message.
	jc	I_EMsg		;"Lock" error?  Display msg. & exit!
I_REnI:	sti			;Re-enable CPU interrupts.
	mov	eax,IOAdr	;Get final driver 32-bit address.
	add	PRDAd,eax	;Relocate "No XMS" PRD address.
	add	GDTPAdr,eax	;Relocate "real mode" GDT base addr.
	add	@CLAddr,eax	;Relocate command-list source address.
	add	@WrkBf1,eax	;Relocate "working" buffer addresses.
	add	@WrkBf2,eax
	add	@CBAddr,eax	;Relocate cache-entry buffer address.
	cmp	EFlag,0		;Are we running under a DOS emulator?
	jne s	I_XMgr		;If so, no need for any PCI-bus scan!
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	al,001h
	call	I_In1A
	cmp	edx,PCMsg2.dwd	;Is PCI BIOS V2.0C or newer?
	je s	I_ScnC		;Yes, scan for all IDE controllers.
	inc	EFlag		;Avoid disk DPTE data checks below.
	mov	dx,(NPMsg-@)	;Display "No V2.0C+ PCI" message.
	call	I_Msg
	jmp s	I_XMgr		;Go see if we have an XMS manager.
I_ScnC:	mov	al,LBABuf.lb	;Get next "interface bit" value.
	and	ax,00003h
	or	ax,PCISubC	;"Or" in subclass & current function.
	call	I_PCIC		;Test for specific PCI class/subclass.
	rol	LBABuf.lb,4	;Swap both "interface bit" values.
	mov	al,LBABuf.lb	;Load next "interface bit" value.
	or	al,al		;Both "interface" values tested?
	jns s	I_ScnC		;No, loop back and test 2nd one.
	add	PCISubC.lb,004h	;More PCI function codes to try?
	jnc s	I_ScnC		;Yes, loop back & try next function.
	test	al,001h		;Have we tested "Native PCI" ctlrs.?
	mov	LBABuf.lb,093h	;(Set "Native PCI" interface bits).
	jz s	I_ScnC		;No, loop back and test them, also.
I_XMgr:	cmp	SA20Rq.lb,090h	;Did we find an XMS manager?
	je	I_HDCh		;No, scan for any hard-disks to use.
	mov	dx,CStack.hw	;Get "reserved" XMS memory size.
	or	dx,dx		;Does user want any "reserved" XMS?
	jz s	I_XGet		;No, get driver's actual XMS memory.
	mov	ah,009h		;Get 15-MB or 63-MB "reserved" XMS
	call	I_XMS		;  memory, which we "release" below.
	jnz s	I_XErr		;If error, display message and exit!
	mov	CStack.lw,dx	;Save reserved-XMS "handle" number.
I_XGet:	mov	ah,009h		;Get XMS V2.0 "allocate" command.
	mov	cx,RqXMS.hw	;Do we need 64-MB+ of XMS memory?
	jcxz	I_XReq		;No, request our XMS memory now.
	mov	ah,089h		;Use XMS V3.0 "allocate" command.
I_XReq:	mov	edx,RqXMS	;Request all necessary XMS memory.
	call	I_XMS
	jz s	I_XFre		;If no errors, "free" reserved XMS.
I_XErr:	mov	eax," SMX"	;BAAAD News!  Get "XMS" msg. prefix.
	jmp	I_VErr		;Go display "XMS init error" & exit!
I_XFre:	mov	XMSHndl,dx	;Save our XMS "handle" numbers.
	mov	CLXDH,dx
	mov	@XMHdl,dx
	xchg	CStack.lw,dx
	or	dx,dx		;Any XMS reserved by /R15 or /R63?
	jz s	I_XLok		;No, go "lock" our XMS memory.
	mov	ah,00Ah		;"Free" our reserved XMS memory.
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
I_XLok:	mov	ah,00Ch		;"Lock" our driver's XMS memory.
	mov	dx,CStack.lw
	call	I_XMS
	jnz s	I_XErr		;If error, display message and exit!
	shl	edx,16		;Get unaligned 32-bit buffer address.
	or	dx,bx
	mov	esi,edx		;Initialize command-list XMS offset.
	mov	eax,edx		;Copy 32-bit address to EAX-reg.
	jz s	I_XBAd		;Any low-order XMS buffer "offset"?
	mov	ax,0FFFFh	;Yes, align address to an even 64K.
	inc	eax
I_XBAd:	mov	@XBAddr,eax	;Save aligned "main buffer" address.
	mov	@SXBAdr,eax
	add	XMSBase,eax	;Initialize cache-data base address.
	mov	cx,ax		;Get buffer "offset" in XMS memory.
	sub	cx,dx
	mov	@XMOffs.lw,cx	;Set offset in "SXMMov" subroutine.
	mov	edx,000010000h	;Put command-list after XMS buffer.
	jcxz	I_PRDA		;Is buffer already on a 64K boundary?
	or	edx,-32		;No, put command-list before buffer.
	dec	XMSBase.hw	;Decrement cache-data base by 64K.
I_PRDA:	add	eax,edx		;Set our 32-bit PRD address.
	mov	PRDAd,eax
	sub	eax,esi		;Set final command-list XMS offset.
	mov	CLXDA,eax
	cmp	DNam2.lb,'-'	;"Stand alone" driver requested?
	je	I_HDCh		;Yes, check BIOS hard-disk count.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
	jnz	I_A20E		;If error, display message & exit!
	mov	bx,(CCTbl-@)	;Point to "common" cache parameters.
I_SetX:	cmp	[bx+6].lw,0	;Is this parameter table "active"?
	je	I_SX5		;No, skip to next parameter table.
	mov	eax,XMSBase	;Set this cache's XMS data address.
	mov	[bx+12],eax
	mov	edx,[bx+16]	;Get needed cache XMS in 1K blocks.
	shl	edx,10		;Convert from 1K blocks to bytes.
	add	eax,edx		;Set this cache's data-table address.
	mov	[bx+16],eax
	xchg	eax,edx		;Save data-table address in EDX-reg.
	movzx	eax,[bx+6].lw	;Get binary-search table size.
	shl	eax,1
	mov	ecx,eax		;Save search-table size in ECX-reg.
	shl	eax,1		;Get offset to binary-search table.
	add	eax,ecx
	shl	eax,1
	add	eax,edx		;Set binary-search table address.
	mov	[bx+20],eax
	add	eax,ecx		;Set binary-search buffer address.
	mov	[bx+24],eax
	add	eax,ecx		;Update driver's XMS "base" address
	mov	XMSBase,eax	;  for use in setting-up next cache.
	mov	dx,[bx+6]	;Initialize search-table count.
	xor	bp,bp		;Initialize search-table index.
I_SX1:	mov	ax,bp		;Set next 300 search-table indexes
	mov	ecx,300		;  in our search-table index buffer.
	mov	si,(EDDBuff-@)
I_SX2:	mov	[si],ax
	inc	ax
	inc	si
	inc	si
	loop	I_SX2
	xor	esi,esi		;Set 32-bit move source address.
	mov	si,(EDDBuff-@)	;(Offset of our indexes buffer +
	add	esi,IOAdr	;  32-bit driver "base" address).
	movzx	edi,bp		;Set 32-bit move destination addr.
	shl	edi,1		;(2 * current "reset" index +
	add	edi,[bx+20]	;  binary-search table address).
	mov	bp,ax		;Update next cache-table index.
	mov	cx,300		;Get 300-word move length.
	cmp	cx,dx		;At least 300 words left to go?
	jbe s	I_SX3		;Yes, use full 300-word count.
	mov	cx,dx		;Use remaining word count.
I_SX3:	shl	cx,1		;Convert word count to byte count.
	push	bx		;Save cache parameter-table ptr.
	push	dx		;Save move count and cache index.
	push	bp
	call	MvData		;Move 300 indexes into search table.
	pop	bp		;Reload cache index and move count.
	pop	dx
	pop	bx		;Reload cache parameter-table ptr.
	jnc s	I_SX4		;If no XMS errors, check move count.
	call	I_A20D		;BAD News!  Do "A20 local-disable".
	jmp	I_XErr		;Go display "XMS init error" & exit!
I_SX4:	sub	dx,300		;More search-table indexes to set?
	ja s	I_SX1		;Yes, loop back and do next group.
I_SX5:	add	bx,PTSIZE	;Skip to next cache parameter table.
	cmp	bx,(CDTbl-@)	;More cache tables left to set up?
	jbe	I_SetX		;Yes, go check next parameter table.
	call	I_A20D		;Issue "A20 local-disable" request.
	xor	ax,ax		;Point ES-reg. to low memory.
	mov	es,ax
	mov	al,es:[HDWRFL]	;Get BIOS "hardware installed" flag.
	test	al,001h		;Any diskettes on this system?
	jz s	I_HDCh		;No, scan for available hard-disks.
	mov	al,011h		;Use diskette media-change bits
	and	al,es:[MCHDWFL]	;  for our number of diskettes.
I_FScn:	test	al,001h		;Can next unit signal media-changes?
	jz s	I_FMor		;No?  CANNOT use this old "clunker"!
	push	ax		;Save our diskette counter.
	mov	al,0C0h		;Get "diskette" device-type flags.
	call	I_CHSD		;Get and save diskette's CHS values.
	pop	ax		;Reload our diskette counter.
	inc	@LastU		;Bump our unit-table index.
I_FMor:	inc	HDUnit		;Bump BIOS unit number.
	inc	UnitNo.lb	;Bump error-message unit number.
	shr	al,4		;Another diskette to check?
	jnz s	I_FScn		;Yes, loop back and do next one.
I_HDCh:	cmp	BiosHD,0	;Any BIOS hard-disks to use?
	je	I_AnyD		;No, see if we found a diskette.
	mov	HDUnit,080h	;Set 1st BIOS hard-disk unit.
I_Next:	mov	ah,HDUnit	;Set unit no. in error message.
	mov	cx,2
	mov	si,(UnitNo-@)
	call	I_HexA
	mov	[si].lb,'h'
	mov	ah,041h		;Get EDD "extensions" for this disk.
	mov	bx,055AAh
	call	I_In13
	jc s	I_DRdy		;If none, check if disk is "ready".
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne s	I_DRdy		;No, check if this disk is "ready".
	test	cl,007h		;Any "EDD extensions" for this disk?
	jz s	I_DRdy		;No, check if this disk is "ready".
	push	cx		;Save disk "EDD extensions" flags.
	mov	si,(EDDBuff-@)	;Point to "EDD" input buffer.
	mov	[si].lw,30	;Set 30-byte buffer size.
	or	[si+26].dwd,-1	;Init NO "DPTE" data!  A bad BIOS may
				;  NOT post this dword for USB sticks
				;  etc.!  Many Thanks to Daniel Nice!
	mov	ah,048h		;Get this disk's "EDD" parameters.
	call	I_In13
	pop	cx		;Reload disk "EDD extensions" flags.
	jc s	I_Drdy		;If error, check if disk is "ready".
	test	[si+2].lb,004h	;Is this HARD disk flagged "removable"?
	jnz	I_IgnD		;If so, we have NO logic to SUPPORT IT!
	cmp	EFlag,0		;Are we running under a DOS emulator?
	jne s	I_DRdy		;Yes, check if this disk is "ready".
	cmp	[si].lw,30	;Did we get at least 30 bytes?
	jb s	I_DRdy		;No, check if this disk is "ready".
	test	cl,004h		;Does this disk provide "DPTE" data?
	jz s	I_DRdy		;No, check if this disk is "ready".
	cmp	[si+26].dwd,-1	;"Null" drive parameter-table pointer?
	je s	I_DRdy		;Yes, check if this disk is "ready".
I_DPTE:	les	si,[si+26]	;Get this disk's "DPTE" pointer.
	mov	bx,15		;Calculate "DPTE" checksum.
	xor	cx,cx
I_CkSm:	add	cl,es:[bx+si]
	dec	bx
	jns s	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
	mov	dx,(DPTEMsg-@)	;Display "DPTE data BAD" message.  May
	call	I_Msg		;  be from a VirtualBox "bug"!  So, if
	mov	dx,(UnitMsg-@)	;  this disk shows "ready" (below), we
	call	I_Msg		;  will "call the BIOS" to do its I-O!
I_DRdy:	mov	ah,010h		;Non-UltraDMA:  Check for disk "ready".
	call	I_In13
	jc	I_IgnD		;If "not ready", must IGNORE this disk!
	call	I_CHSB		;Get and save this disk's CHS values.
	jmp s	I_NoUJ		;Go bump non-UltraDMA disk count.
I_EDOK:	call	I_CHSB		;Get and save this disk's CHS values.
	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,es:[si+4]
	shr	bl,4
	mov	ax,es:[si]	;Get disk's IDE base address.
	mov	CtlrNo.lb,'0'	;Reset display to "Ctlr. 0".
	mov	si,(Ctl1Tbl-@)	;Point to IDE address table.
I_ITbl:	cmp	[si].lw,-1	;Is this IDE table active?
	je s	I_NoUJ		;No, go bump non-UltraDMA disk count.
	cmp	ax,[si+2]	;Is disk on this primary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust index for secondary channel.
	inc	bx
	cmp	ax,[si+4]	;Is disk on this secondary channel?
	je s	I_ChMS		;Yes, set disk channel and unit.
	inc	bx		;Adjust values for next controller.
	inc	bx
	add	si,CTLTSIZ
	inc	CtlrNo.lb
	cmp	si,(CtlTEnd-@)	;More IDE controllers to check?
	jb s	I_ITbl		;Yes, loop back and check next one.
I_NoUJ:	jmp s	I_NoUD		;Go bump non-UltraDMA disk count.
I_ChMS:	push	bx		;Save disk's caching unit number.
	mov	IdeDA,ax	;Save disk's base IDE address.
	add	ax,CDSEL	;Point to IDE device-select register.
	xchg	ax,dx
	mov	al,bl		;Get drive-select command byte.
	shl	al,4
	or	al,LBABITS
	out	dx,al		;Select master or slave disk.
	push	ax		;Save drive-select and caching unit.
	push	bx
	mov	dx,(CtlMsg-@)	;Display IDE controller number.
	call	I_Msg
	pop	ax		;Reload caching unit number.
	mov	dx,(PriMsg-@)	;Point to "Primary" channel name.
	test	al,002h		;Is this a primary-channel disk?
	jz s	I_PRNm		;Yes, display "Primary" channel.
	mov	dx,(SecMsg-@)	;Point to "Secondary" channel name.
I_PRNm:	call	I_Msg		;Display disk's IDE channel name.
	pop	ax		;Reload drive-select byte.
	mov	dx,(MstMsg-@)	;Point to "master" disk name.
	test	al,010h		;Is this disk the master?
	jz s	I_MSNm		;Yes, display "master" name.
	mov	dx,(SlvMsg-@)	;Point to "slave" disk name.
I_MSNm:	call	I_Msg		;Display disk's master/slave name.
	call	I_ValD		;Validate disk as an UltraDMA unit.
	pop	ax		;Reload caching unit number.
	jc s	I_UDEr		;If any errors, display message.
	mov	bx,@LastU	;Change this disk's device-type to
	mov	[bx+TypeF-@],al	;  "IDE" and include channel/unit.
	jmp s	I_More		;Go check for any more BIOS disks.
I_UDEr:	call	I_Msg		;NOT UltraDMA -- Display error msg.
I_NoUD:	cmp	DNam2.lb,'-'	;"Stand alone" driver requested?
	je s	I_IgnD		;Yes, ignore disk and scan for more.
	inc	NonUCt		;Bump number of non-UltraDMA disks.
	cmp	NonUCt,'9'	;Over 9 non-UltraDMA hard disks?
	jbe s	I_More		;No, check for more disks.
	mov	NonUCt.lw,"+9"	;Set 9+ non-UltraDMA count.
I_More:	inc	@LastU		;Bump our unit-table indexes.
	inc	@SLastU
I_IgnD:	inc	HDUnit		;Bump BIOS unit number.
	cmp	@LastU,MAXBIOS	;More entries in our units table?
	jae s	I_AnyN		;No, check for non-UltraDMA disks.
	dec	BiosHD		;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyN:	cmp	NonUCt.lb,'0'	;Do we have any non-UltraDMA disks?
	je s	I_AnyD		;No, check for any drives to use.
	mov	dx,(NonUMsg-@)	;Display "Non-UltraDMA disks" msg.
	call	I_Msg
I_AnyD:	cmp	DNam2.lb,'-'	;"Stand alone" driver requested?
	jne s	I_HMA		;No, see if we will load in the HMA.
	mov	dx,(NDMsg-@)	;Point to "Nothing to use" message.
	cmp	@SLastU,0	;Did we find any hard-disks to use?
	jz	I_Err		;No?  Display error message and exit!
I_HMA:	shr	HFlag,1		;Will we be loading in the HMA?
	jz	I_SADv		;No, set "stand alone" dvr. if needed.
	mov	ax,04A02h	;Request needed memory in the HMA.
	mov	bx,HMASize
	call	I_In2F
	push	es		;Get 32-bit HMA segment/offset addr.
	push	di
	pop	eax
	mov	@HDMain,eax	;Set caching logic entry address.
	mov	@SMainE,eax	;Set "stand alone" entry address.
	inc	eax		;Is our HMA address = -1 (no HMA)?
	jnz s	I_HMA1		;No, do all needed HMA adjustments.
	mov	eax," AMH"	;Get "HMA" error-message prefix.
I_HMAX:	jmp	I_VErr		;Go display error message & exit!
I_HMA1:	xor	ecx,ecx		;Get 32-bit HMA segment address.
	mov	cx,es
	shl	ecx,4
	movzx	eax,di		;Adjust real-mode GDT base addr.
	add	ax,(GDT-HDReq)
	add	eax,ecx
	mov	GDTPAdr,eax
	lea	ax,[di-(HDReq-@)] ;Get caching HMA logic offset.
	add	CCTblP,ax	  ;Adjust parameter-table pointers.
	add	U1TblP,ax
	add	U2TblP,ax
	add	CDTblP,ax
	add	@TypeF,ax	  ;Adjust disk "CHS tables" offsets.
	add	@CHSec,ax
	add	@CHSHd,ax
	add	@GDTP,ax	;Adjust "MvData" routine offsets.
	add	@MvDesc,ax
	add	@CtlTbl,ax	;Adjust disk controller-table ptr.
	lea	ax,[di-SLMLEN]	;Get "stand alone" HMA logic offset.
	add	@SCHSec,ax	;Adjust disk parameter-table ptrs.
	add	@SCHSHd,ax
	add	@SCtlUn,ax
	add	@SCtlTb,ax	;Adjust disk controller-table ptr.
	add	@XMSBlk,ax	;Adjust XMS block addresses.
	add	@CLXBlk,ax
	mov	dx,(SLMLEN-SABASE) ;For the "stand alone" driver,
	call	I_SetS		   ;  move down low-memory logic.
	mov	ah,005h		;Issue "A20 local-enable" request.
	call	I_XMS
I_A20E:	mov	eax," 02A"	;Get "A20" error-message prefix.
	jnz s	I_HMAX		;If any "A20" error, bail out QUICK!
	mov     cx,HMASize      ;Move required logic up to the HMA.
	mov	si,MvUpAdr
	les	di,@SMainE
	rep	movsb
	call	I_A20D		;Issue "A20 local-disable" request.
	jmp s	I_Done		;Go post "Init" packet results.
I_SADv:	mov	dx,(SLMLEN-SABASE+SHMALEN) ;If "stand alone" driver,
	call	I_SetS			   ;  move all logic down.
I_Done:	xor	ax,ax		;Done!  Load & reset driver length.
	xchg	ax,VDSLn.lw
	les	bx,InitPkt	;Set results in DOS "Init" packet.
	mov	es:[bx].RPLen,ax
	mov	es:[bx].RPStat,RPDON
	mov	si,EntAddr	;Set stack address in "Entry" logic.
	mov	[si+(@Stack-Entry)],ax
	xor	bx,bx		;Clear driver stack (helps debug).
	xchg	ax,bx
	mov	cx,StakSiz
I_ClrS:	dec	bx
	mov	[bx],al
	loop	I_ClrS
	mov	ax,02513h	;Set "Entry" address in DX-reg. and
	mov	dx,si		;  "hook" this driver into Int 13h. 
	int	021h
I_End:	popad			;Reload all 32-bit CPU registers.
I_Bye:	pop	dx		;Reload 16-bit registers we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; Init subroutine to set up for our "stand alone" driver.
;
I_SetB:	mov	VDSLn.lw,SLMEnd	 ;Set all "stand alone" parameters.
	mov	HMASize.dwd,((SABASE*65536)+SHMALEN)
	mov	StakSiz.dwd,(((SHDREQ-@)*65536)+STACKSA)
	mov	DNam2.dwd,"$AS-" ;Set "stand alone" driver name.
	mov	EFlag,0		 ;Reset "DOS emulator active" flag.
	ret			 ;Exit.
;
; Init subroutine to finalize our "stand alone" driver and move
;   it down to low-memory for execution.   At entry the DX-reg.
;   has the number of bytes to be moved down.
;
I_SetS:	cmp	DNam2.lb,'-'	;"Stand alone" driver requested?
	jne s	I_SSAX		;No, just exit below.
	cld			;Ensure FORWARD "string" commands!
	push	ds		;Set ES-reg. same as DS-reg.
	pop	es
	mov	cx,DDLEN	;Move "DoDMA" logic up to "SDoDMA".
	mov	si,(DDBase-@)
	mov	di,(SDDBase-@)
	rep	movsb
	mov	cx,(CTLTSIZ*10)	;Copy controller data from caching
	mov	si,(Ctl1Tbl-@)	;  driver to "stand alone" driver.
	mov	di,(SCtlTbl-@)
	rep	movsb
	mov	cl,(MAXBIOS*3)	;Copy disk parameters from caching
	mov	si,(TypeF-@)	;  driver to "stand alone" driver.
	rep	movsb
	mov	cx,dx		;Move all necessary "stand alone"
	mov	si,(SEntry-@)	;  driver logic down to low memory.
	mov	di,SABase
	rep	movsb
I_SSAX:	ret			;Exit.
;
; Init subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD:	xor	bx,bx		;Zero BX-reg. for timeout checks.
	mov	bp,[bx+IdeDA-@]	;Get drive's IDE address in BP-reg.
	lea	dx,[bp+CCMD]	;Point to IDE command reg.
	mov	al,0ECh		;Issue "Identify ATA Device" command.
	out	dx,al
	mov	ah,CMDTO	;Use 500-msec command timeout.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	add	ah,es:[si]	;Set timeout limit in AH-reg.
I_VDTO:	cmp	ah,es:[si]	;Has our I-O timed out?
	je s	I_DErr		;Yes?  Exit & display "Identify" msg.
	lea	dx,[bp+CSTAT]	;Point to IDE status reg.
	in	al,dx		;Read IDE primary status.
	test	al,BSY+DRQ	;Controller still busy, or no DRQ?
	jle s	I_VDTO		;If either, loop back & test again.
	test	al,ERR		;Did command cause any errors?
	jz s	I_PIO		;No, read I.D. data using PIO.
I_DErr:	mov	dx,(IEMsg-@)	;Point to "Identify error" msg.
I_SErr:	stc			;Set carry flag (error) & exit.
	ret
I_PIO:	mov	dx,bp		;Point to IDE data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	shl	ax,1		;Save "ATA/ATAPI" flag in carry bit.
	mov	cx,27		;Skip I.D. bytes 2-53 and
I_Skp1:	in	ax,dx		;  read I.D. bytes 54-55.
	loop	I_Skp1
	push	cs		;Point to disk-name message.
	pop	es
	mov	di,(DName-@)
	mov	cl,26		;Read & swap disk name into message.
I_RdNm:	xchg	ah,al		;(I.D. bytes 54-93.  Bytes 94-105 are
	stosw			;  also read but are ignored.   Bytes
	in	ax,dx		;  106-107 are left in the AX-reg.).
	loop	I_RdNm
	xchg	ax,bx		;Save "UltraDMA valid" flag in BL-reg.
	mov	cl,35		;Skip I.D. bytes 108-175 &
I_Skp2:	in	ax,dx		;  read I.D. bytes 176-177.
	loop	I_Skp2
	mov	bh,ah		;Save "UltraDMA mode" flags in BH-reg.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3:	in	ax,dx
	loop	I_Skp3
	mov	dx,(UEMsg-@)	;Point to "is not UltraDMA" message.
	rcr	bl,1		;Shift "ATA/ATAPI" flag into BL-reg.
	and	bl,082h		;ATAPI disk, or UltraDMA bits invalid?
	jle s	I_SErr		;Yes?  Exit & display error message.
	mov	di,(Modes-@)	;Point to UltraDMA mode table.
	or	bh,bh		;Will disk do UltraDMA mode 0?
	jz s	I_SErr		;No?  Exit & display message!
I_NxtM:	cmp	bh,bl		;Will disk do next UltraDMA mode?
	jb s	I_GotM		;No, use current mode.
	inc	di		;Point to next mode table value.
	inc	di
	shl	bl,1		;More UltraDMA modes to check?
	jnz s	I_NxtM		;Yes, loop back.
I_GotM:	mov	si,(DNEnd-@)	;Point to end of disk name.
I_NxtN:	cmp	si,(DName-@)	;Are we at the disk-name start?
	je s	I_Name		;Yes, disk name is all spaces!
	dec	si		;Decrement disk name pointer.
	cmp	[si].lb,' '	;Is this name byte a space?
	je s	I_NxtN		;No, continue scan for non-space.
	inc	si		;Skip non-space character.
	mov	[si].lw," ,"	;End disk name with comma & space.
	inc	si
	inc	si
I_Name:	mov	[si].dwd,"-ATA"	;Set "ATA-" after drive name.
	add	si,4
	mov	ax,[di]		;Set UltraDMA "mode" in message.
	mov	cl,00Fh
	and	cl,al
	call	I_HexA
	mov	[si].dwd,0240A0D2Eh ;Set message terminators.
	mov	dx,(DNMsg-@)	    ;Display mfr. name/"mode" & exit.
	jmp s	I_Msg
;
; Init subroutine to set BIOS CHS data for a hard-disk or diskette.
;
I_CHSB:	mov	al,080h		;Get "BIOS disk" device-type.
I_CHSD:	push	ax		;Save unit's device-type flag.
	mov	ah,008h		;Get BIOS CHS data for this unit.
	call	I_In13
	jc s	I_CHSE		;If BIOS error, zero CHS sectors.
	and	cl,03Fh		;Get sectors/head value (low 6 bits).
	jz s	I_CHSE		;If zero, ERROR!  Zero CHS sectors.
	inc	dh		;Get heads/cylinder (BIOS value + 1).
	jnz s	I_CHST		;If non-zero, save data in our table.
I_CHSE:	xor	cl,cl		;Error!  Zero unit's CHS sectors.
I_CHST:	mov	di,@LastU	;Point to "active units" table.
	mov	al,HDUnit	;Set BIOS unit number in our table.
	mov	[di+Units-@],al
	mov	[di+CHSec-@],cl ;Set unit's CHS data in our table.
	mov	[di+CHSHd-@],dh
	pop	ax		;Reload and set device-type flag.
	mov	[di+TypeF-@],al
	mov	dx,(CHSMsg-@)	;Point to "CHS data BAD" message.
	or	cl,cl		;Valid CHS values for this unit?
	jz s	I_Msg		;No?  Display error msg. and exit!
	ret			;All is well -- exit.
;
; Init subroutines to issue "external" calls.
;
I_A20D:	mov	ah,006h		;"A20 local-disable" -- get XMS code.
I_XMS:	call	@XMgr3.dwd	;XMS -- issue desired request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_VDS:	mov	di,(VDSLn-@)	;VDS -- Point to parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In13:	mov	dl,HDUnit	;BIOS data -- set BIOS unit in DL-reg.
	int	013h		;Issue BIOS data interrupt.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_PCID:	push	bx		;Save PCI bus/device/function codes.
	push	si		;Save IDE address-table pointer.
	mov	al,00Ah		;Set "PCI doubleword" request code.
	call	I_In1A		;Get desired 32-bit word from PCI.
	pop	si		;Reload IDE address-table pointer.
	pop	bx		;Reload PCI bus/device/function.
	ret			;Exit.
I_In1A:	mov	ah,0B1h		;Issue PCI BIOS interrupt.
	int	01Ah
	jmp s	I_IntX		;Restore driver settings, then exit.
I_Msg:	push	bx		;Message -- save our BX-register.
	mov	ah,009h		;Issue DOS "display string" request.
	int	021h
	pop	bx		;Reload our BX-register.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In21:	int	021h		;General DOS request -- issue Int 21h.
	jmp s	I_IntX		;Restore driver settings, then exit.
I_In2F:	int	02Fh		;"Multiplex" -- issue XMS/HMA request.
I_IntX:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
I_Ret:	ret			;Exit.
CODE	ends
	end
