#include <string.h>
#include <stdlib.h>

#ifdef AMIGA
#include "AmigaInt.h"
#else

#ifdef WIN32
#include "Win32Int.h"
#else
#include <stdint.h>
#include <unistd.h>
#endif

#endif

#include "FixP.h"
#include "Vec.h"
#include "Enums.h"
#include "CActor.h"
#include "MapWithCharKey.h"
#include "Common.h"
#include "Vec.h"
#include "LoadBitmap.h"
#include "Engine.h"
#include "MapWithCharKey.h"
#include "CTile3DProperties.h"
#include "CRenderer.h"
#include "VisibilityStrategy.h"

#define kMinZCull 0

void projectAllVertices(const uint8_t count) {
	FixP_t halfWidth = intToFix(HALF_XRES);
	FixP_t halfHeight = intToFix(HALF_YRES);
	FixP_t zero = 0;
	FixP_t one = intToFix(1);
	FixP_t two = intToFix(2);
	FixP_t bias = Div(one, intToFix(128));
	FixP_t projected;
	FixP_t oneOver;
	int c;
	for (c = 0; c < count; ++c) {
		struct Projection *vertex = &projectionVertices[c];

		FixP_t z = (vertex->first.mZ);

		if (z < one) {
			z = one;
		}

		projected = Div(z, two);

		if (projected == zero) {
			projected += bias;
		}

		oneOver = Div(halfHeight, projected);

		vertex->second.mX = (halfWidth + Mul(vertex->first.mX, oneOver));
		vertex->second.mY =
				(halfHeight
				 - Mul(vertex->first.mY + playerHeight + walkingBias, oneOver));
	}
}

void drawBillboardAt(const struct Vec3 center,
					 const uint8_t * __restrict__ texture,
					 const FixP_t scale,
					 const int size) {
	FixP_t one = intToFix(1);
	FixP_t zero = 0;
	FixP_t minusOne = intToFix(-1);
	FixP_t minusScale = (-scale);
	FixP_t halfScale = Div(scale, intToFix(2));
	struct Vec3 scaledCenter;
	struct Vec2 ulz0;
	struct Vec2 lrz0;
	int z = fixToInt(center.mZ);

	if (center.mZ <= kMinZCull) {
		return;
	}

	initVec3(&scaledCenter, center.mX, (center.mY), center.mZ);
	initVec3(&projectionVertices[0].first, scaledCenter.mX, scaledCenter.mY,
			 scaledCenter.mZ);
	initVec3(&projectionVertices[1].first, scaledCenter.mX, scaledCenter.mY,
			 scaledCenter.mZ);
	addToVec3(&projectionVertices[0].first, minusOne, scale, zero);
	addToVec3(&projectionVertices[1].first, one, minusScale, zero);

	projectAllVertices(2);

	ulz0 = projectionVertices[0].second;
	lrz0 = projectionVertices[1].second;

	if (z >= distanceForDarkness && useDither) {
		drawMask(ulz0.mX, ulz0.mY, lrz0.mX, lrz0.mY);
	} else {
		drawFrontWall(ulz0.mX, ulz0.mY, lrz0.mX, lrz0.mY, texture,
					  (halfScale * 2), z, TRUE, size);
	}
}

void drawColumnAt(const struct Vec3 center,
				  const FixP_t scale,
				  const struct Texture *  __restrict__  texture,
				  const uint8_t mask,
				  const int enableAlpha,
				  const int repeatTexture) {

	FixP_t one = intToFix(1);
	FixP_t minusOne = intToFix(-1);
	const FixP_t halfScale = scale;
	const FixP_t minusHalfScale = (-scale);
	const FixP_t textureScale = (repeatTexture ? halfScale : one);
	struct Vec3 scaledCenter;
	struct Vec2 p0;
	struct Vec2 p1;
	struct Vec2 p2;
	struct Vec2 p3;
	int z = fixToInt(center.mZ);

	if (center.mZ <= kMinZCull) {
		return;
	}

	initVec3(&scaledCenter, center.mX, center.mY, center.mZ);

	/*
		 2|\             /|
		  | \  center   / |
		  |  \    *    /  |
		  |   \0__|___/   |3
		  |   |   |  |   /
		   \  |   X  |  /
			\ |      | /
			 \|_____1|/

  */
	memcpy (&projectionVertices[0].first, &scaledCenter, sizeof(struct Vec3));
	memcpy (&projectionVertices[1].first, &scaledCenter, sizeof(struct Vec3));
	memcpy (&projectionVertices[2].first, &scaledCenter, sizeof(struct Vec3));
	memcpy (&projectionVertices[3].first, &scaledCenter, sizeof(struct Vec3));

	addToVec3(&projectionVertices[0].first, minusOne, halfScale, minusOne);
	addToVec3(&projectionVertices[1].first, one, minusHalfScale, minusOne);
	addToVec3(&projectionVertices[2].first, minusOne, halfScale, one);
	addToVec3(&projectionVertices[3].first, one, minusHalfScale, one);

	projectAllVertices(4);

	p0 = projectionVertices[0].second;
	p1 = projectionVertices[1].second;
	p2 = projectionVertices[2].second;
	p3 = projectionVertices[3].second;

	if (enableAlpha && (mask & MASK_FRONT)) {
		if (z >= distanceForDarkness && useDither) {
			drawMask(p2.mX, p2.mY, p3.mX, p3.mY);
		} else {
				drawFrontWall(p2.mX, p2.mY, p3.mX, p3.mY, texture->rotations[0],
							  (textureScale), z, enableAlpha, NATIVE_TEXTURE_SIZE);
		}
	}

	if ((mask & MASK_RIGHT) && fixToInt(center.mX) > 0) {
		if (z >= distanceForDarkness && useDither) {
			maskWall(p2.mX, p0.mX, p2.mY, p3.mY, p0.mY, p1.mY);
		} else {
			drawWall(p2.mX, p0.mX, p2.mY, p3.mY, p0.mY, p1.mY, texture->rowMajor,
						 (textureScale), z);
		}
	}

	if ((mask & MASK_LEFT) && fixToInt(center.mX) < 0) {
		if (z >= distanceForDarkness && useDither) {
			maskWall(p1.mX, p3.mX, p0.mY, p1.mY, p2.mY, p3.mY);
		} else {
			drawWall(p1.mX, p3.mX, p0.mY, p1.mY, p2.mY, p3.mY, texture->rowMajor,
						 (textureScale), z);
		}
	}

	if ((mask & MASK_BEHIND) || (mask & MASK_FRONT)) {
		if (z >= distanceForDarkness && useDither) {
			drawMask(p0.mX, p0.mY, p1.mX, p1.mY);
		} else {
				drawFrontWall(p0.mX, p0.mY, p1.mX, p1.mY, texture->rotations[0],
							  (textureScale), z, enableAlpha, NATIVE_TEXTURE_SIZE);
		}
	}
}

void drawFloorAt(const struct Vec3 center,
				 const struct Texture * __restrict__ texture, uint8_t cameraDirection) {

	FixP_t one = intToFix(1);
	FixP_t zero = 0;
	FixP_t minusOne = intToFix(-1);
	FixP_t threshold = 0;
	struct Vec2 llz0;
	struct Vec2 lrz0;
	struct Vec2 llz1;
	struct Vec2 lrz1;

	if (center.mZ <= kMinZCull) {
		return;
	}

	memcpy (&projectionVertices[0].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[1].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[2].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[3].first, &center, sizeof(struct Vec3));

	addToVec3(&projectionVertices[0].first, minusOne, zero, minusOne);
	addToVec3(&projectionVertices[1].first, one, zero, minusOne);
	addToVec3(&projectionVertices[2].first, minusOne, zero, one);
	addToVec3(&projectionVertices[3].first, one, zero, one);

	projectAllVertices(4);

	llz0 = projectionVertices[0].second;
	lrz0 = projectionVertices[1].second;
	llz1 = projectionVertices[2].second;
	lrz1 = projectionVertices[3].second;

	if (center.mY <= threshold) {

		int z = fixToInt(center.mZ);

		if (z >= distanceForDarkness && useDither) {
			maskFloor(llz1.mY, lrz0.mY, llz1.mX, lrz1.mX, llz0.mX, lrz0.mX
#ifdef FLAT_FLOOR_CEILING
					, 0
#endif
					);
		} else {
#ifndef FLAT_FLOOR_CEILING
			drawFloor(llz1.mY, lrz0.mY, llz1.mX, lrz1.mX, llz0.mX, lrz0.mX, z,
						  texture->rotations[cameraDirection]);
#else
			maskFloor(llz1.mY, lrz0.mY, llz1.mX, lrz1.mX, llz0.mX, lrz0.mX, 254);
#endif
		}
	}
}

void drawCeilingAt(const struct Vec3 center,
				   const struct Texture *  __restrict__  texture, uint8_t cameraDirection) {

	FixP_t one = intToFix(1);
	FixP_t minusOne = intToFix(-1);
	FixP_t zero = 0;
	FixP_t threshold = zero;
	struct Vec2 llz0;
	struct Vec2 lrz0;
	struct Vec2 llz1;
	struct Vec2 lrz1;

	if (center.mZ <= kMinZCull) {
		return;
	}

	memcpy (&projectionVertices[0].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[1].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[2].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[3].first, &center, sizeof(struct Vec3));

	addToVec3(&projectionVertices[0].first, minusOne, zero, minusOne);
	addToVec3(&projectionVertices[1].first, one, zero, minusOne);
	addToVec3(&projectionVertices[2].first, minusOne, zero, one);
	addToVec3(&projectionVertices[3].first, one, zero, one);

	projectAllVertices(4);

	llz0 = projectionVertices[0].second;
	lrz0 = projectionVertices[1].second;
	llz1 = projectionVertices[2].second;
	lrz1 = projectionVertices[3].second;

	if (center.mY >= threshold) {

		int z = fixToInt(center.mZ);

		if (z >= distanceForDarkness && useDither) {
			maskFloor(llz1.mY, lrz0.mY, llz1.mX, lrz1.mX, llz0.mX, lrz0.mX
#ifdef FLAT_FLOOR_CEILING
					, 0
#endif
					);
		} else {
#ifndef FLAT_FLOOR_CEILING
			drawFloor(llz1.mY, lrz0.mY, llz1.mX, lrz1.mX, llz0.mX, lrz0.mX, z,
						  texture->rotations[cameraDirection]);
#else
			maskFloor(llz1.mY, lrz0.mY, llz1.mX, lrz1.mX, llz0.mX, lrz0.mX, 254);
#endif
		}
	}
}

void drawLeftNear(const struct Vec3 center,
				  const FixP_t scale,
				  const uint8_t *  __restrict__ texture,
				  const uint8_t mask,
				  const int repeatTexture) {

	FixP_t one = intToFix(1);
	FixP_t minusOne = intToFix(-1);
	FixP_t halfScale = scale;
	FixP_t minusHalfScale = (-scale);
	const FixP_t textureScale = (repeatTexture ? halfScale : one);
	FixP_t depth = one;
	FixP_t minusDepth = minusOne;
	struct Vec2 ulz0;
	struct Vec2 urz0;
	struct Vec2 llz0;
	struct Vec2 lrz0;
	int z = fixToInt(center.mZ);

	if (center.mZ <= kMinZCull) {
		return;
	}

	if (mask & MASK_BEHIND) {

		memcpy (&projectionVertices[0].first, &center, sizeof(struct Vec3));
		memcpy (&projectionVertices[1].first, &center, sizeof(struct Vec3));

		addToVec3(&projectionVertices[0].first, minusOne, minusHalfScale, minusOne);
		addToVec3(&projectionVertices[1].first, one, halfScale, minusOne);

		projectAllVertices(2);

		drawMask(projectionVertices[0].second.mX, projectionVertices[0].second.mY,
				 projectionVertices[1].second.mX, projectionVertices[1].second.mY);

		return;
	}

	if (cameraDirection == kWest || cameraDirection == kEast) {
		depth = minusOne;
		minusDepth = one;
	}

	memcpy (&projectionVertices[0].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[1].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[2].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[3].first, &center, sizeof(struct Vec3));

	addToVec3(&projectionVertices[0].first, minusOne, halfScale, minusDepth);
	addToVec3(&projectionVertices[1].first, one, halfScale, depth);
	addToVec3(&projectionVertices[2].first, minusOne, minusHalfScale, minusDepth);
	addToVec3(&projectionVertices[3].first, one, minusHalfScale, depth);

	projectAllVertices(4);

	ulz0 = projectionVertices[0].second;
	urz0 = projectionVertices[1].second;
	llz0 = projectionVertices[2].second;
	lrz0 = projectionVertices[3].second;

	if (z >= distanceForDarkness && useDither) {
		maskWall(ulz0.mX, urz0.mX, ulz0.mY, llz0.mY, urz0.mY, lrz0.mY);
	} else {
		drawWall(ulz0.mX, urz0.mX, ulz0.mY, llz0.mY, urz0.mY, lrz0.mY, texture,
					 textureScale, z);
	}
}

void drawRightNear(const struct Vec3 center,
				   const FixP_t scale,
				   const uint8_t *  __restrict__  texture,
				   const uint8_t mask,
				   const int repeatTexture) {

	FixP_t one = intToFix(1);
	FixP_t minusOne = intToFix(-1);
	FixP_t halfScale = scale;
	FixP_t minusHalfScale = (-scale);
	const FixP_t textureScale = (repeatTexture ? halfScale : one);
	FixP_t depth = intToFix(1);
	FixP_t minusDepth = intToFix(-1);
	int z = fixToInt(center.mZ);
	struct Vec2 ulz0;
	struct Vec2 urz0;
	struct Vec2 llz0;
	struct Vec2 lrz0;

	if (center.mZ <= kMinZCull) {
		return;
	}

	if (mask & MASK_BEHIND) {

		memcpy (&projectionVertices[0].first, &center, sizeof(struct Vec3));
		memcpy (&projectionVertices[1].first, &center, sizeof(struct Vec3));

		addToVec3(&projectionVertices[0].first, minusOne, minusHalfScale, minusOne);
		addToVec3(&projectionVertices[1].first, one, halfScale, minusOne);

		projectAllVertices(2);

		drawMask(projectionVertices[0].second.mX, projectionVertices[0].second.mY,
				 projectionVertices[1].second.mX, projectionVertices[1].second.mY);

		return;
	}

	if (cameraDirection == kWest || cameraDirection == kEast) {

		depth = minusOne;
		minusDepth = one;
	}

	memcpy (&projectionVertices[0].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[1].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[2].first, &center, sizeof(struct Vec3));
	memcpy (&projectionVertices[3].first, &center, sizeof(struct Vec3));

	addToVec3(&projectionVertices[0].first, minusOne, halfScale, depth);
	addToVec3(&projectionVertices[1].first, one, halfScale, minusDepth);
	addToVec3(&projectionVertices[2].first, minusOne, minusHalfScale, depth);
	addToVec3(&projectionVertices[3].first, one, minusHalfScale, minusDepth);

	projectAllVertices(4);

	ulz0 = projectionVertices[0].second;
	urz0 = projectionVertices[1].second;
	llz0 = projectionVertices[2].second;
	lrz0 = projectionVertices[3].second;

	if (z >= distanceForDarkness && useDither) {
		maskWall(ulz0.mX, urz0.mX, ulz0.mY, llz0.mY, urz0.mY, lrz0.mY);
	} else {
		drawWall(ulz0.mX, urz0.mX, ulz0.mY, llz0.mY, urz0.mY, lrz0.mY, texture,
					 textureScale, z);
	}
}
