/*
 * This file is part of the ipcfg tool
 * http://picotcp4dos.sourceforge.net
 *
 * Copyright (C) 2015 Mateusz Viste
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <stdint.h>  /* uint32_t */
#include <stdio.h>
#include <stdlib.h>  /* atoi() */
#include <string.h>  /* strcmp() */

#include "picoip.h"
#include "..\picotcp.h" /* pico_* */

#include "parsecli.h" /* include self for control */

/* analyzes a string, returns:
 * -1 for 'del'
 * +1 for 'add'
 *  0 for anything else */
static int add_or_del(char *s) {
  if (strcmp(s, "add") == 0) return(1);
  if (strcmp(s, "del") == 0) return(-1);
  return(0);
}


/* validate an ip address string, returns the iplen on success (4 or 16), -1 on error */
static int validate_ip(char *s, unsigned char *res) {
  uint32_t ip4;
  /* test for IPv4 */
  if (pico_string_to_ipv4(s, &ip4) == 0) {
    picoip2ip(ip4, res);
    return(4);
  }
  /* test for IPv6 */
  /* if (pico_string_to_ipv6(s, res) == 0) return(16); */ /* TODO uncomment when IPv6 support works */
  return(-1);
}


/* validate an ip/mask string, returns the iplen on success (4 or 16), -1 on error */
static int validate_ipmask(char *s, unsigned char *res, int *masklen) {
  int i, slashpos = 0;
  char *maskstart = NULL;
  /* separate the IP address from mask */
  for (i = 0; s[i] != 0; i++) {
    if (s[i] == '/') {
      slashpos = i;
      s[i] = 0;
      maskstart = s + slashpos + 1;
      break;
    }
  }
  /* if no slash found, abort */
  if (slashpos == 0) return(-1);
  /* validate the mask part */
  if ((atoi(maskstart) == 0) && ((maskstart[0] != '0') || (maskstart[1] != 0))) {
    s[slashpos] = '/';
    return(-1);
  }
  *masklen = atoi(maskstart);
  /* validate the ip part */
  i = validate_ip(s, res);
  s[slashpos] = '/'; /* restore the / sign I replaced by a 0 before */
  /* validate that the masklen is consistent with the ip family */
  if (i == 4) {
    if ((*masklen < 0) || (*masklen > 32)) return(-1);
  } else if (i == 16) {
    if ((*masklen < 0) || (*masklen > 128)) return(-1);
  }
  return(i);
}


/* validate a network/mask string, returns the iplen on success (4 or 16), -1 on error */
static int validate_netwmask(char *s, unsigned char *res, int *masklen) {
  int iplen;
  /* proceed as with normal ip/mask, just check out the network bits later */
  iplen = validate_ipmask(s, res, masklen);
  /* check that the netmask is consistent with network IP */
  if (iplen == 4) {
    /* TODO validate network bits */
  } else if (iplen == 6) {
    /* TODO validate network bits */
  }
  return(iplen);
}


/* analyzes command line arguments, and selects the requested action, fills
   the arguments list for the given action and returns the number of relevent
   arguments passed in *arg (0 or 1) */
enum command_t parsecli(unsigned char *arg1, unsigned char *arg2, int *argint, char **errmsg, int argc, char **argv) {
  char *INV_SUBCMD_ADD_DEL = "invalid subcommand, must be either 'add' or 'del'";
  char *INV_NUM_ARGS = "invalid number of arguments";
  char *INV_IPADDR = "invalid IP address";
  char *INV_IPMASK = "invalid IP/masklen pair";
  char *INV_INT = "invalid interrupt, must be in range 1..0xff";
  char *INV_INCOMPATGW = "gateway is either malformed or incompatible with destination prefix";
  int action;

  /* assume a 'help' action by default with no arg */
  *errmsg = NULL;

  /* if no argument -> help screen */
  if ((argc < 2) || (strcmp(argv[1], "/?") == 0) || (strcmp(argv[1], "/h") == 0)) {
    return(CMD_HELP);
  }

  /* */
  if (strcmp(argv[1], "int") == 0) {
    long tmp;
    /* if on arg -> show, otherwise if 1 arg -> set */
    if (argc == 2) return(CMD_INT_SHOW);
    /* must be 3 args */
    if (argc != 3) {
      *errmsg = INV_NUM_ARGS;
      return(CMD_HELP);
    }
    /* third arg must be a hex integer in range 1..0xff */
    tmp = strtol(argv[2], NULL, 16);
    if ((tmp < 1) || (tmp > 0xff)) {
      *errmsg = INV_INT;
      return(CMD_HELP);
    }
    *argint = tmp;
    return(CMD_INT_SET);
  } else if (strcmp(argv[1], "ip") == 0) {
    if (argc == 2) return(CMD_IP_SHOW);
    /* else I need exactly 2 additional arguments (add|del ip/masklen) */
    if (argc != 4) {
      *errmsg = INV_NUM_ARGS;
      return(CMD_HELP);
    }
    /* the action must be either add or del */
    action = add_or_del(argv[2]);
    if (action == 0) {
      *errmsg = INV_SUBCMD_ADD_DEL;
      return(CMD_HELP);
    }
    /* the final argument must be a valid IP/masklen (v4 or v6) */
    if (validate_ipmask(argv[3], arg1, argint) < 0) {
      *errmsg = INV_IPMASK;
      return(CMD_HELP);
    }
    /* all good */
    if (action < 0) return(CMD_IP_DEL);
    return(CMD_IP_ADD);
  }

  if (strcmp(argv[1], "route") == 0) {
    int prefixfamily;
    if (argc == 2) return(CMD_ROUTE_SHOW);
    /* else I need 2 or 3 additional arguments (add|del prefix/mask [gw]) */
    if ((argc != 4) && (argc != 5)) {
      *errmsg = INV_NUM_ARGS;
      return(CMD_HELP);
    }
    /* the action must be either add or del */
    action = add_or_del(argv[2]);
    if (action == 0) {
      *errmsg = INV_SUBCMD_ADD_DEL;
      return(CMD_HELP);
    }
    /* the next argument must be a valid prefix/mask (v4 or v6) */
    prefixfamily = validate_netwmask(argv[3], arg1, argint);
    if (prefixfamily < 0) {
      *errmsg = INV_IPMASK;
      return(CMD_HELP);
    }

    if (action > 0) { /* add */
      if (argc != 5) {
        *errmsg = INV_NUM_ARGS;
        return(CMD_HELP);
      }
      /* the last argument must be a valid gw ip (v4 or v6, same family as prefix) */
      if (validate_ip(argv[4], arg2) != prefixfamily) {
        *errmsg = INV_INCOMPATGW;
        return(CMD_HELP);
      }
      return(CMD_ROUTE_ADD);
    } else { /* del */
      if (argc != 4) {
        *errmsg = INV_NUM_ARGS;
        return(CMD_HELP);
      }
      return(CMD_ROUTE_DEL);
    }
  }

  if (strcmp(argv[1], "dhcp") == 0) {
    if (argc != 2) {
      *errmsg = INV_NUM_ARGS;
      return(CMD_HELP);
    }
    return(CMD_DHCP);
  }

  if (strcmp(argv[1], "dns") == 0) {
    if (argc == 2) return(CMD_DNS_SHOW);
    /* else I need exactly 2 additional arguments (add|del dnsserver) */
    if (argc != 4) {
      *errmsg = INV_NUM_ARGS;
      return(CMD_HELP);
    }
    /* the action must be either add or del */
    action = add_or_del(argv[2]);
    if (action == 0) {
      *errmsg = INV_SUBCMD_ADD_DEL;
      return(CMD_HELP);
    }
    /* the final argument must be a valid IP (v4 or v6) */
    if (validate_ip(argv[3], arg1) < 0) {
      *errmsg = INV_IPADDR;
      return(CMD_HELP);
    }
    /* all good */
    if (action < 0) return(CMD_DNS_DEL);
    return(CMD_DNS_ADD);
  }

  /* else it's an unknown command */
  *errmsg = "invalid command";
  return(CMD_HELP);
}
