/*
 * picoTCP declaration headers for the DOS port of picoTCP - extracted from
 * the picoTCP distribution and integrated into a single file for SFN compat.
 *
 * http://picotcp4dos.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 */

#include <stdint.h>

#ifndef PICOTCP_H_SENTINEL
#define PICOTCP_H_SENTINEL

#define MAX_DEVICE_NAME 16

#define PACKED_STRUCT_DEF   _Packed struct
#define PEDANTIC_STRUCT_DEF struct
#define PACKED_UNION_DEF    _Packed union
#define WEAK

extern int pico_err; /* provided by picotcp, similar to errno */



/****** CONSTANTS ***********************************************************/

#define PICO_SIZE_IP4    4
#define PICO_SIZE_IP6   16
#define PICO_SIZE_ETH    6
#define PICO_SIZE_TRANS  8
#define PICO_IP4_ANY (0x00000000L)
#define PICO_IP4_BCAST (0xffffffffL)

#define PICO_PROTO_IPV4   0
#define PICO_PROTO_ICMP4  1
#define PICO_PROTO_IGMP  2
#define PICO_PROTO_TCP    6
#define PICO_PROTO_UDP    17
#define PICO_PROTO_IPV6   41
#define PICO_PROTO_ICMP6  58


/****** LOW-LEVEL PROTOCOLS AND STRUCTURES **********************************/

struct pico_device {
  char name[MAX_DEVICE_NAME];
  uint32_t hash;
  uint32_t overhead;
  uint32_t mtu;
  struct pico_ethdev *eth; /* Null if non-ethernet */
  struct pico_queue *q_in;
  struct pico_queue *q_out;
  int (*link_state)(struct pico_device *self);
  int (*send)(struct pico_device *self, void *buf, int len); /* Send function. Return 0 if busy */
  int (*poll)(struct pico_device *self, int loop_score);
  void (*destroy)(struct pico_device *self);
  int (*dsr)(struct pico_device *self, int loop_score);
  int __serving_interrupt;
  /* used to signal the upper layer the number of events arrived since the last processing */
  volatile int eventCnt;
};

PACKED_STRUCT_DEF pico_ip4 {
  uint32_t addr;
};

PACKED_STRUCT_DEF pico_ip6 {
  uint8_t addr[16];
};

union pico_address {
  struct pico_ip4 ip4;
  struct pico_ip6 ip6;
};

PACKED_STRUCT_DEF pico_eth {
  uint8_t addr[6];
  uint8_t padding[2];
};


/****** INIT FUNCTIONS ******************************************************/

int pico_device_init(struct pico_device *dev, const char *name, uint8_t *mac);
int32_t pico_stack_recv(struct pico_device *dev, uint8_t *buffer, uint32_t len);
int pico_stack_init(void);
void pico_stack_tick(void);
void pico_stack_loop(void);
void pico_mem_init(uint32_t memsize);
void pico_mem_deinit(void);


/****** IPv4 ****************************************************************/

int pico_ipv4_compare(struct pico_ip4 *a, struct pico_ip4 *b);
int pico_ipv4_to_string(char *ipbuf, const uint32_t ip);
int pico_string_to_ipv4(const char *ipstr, uint32_t *ip);
int pico_ipv4_valid_netmask(uint32_t mask);
int pico_ipv4_is_unicast(uint32_t address);
int pico_ipv4_is_multicast(uint32_t address);
int pico_ipv4_is_broadcast(uint32_t addr);
int pico_ipv4_is_loopback(uint32_t addr);
int pico_ipv4_is_valid_src(uint32_t addr, struct pico_device *dev);

int pico_ipv4_link_add(struct pico_device *dev, struct pico_ip4 address, struct pico_ip4 netmask);
int pico_ipv4_link_del(struct pico_device *dev, struct pico_ip4 address);
int pico_ipv4_rebound(struct pico_frame *f);

int pico_ipv4_frame_push(struct pico_frame *f, struct pico_ip4 *dst, uint8_t proto);
struct pico_ipv4_link *pico_ipv4_link_get(struct pico_ip4 *address);
struct pico_ipv4_link *pico_ipv4_link_by_dev(struct pico_device *dev);
struct pico_ipv4_link *pico_ipv4_link_by_dev_next(struct pico_device *dev, struct pico_ipv4_link *last);
struct pico_device *pico_ipv4_link_find(struct pico_ip4 *address);
struct pico_ip4 *pico_ipv4_source_find(const struct pico_ip4 *dst);
struct pico_device *pico_ipv4_source_dev_find(const struct pico_ip4 *dst);
int pico_ipv4_route_add(struct pico_ip4 address, struct pico_ip4 netmask, struct pico_ip4 gateway, int metric, struct pico_ipv4_link *link);
int pico_ipv4_route_del(struct pico_ip4 address, struct pico_ip4 netmask, int metric);
struct pico_ip4 pico_ipv4_route_get_gateway(struct pico_ip4 *addr);
void pico_ipv4_route_set_bcast_link(struct pico_ipv4_link *link);
void pico_ipv4_unreachable(struct pico_frame *f, int err);


/****** IPv6 ****************************************************************/

int pico_string_to_ipv6(const char *ipstr, uint8_t *ip);


/****** DNS CLIENT **********************************************************/

#define PICO_DNS_NS_DEL 0
#define PICO_DNS_NS_ADD 1
int pico_dns_client_init(void);
int pico_dns_client_nameserver(struct pico_ip4 *ns, uint8_t flag);
int pico_dns_client_getaddr(const char *url, void (*callback)(char *ip, void *arg), void *arg);
int pico_dns_client_getname(const char *ip, void (*callback)(char *url, void *arg), void *arg);
int pico_dns_client_getaddr6(const char *url, void (*callback)(char *, void *), void *arg);
int pico_dns_client_getname6(const char *url, void (*callback)(char *, void *), void *arg);


/****** DHCP CLIENT *********************************************************/

int pico_dhcp_initiate_negotiation(struct pico_device *device, void (*callback)(void*cli, int code), uint32_t *xid);
void *pico_dhcp_get_identifier(uint32_t xid);
struct pico_ip4 pico_dhcp_get_address(void *cli);
struct pico_ip4 pico_dhcp_get_gateway(void *cli);
struct pico_ip4 pico_dhcp_get_netmask(void *cli);
struct pico_ip4 pico_dhcp_get_nameserver(void*cli, int index);
int pico_dhcp_client_abort(uint32_t xid);
char *pico_dhcp_get_hostname(void);
char *pico_dhcp_get_domain(void);

/* possible codes for the callback */
#define PICO_DHCP_SUCCESS 0
#define PICO_DHCP_ERROR   1
#define PICO_DHCP_RESET   2


/****** ICMP ****************************************************************/

typedef uint64_t pico_time;
struct pico_icmp4_stats {
    struct pico_ip4 dst;
    unsigned long size;
    unsigned long seq;
    pico_time time;
    unsigned long ttl;
    int err;
};

int pico_icmp4_port_unreachable(struct pico_frame *f);
int pico_icmp4_proto_unreachable(struct pico_frame *f);
int pico_icmp4_dest_unreachable(struct pico_frame *f);
int pico_icmp4_mtu_exceeded(struct pico_frame *f);
int pico_icmp4_ttl_expired(struct pico_frame *f);
int pico_icmp4_frag_expired(struct pico_frame *f);
int pico_icmp4_ping(char *dst, int count, int interval, int timeout, int size, void (*cb)(struct pico_icmp4_stats *));
int pico_icmp4_ping_abort(int id);


/****** TCP *****************************************************************/

#define PICO_SOCK_EV_RD 1u
#define PICO_SOCK_EV_WR 2u
#define PICO_SOCK_EV_CONN 4u
#define PICO_SOCK_EV_CLOSE 8u
#define PICO_SOCK_EV_FIN 0x10u
#define PICO_SOCK_EV_ERR 0x80u

#define PICO_SHUT_RD   1
#define PICO_SHUT_WR   2
#define PICO_SHUT_RDWR 3

struct pico_msginfo {
    struct pico_device *dev;
    uint8_t ttl;
    uint8_t tos;
};

struct pico_socket *pico_socket_open(uint16_t net, uint16_t proto, void (*wakeup)(uint16_t ev, struct pico_socket *s));

int pico_socket_read(struct pico_socket *s, void *buf, int len);
int pico_socket_write(struct pico_socket *s, const void *buf, int len);

int pico_socket_sendto(struct pico_socket *s, const void *buf, int len, void *dst, uint16_t remote_port);
int pico_socket_sendto_extended(struct pico_socket *s, const void *buf, const int len, void *dst, uint16_t remote_port, struct pico_msginfo *msginfo);

int pico_socket_recvfrom(struct pico_socket *s, void *buf, int len, void *orig, uint16_t *local_port);
int pico_socket_recvfrom_extended(struct pico_socket *s, void *buf, int len, void *orig, uint16_t *remote_port, struct pico_msginfo *msginfo);

int pico_socket_send(struct pico_socket *s, const void *buf, int len);
int pico_socket_recv(struct pico_socket *s, void *buf, int len);

int pico_socket_bind(struct pico_socket *s, void *local_addr, uint16_t *port);
int pico_socket_getname(struct pico_socket *s, void *local_addr, uint16_t *port, uint16_t *proto);
int pico_socket_getpeername(struct pico_socket *s, void *remote_addr, uint16_t *port, uint16_t *proto);

int pico_socket_connect(struct pico_socket *s, const void *srv_addr, uint16_t remote_port);
int pico_socket_listen(struct pico_socket *s, const int backlog);
struct pico_socket *pico_socket_accept(struct pico_socket *s, void *orig, uint16_t *port);
int8_t pico_socket_del(struct pico_socket *s);

int pico_socket_setoption(struct pico_socket *s, int option, void *value);
int pico_socket_getoption(struct pico_socket *s, int option, void *value);

int pico_socket_shutdown(struct pico_socket *s, int mode);
int pico_socket_close(struct pico_socket *s);


#endif
