/*
 * Ping tool for DOS, based on the picotcp4dos library.
 * Copyright (C) 2015 Mateusz Viste
 *
 * http://picotcp4dos.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <dos.h>
#include <stdio.h>
#include <unistd.h> /* sleep() */
#include <stdlib.h> /* malloc(), free() */
#include <time.h>

#include <picotcp.h>
#include <picodos.h>

#include "version.h"


int icmpres = 0;
int exitcode = 1;
char *dnsres = NULL;


/* callback function called when we got a DNS result */
static void cb_dns(char *ip, void *arg) {
  if (ip == NULL) {
    dnsres = "";
  } else {
    dnsres = ip;
  }
}


static void cb_ping(struct pico_icmp4_stats *s) {
  char *host;
  if (s == NULL) {
    printf("Ooops: cb_ping() called with s == NULL\n");
    return;
  }
  host = malloc(64);
  if (host != NULL) pico_ipv4_to_string(host, s->dst.addr);
  if (s->err == 0) {
    long fullsec, csecs;
    fullsec = s->time / 1000;
    csecs = (s->time % 1000) / 10;
    printf("%lu bytes from %s: icmp_req=%lu ttl=%lu time=%01ld.%02lds\n", s->size, host, s->seq, s->ttl, fullsec, csecs);
    icmpres++;
    exitcode = 0; /* if at least one ping is good, exit with 0 */
  } else {
    printf("PING %lu to %s: Error %d\n", s->seq, host, s->err);
    icmpres++;
  }
  free(host);
}


/*
void cb_ping6(struct pico_icmp6_stats *s) {
    char host[50];
    pico_ipv6_to_string(host, s->dst.addr);
    if (s->err == 0) {
        dbg("%lu bytes from %s: icmp_req=%lu ttl=%lu time=%lu ms\n", s->size, host, s->seq,
            s->ttl, (long unsigned int)s->time);
        if (s->seq >= NUM_PING)
            exit(0);
    } else {
        dbg("PING %lu to %s: Error %d\n", s->seq, host, s->err);
        exit(1);
    }
} */


/*
void ping_abort_timer(pico_time now, void *_id) {
  int *id = (int *) _id;
  printf("Ping: aborting...\n");
  if (!IPV6_MODE) {
    pico_icmp4_ping_abort(*id);
  } else {
    pico_icmp6_ping_abort(*id);
  }
} */



int main(int argc, char **argv) {
  struct pico_device picodev;
  time_t starttime;
  int res;
  struct pico_ip4 dst;
  char *dstptr = NULL;

  if (argc != 2) {
    printf("invalid number of arguments\nusage: ping host\n");
    return(1);
  }
  if ((argv[1][0] == '/') || (argv[1][0] == '-')) {
    printf("picoTCP ping / Copyright (C) " PDATE " Mateusz Viste / ver " PVER "\n" PURL "\n\nusage: ping host\n");
    return(1);
  }
  dstptr = argv[1];

#ifdef DBG
  printf("picoinit()... ");
  sleep(1);
#endif
  res = picoinit(&picodev, 0);
#ifdef DBG
  printf("OK\n");
  sleep(1);
#endif

  if (res != 0) {
    printf("ERROR: Failed to init picotcp (error %d)\n", res);
    return(1);
  }

  if (pico_string_to_ipv4(dstptr, &dst.addr) == 0) { /* no DNS resolution needed */
    dnsres = dstptr;
  } else { /* DNS resolution is required */
    dnsres = NULL;
    if (pico_dns_client_getaddr(dstptr, cb_dns, NULL) != 0) {
      printf("DNS failure\n");
      picoquit(&picodev);
      return(1);
    }
    #ifdef DBG
    printf("while() pico_stack_tick()\n");
    sleep(1);
    #endif
    starttime = time(NULL);
    while (dnsres == NULL) {
      union REGS regs;
      int86(0x28, &regs, &regs); /* DOS 2+ IDLE INTERRUPT */
      pico_stack_tick();
      if (time(NULL) - starttime > 4) dnsres = "";
    }
    if (dnsres[0] == 0) {
      printf("unknown hostname\n");
      free(dnsres);
      picoquit(&picodev);
      return(1);
    }
    if (pico_string_to_ipv4(dnsres, &dst.addr) != 0) {
      printf("ERROR: %s is not a valid IP address\n", dnsres);
      free(dnsres);
      picoquit(&picodev);
      return(1);
    }
  }

  printf("PING %s (%s)...\n", dstptr, dnsres);
#ifdef DBG
  sleep(1);
#endif

  /* if (ipv6) { */
  res = pico_icmp4_ping(dnsres, 3, 1000, 5000, 64, cb_ping);
  /* } else {
    res = pico_icmp6_ping(dest, NUM_PING, 1000, 10000, 64, cb_ping6, NULL);
  } */

  free(dnsres);

  if (res == -1) {
    printf("ERROR: internal error (%d, %d)\n", res, pico_err);
    picoquit(&picodev);
    return(1);
  }

  starttime = time(NULL);

  while (icmpres < 3) {
    time_t now = time(NULL);
    union REGS regs;
    int86(0x28, &regs, &regs); /* DOS 2+ IDLE INTERRUPT */
#ifdef DBG
    printf("pico_stack()\n");
    sleep(1);
#endif
    pico_stack_tick();
#ifdef DBG
    printf("pico_stack() OK\n");
    sleep(1);
#endif
    if (now - starttime > 10) {
      printf("timeout\n");
      break;
    }
  }

  picoquit(&picodev);
  return(exitcode);
}
