-- pictures.e - Writes/reads scattered data to/from bitmaps - HIP 2.0
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file contains routines for writing a sequence of bits (as
-- generated by scatter.e) into an image stored in memory, and for
-- reading the bits written in an image.


constant include_name = "pictures.e"

include errors.e
include pal.e


constant bit_pos = #10000

function create_24bit_table (atom bits)
    -- Create substitution table for 24-bit pictures

    atom subst, mask, step
    sequence tables, cur_table

    -- log_info_s ("create_24bit_table: start %d", bits, LOG_DETAIL)

    step = power(2, bits)

    tables = repeat( {}, step)

    mask = and_bits (#FF, not_bits(step-1))

    for n = 1 to length(tables) do
	cur_table = repeat(0, 256)
	for x = 0 to 255 do
	    -- Set the low-order bits to the ones of the current table
	    subst = or_bits (and_bits (x, mask), n-1)

	    if subst > x and subst >= step then subst = subst - step end if

	    if (subst+step)-x < x-subst and subst+step <= 255 then
		cur_table [x+1] = subst+step
	    else
		cur_table [x+1] = subst
	    end if
	end for

	tables[n] = cur_table

    end for

    return tables
end function


function create_8bit_table (sequence pal, sequence sorted, atom max_bits)
    -- Create substitution table for 8-bit pictures

    sequence lists, original, low_table, high_table

    -- log_info_s ("create_8bit_table: start %d", max_bits, LOG_DETAIL)

    -- Generate low_table
    lists = color_lists (pal, sorted, max_bits - 1)
    original = lists[2]    lists = lists[1]

    low_table = create_substitution_table (pal, lists, original)

    -- Generate high_table
    lists = color_lists (pal, sorted, max_bits)
    original = lists[2]    lists = lists[1]

    high_table = create_substitution_table (pal, lists, original)

    return {low_table, high_table}
end function

function find_alternative_rgb (sequence rgb,
    sequence transparent, sequence scattered)

    -- Find the color (corresponding to the bits in scattered)
    -- which is closest to the color given
    -- Make sure the transparent color is not selected

    atom min_distance, distance
    sequence bits_per_component, masked, mask, best_color, cur_rgb,
	     power_bits

    -- log_info_s ("find_alternative_rgb: start %02x%02x%02x %02x%02x%02x %02x%02x%02x", rgb&transparent&scattered, LOG_DETAIL)

    bits_per_component = floor (scattered / bit_pos)
    power_bits = power(2, bits_per_component)
    mask = not_bits ( power_bits - 1 )

    masked = and_bits (rgb, mask)

    for x = 1 to 3 do
	if masked [x] = and_bits (255, mask[x]) then    -- Maximum possible value
	    masked [x] = masked [x] - power_bits [x]
	end if
    end for

    masked = masked + and_bits (scattered, 255)

    min_distance = max_distance

    for change_1 = 0 to 1 do
	for change_2 = 0 to 1 do
	    for change_3 = 0 to 1 do

		cur_rgb = masked + ({change_1, change_2, change_3} * power_bits)
		if compare(cur_rgb, transparent) then
		    distance = color_distance (rgb, cur_rgb)
		    if distance < min_distance then
			min_distance = distance
			best_color = cur_rgb
		    end if
		end if
	    end for
	end for
    end for

    return best_color
end function


atom img_size, min_high, bpp, pos_in_img, pos_in_scattered, bytes_left
sequence scattered, low_table, high_table, converted_table
object transparent

constant bytes_per_call = 2048 * 4 * 3  -- This number must be divisible by
    -- 3, otherwise the program will not handle 24-bit pictures correctly.

global procedure init_write_data_in_img (sequence scattered_, atom max_bits,
    sequence img_info,sequence hip_info)
    -- Initialize writing the bits in the picture

    sequence temp

    -- log_info ("init_write_data_in_img: start", LOG_IMPORTANT)

    scattered = scattered_

    bpp = img_info [IMG_BPP]

    transparent = img_info [IMG_TRANSPARENT]

    min_high = bit_pos * max_bits


    if bpp = 8 then
	-- Create color substitution table
	-- log_info ("init_write_data_in_img: calling create_8bit_table", LOG_DETAIL)

	temp =  create_8bit_table (img_info [IMG_PAL], hip_info [HIP_SORTEDPAL], max_bits)
	low_table = temp [1]
	high_table = temp [2]

	img_size = img_info [IMG_HEIGHT] * img_info [IMG_WIDTH]


    elsif bpp = 24 then

	-- log_info ("init_write_data_in_img: calling create_24bit_table", LOG_DETAIL)
	low_table = create_24bit_table (max_bits-1)
	high_table = create_24bit_table (max_bits)

	img_size = img_info [IMG_HEIGHT] * img_info [IMG_WIDTH] * 3

    else
	abort_error (include_name, "init_write_data_in_img", "Unsupported BPP")
    end if


    bytes_left = img_size

    pos_in_img = img_info [IMG_ADDRESS]
    pos_in_scattered = 1

    -- log_info_s ("init_write_data_in_img: bmp size = %d ; exit", img_size, LOG_DETAIL)


end procedure

global function go_write_in_img ()
    -- Write the bits in a part of the picture
    -- return INFO_LASTBLOCK if there is no need to call the routine again,
    -- 0 otherwise

    atom cur_scattered

    sequence img_block, rgb, rgb_to_write

    -- log_info_s ("go_write_in_img: start %d bytes left", bytes_left, LOG_DETAIL)

    if bytes_left <= bytes_per_call then
	-- Last block
	img_block = peek ( {pos_in_img, bytes_left} )

    else
	-- Not last block
	img_block = peek ({pos_in_img, bytes_per_call})
    end if


    if bpp = 8 then
	-- 8-bit picture
	for x = 1 to length (img_block) do
	    if img_block [x] != transparent then

		cur_scattered = scattered[pos_in_scattered]
		if cur_scattered >= min_high then
		    -- Pixel will contain the higher number of bits
		    img_block [x] = high_table [and_bits(cur_scattered, 255)+1][img_block [x]+1]
		else
		    img_block [x] = low_table [and_bits(cur_scattered, 255)+1][img_block [x]+1]
		end if
		pos_in_scattered = pos_in_scattered + 1
	    end if
	end for


    elsif bpp = 24 then
	-- 24-bit picture
	rgb_to_write = {-1, -1, -1}

	for xa = 1 to length (img_block) by 3 do
	    rgb = img_block [xa..xa+2]

	    if compare(rgb, transparent) then
		-- not transparent color

		for x = 1 to 3 do
		    -- replace each color by the closest appropriate value

		    cur_scattered = scattered[pos_in_scattered]
		    if cur_scattered >= min_high then
			-- Pixel will contain the higher number of bits
			rgb_to_write [x] = high_table [and_bits(cur_scattered, 255)+1][rgb[x]+1]
		    else
			rgb_to_write [x] = low_table [and_bits(cur_scattered, 255)+1][rgb[x]+1]

		    end if

		    pos_in_scattered = pos_in_scattered + 1

		end for

		if compare(rgb_to_write, transparent) = 0 then
		    -- The color created is the transparent one,
		    -- find closest non-transparent color
		    rgb_to_write = find_alternative_rgb (rgb, transparent, scattered [pos_in_scattered-3..pos_in_scattered-1])
		end if

		-- Write new color to the image
		img_block [xa..xa+2] = rgb_to_write

	    end if

	end for


    else
	abort_error (include_name, "go_write_in_img", "Unsupported BPP")

    end if

    poke(pos_in_img, img_block)
    pos_in_img = pos_in_img + bytes_per_call
    bytes_left = bytes_left - bytes_per_call

    if bytes_left > 0 then
	return 0
    else
	return INFO_LASTBLOCK
    end if

end function



global procedure init_read_data_from_img (sequence img_info, sequence hip_info)
    -- Initialize reading the bits in the picture

    -- log_info ("init_read_data_in_img: start", LOG_IMPORTANT)

    scattered = repeat (-1, hip_info [HIP_AVAILABLEBYTES])
    transparent = img_info [IMG_TRANSPARENT]

    bpp = img_info [IMG_BPP]

    if bpp = 8 then

	-- log_info ("init_read_data_from_img: calling inverse_transform", LOG_IMPORTANT)
	img_size = img_info [IMG_WIDTH] * img_info [IMG_HEIGHT]

	converted_table = inverse_transform ( hip_info [HIP_SORTEDPAL])

    elsif bpp = 24 then
	img_size = img_info [IMG_WIDTH] * img_info [IMG_HEIGHT] * 3

    elsif bpp != 24 then
	-- Unsupported BPP
	abort_error (include_name, "init_read_data_from_img", "Unsupported BPP")
    end if

    bytes_left = img_size

    pos_in_img = img_info [IMG_ADDRESS]

    pos_in_scattered = 1

    -- log_info_s ("init_read_data_from_img: bmp size = %d ; exit", img_size, LOG_IMPORTANT)

end procedure

global function go_read_from_img ()
    -- Read the bits in a part of the picture
    -- return INFO_LASTBLOCK if there is no need to call the routine again,
    -- 0 otherwise

    atom pixel_index
    sequence img_block, pixel_rgb

    -- log_info_s ("go_read_from_img: start %d bytes left", bytes_left, LOG_DETAIL)

    if bytes_left <= bytes_per_call then
	img_block = peek ({pos_in_img, bytes_left})
    else
	img_block = peek ({pos_in_img, bytes_per_call})
    end if


    if bpp = 8 then

	for x = 1 to length (img_block) do

	    pixel_index = img_block [x]

	    if pixel_index != transparent then
		scattered [pos_in_scattered] = converted_table [pixel_index + 1]
		pos_in_scattered = pos_in_scattered + 1
	    end if

	end for

    elsif bpp = 24 then

	for xa = 1 to length (img_block) by 3 do

	    pixel_rgb = img_block [xa..xa+2]

	    if compare (pixel_rgb, transparent) then
		for x = 1 to 3 do
		    scattered [pos_in_scattered] = pixel_rgb [x]
		    pos_in_scattered = pos_in_scattered + 1
		end for
	    end if

	end for


    else
	abort_error (include_name, "go_read_from_img", "Unsupported BPP")

    end if



    pos_in_img = pos_in_img + bytes_per_call
    bytes_left = bytes_left - bytes_per_call

    if bytes_left > 0 then
	return 0
    else
	return INFO_LASTBLOCK
    end if

end function

global function get_pic_data()
    -- Returns the bits read from the picture
    return scattered
end function

global function pic_completion_status ()
    -- Returns a number (0 to 1) indicating the progress
    return 1 - (bytes_left / img_size)
end function

global procedure end_pic ()
    -- Erase variables used in the reading/writing process

    -- log_info ("end_pic: start", LOG_IMPORTANT)
    scattered = {}
    low_table = {}
    high_table = {}
    converted_table = {}
    transparent = -1
end procedure

