/*
 * This file is part of the ipcfg tool
 * http://picotcp4dos.sourceforge.net
 *
 * Copyright (C) 2015 Mateusz Viste
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <stdio.h>  /* FILE */

#include "picoconf.h" /* include self for control */


/* internal routing that reads an offset from file (at position pos) and jumps
 * to it, if valid. returns 0 on success, non-zero otherwise. */
static int jumpfd(FILE *fd, int pos) {
  long offset;
  fseek(fd, pos, SEEK_SET);
  offset = fgetc(fd);
  offset <<= 8;
  offset |= fgetc(fd);
  if (offset < 0) return(-1);
  if (offset == 0) {
    fseek(fd, pos, SEEK_SET);
  } else {
    fseek(fd, offset, SEEK_SET);
  }
  return(0);
}


/* open the picotcp config file. returns a FILE ptr on success, NULL otherwise */
FILE *picoconf_open(char *filename) {
  char sig[4];
  int freadres;
  FILE *fd;
  /* must point to a file that can be opened for read and write */
  fd = fopen(filename, "ab+");
  if (fd == NULL) return(NULL);
  rewind(fd);
  /* if it's a 0-sized file, populate it with default values */
  if (fgetc(fd) == EOF) fwrite("pTCP\0\0\0\0\140\0\0\0\0", 1, 13, fd);
  fclose(fd);
  /* reopen the file for read-only */
  fd = fopen(filename, "rb");
  if (fd == NULL) return(NULL);
  /* a valid sig hdr must be present */
  freadres = fread(sig, 1, 4, fd);
  if ((freadres != 4) || (sig[0] != 'p') || (sig[1] != 'T') || (sig[2] != 'C') || (sig[3] != 'P')) {
    fclose(fd);
    return(NULL);
  }
  return(fd);
}


void picoconf_close(FILE *fd) {
  fclose(fd);
}


int picoconf_getint(FILE *fd, int *pktint) {
  if (pktint == NULL) {
    fseek(fd, 0x08, SEEK_SET);
    return(fgetc(fd));
  }
  *pktint = fgetc(fd);
  *pktint <<= 8;
  *pktint |= fgetc(fd);
  if (*pktint < 0) return(-1);
  return(0);
}


/* returns 0 on success, non-zero otherwise (if no more IP addresses), unless
 * addr is NULL, then it will only return the amount of addresses that can
 * be fetched (or a negative value on error). this function MUST be called
 * first with addr == NULL, and only then you can fetch addresses
 * sequentially */
int picoconf_getip(FILE *fd, unsigned char *addr, int *masklen) {
  unsigned char c;
  /* if first time, move the fd pointer to correct location and read */
  if (addr == NULL) {
    if (jumpfd(fd, 0x0B) != 0) return(-1);
    return(fgetc(fd));
  }
  /* read the IP address/masklen from file */
  if (fread(addr, 1, 16, fd) != 16) return(-1);
  if (fread(&c, 1, 1, fd) != 1) return(-1);
  *masklen = c;
  return(0);
}


/* returns 0 on success, non-zero otherwise (if no more routes), unless addr
 * is NULL, then it will only return the amount of routes that can be fetched
 * (or a negative value on error). this function MUST be called first with
 * addr == NULL, and only then you can fetch routes sequentially */
int picoconf_getrt(FILE *fd, unsigned char *addr, int *prefixlen, unsigned char *gw) {
  /* if first time, move the fd pointer to correct location and read */
  if (addr == NULL) {
    if (jumpfd(fd, 0x04) != 0) return(-1);
    return(fgetc(fd));
  }
  /* read the prefix/len/gw from file */
  if (fread(addr, 1, 16, fd) != 16) return(-1);
  if (fread(gw, 1, 1, fd) != 1) return(-1);
  *prefixlen = gw[0];
  if (fread(gw, 1, 16, fd) != 16) return(-1);
  return(0);
}


/* returns 0 on success, non-zero otherwise (if no more DNS servers), unless
 * addr is NULL, then it will only return the amount of DNS servers that can
 * be fetched (or a negative value on error). this function MUST be called
 * first with addr == NULL, and only then you can fetch addresses
 * sequentially */
int picoconf_getdns(FILE *fd, unsigned char *addr) {
  /* if first time, move the fd pointer to correct location and read */
  if (addr == NULL) {
    if (jumpfd(fd, 0x06) != 0) return(-1);
    return(fgetc(fd));
  }
  /* read the IP address/masklen from file */
  if (fread(addr, 1, 16, fd) != 16) return(-1);
  return(0);
}
