;Wolfware Assembler
;Copyright (c) 1985-1991 Eric Tauck. All rights reserved.

;===============================================;
;                   Reset_Nam                   ;
; Initialize all default name components.       ;
;===============================================;

Reset_Nam Proc Near
 Mov Bx, Def_Pth        ;path
 Mov Byte [Bx], 0
 Mov Bx, Def_Nam        ;name
 Mov Byte [Bx], 0
 Mov Bx, Def_Ext        ;extension
 Mov Byte [Bx], 0
 Ret
 Endp

;===============================================;
;                  Format_Name                  ;
; Write the complete file name to DI using the  ;
; name at SI, the default drive and path in     ;
; DEF_PTH, the extension in DEF_EXT, and the    ;
; default name in DEF_NAM. If a drive, path,    ;
; name, or extension are not specified, the     ;
; default settings are used. The default        ;
; settings are updated to the new name.         ;
;===============================================;

Format_Name Proc Near
 Push Di
 Push Si
 Call Fcaps_Set         ;set proper capital letters
 Pop Si
 Call Extract_Tot       ;extract all individual parts
 Pop Di

;----- construct name

 Mov Si,Def_Pth         ;path storage
 Store_Str              ;move
 Mov Si,Def_Nam         ;name storage
 Store_Str              ;move
 Mov Al,'.'             ;period
 Stosb                  ;store
 Mov Si,Def_Ext         ;extension storage
 Store_Str              ;move
 Sub Al,Al              ;name terminator
 Stosb
 Ret
 Endp                   ;Format_Name

;===============================================;
;                    Reextract                  ;
; Put the (formatted) file name at SI into      ;
; NAME_BUFF and extract all components.         ;
;===============================================;

Reextract Proc Near

;----- name into NAME_BUFF

 Sub Al,Al              ;scan for 00
 Mov Cx,String_Siz      ;max characters to scan
 Push Si
 Call Scan_Char         ;scan
 Pop Bx
 Dec Si                 ;back up to point at 00
 Xchg Bx,Si             ;flip start and end
 Mov Di,Name_Buff       ;storage
 Call Ex_Move           ;move

;----- extract

 Mov Si,Name_Buff       ;name location
 Call Extract_Tot       ;extract
 Ret
 Endp                   ;Reextract

;===============================================;
;                   Extract_Tot                 ;
; Extract all components of the name at SI and  ;
; set default values.                           ;
;===============================================;

Extract_Tot Proc Near
 Push Si
 Call Extract_Pth       ;extract path

 Mov Bp,Sp              ;stack
 Mov Si,[Bp]            ;restore source
 Call Extract_Nam       ;extract name

 Pop Si
 Call Extract_Ext       ;extract extension
 Ret
 Endp                   ;Extract_Tot

;===============================================;
;                  Extract_Pth                  ;
; Extract the drive and path from the file      ;
; name at SI and put into DEF_PTH.              ;
;===============================================;

Extract_Pth Proc Near

;----- locate drive and path

 Push Si
 Call Name_Start        ;find end of drive and path
 Mov Bx,Si              ;end location
 Pop Si
 Inc Si                 ;skip length, start location

;----- extract

 Mov Cx,Pth_Size        ;max size
 Mov Di,Def_Pth         ;name storage
 Call Ex_Move           ;extract
 Ret
 Endp                   ;Extract_Pth

;===============================================;
;                  Extract_Nam                  ;
; Extract the name from the file name at SI     ;
; and put into DEF_NAM.                         ;
;===============================================;

Extract_Nam Proc Near

;----- locate name

 Call Name_Start        ;skip drive and path
 Push Si

 Mov Al,'.'             ;period marks extension
 Call Scan_Char         ;scan for start
 Jc Exnadot             ;jump if found

 Add Si,Cx              ;point to end, end location with no dot
 Jmps Exnacont
Exnadot Dec Si          ;back up to period, end location with dot

Exnacont Mov Bx,Si
 Pop Si                 ;start location

;----- extract

 Mov Cx,Nam_Size        ;max size
 Mov Di,Def_Nam         ;name storage
 Call Ex_Move           ;extract
 Ret
 Endp                   ;Extract_Nam

;===============================================;
;                  Extract_Ext                  ;
; Extract the extension from the file name at   ;
; SI and put into DEF_EXT.                      ;
;===============================================;

Extract_Ext Proc Near

;----- locate extension

 Call Name_Start        ;skip drive and path
 Mov Al,'.'             ;period marks extension
 Call Scan_Char         ;scan for start
 Jnc Exexndot           ;jump if not found

 Mov Bx,Si              ;SI has start location
 Add Bx,Cx              ;stop location

;----- extract

 Mov Cx,Ext_Size        ;max size
 Mov Di,Def_Ext         ;extension storage
 Mov Byte [Di],0        ;clear extension (found dot, no default)
 Call Ex_Move           ;extract
Exexndot Ret
 Endp                   ;Extract_Ext

;===============================================;
;                    Ex_Move                    ;
; Move the string starting at SI and ending at  ;
; BX to DI. No effect if BX=SI. CX should       ;
; contain the maximum bytes to move.            ;
;===============================================;

Ex_Move Proc Near
 Sub Bx,Si              ;BX gets length
 Or Bx,Bx               ;check if zero
 Jz Neexmov             ;jump if so

 Cmp Bx,Cx              ;check if too big
 Jbe Exmsok             ;jump if not
 Mov Bx,Cx              ;max size

Exmsok Mov Ax,Bx
 Stosb                  ;store length

 Mov Cx,Bx
 Rep
 Movsb
Neexmov Ret
 Endp                   ;Ex_Move

;===============================================;
;                  Name_Start                   ;
; Find start of name. Using the name at SI,     ;
; find the right most backslash. If backslash   ;
; not found then find first colon. SI returns   ;
; pointing to byte after the backslash or       ;
; colon. CX returns the bytes remaining. If no  ;
; path then SI simply  points to the first      ;
; byte (after the length) and CX returns the    ;
; length.                                       ;
;===============================================;

Name_Start Proc Near
 Lodsb                  ;load length byte
 Mov Cl,Al
 Sub Ch,Ch              ;CX gets length
 Mov Al,'\'             ;backslash
 Push Cx

Namstloop Call Scan_Char ;scan for backslash
 Jc Namstloop           ;loop back until not found

 Pop Dx                 ;inital bytes remaning
 Cmp Dx,Cx              ;check if same
 Je Namstsam            ;jump if so, not found
 Ret

;----- backslash not found, look for colon

Namstsam Mov Al,':'     ;colon
 Call Scan_Char         ;scan for backslash
 Ret
 Endp                   ;Name_Start

;===============================================;
;                   Scan_Char                   ;
; Scan for character. Given the bytes to scan   ;
; in CX and the location in SI, look for the    ;
; character in AL. CY=1 if found, with SI       ;
; pointing to the  next character and CX with   ;
; the bytes remaining. Otherwise CX and SI are  ;
; unchanged and CY=0.                           ;
;===============================================;

Scan_Char Proc Near
 Push Cx
 Or Cx,Cx               ;check if nul
 Jz Noscmat             ;jump if so, not found

 Mov Di,Si
 Repne
 Scasb                  ;scan
 Jne Noscmat            ;jump if not found

;----- character matched

 Mov Si,Di
 Add Sp,2               ;throw away size
 Stc
 Ret

;----- no match

Noscmat Pop Cx
 Clc
 Ret
 Endp                   ;Scan_Char

;===============================================;
;                   Fcaps_Set                   ;
; Sets the first letter of the drive, all       ;
; subdirectories, the filename, and the         ;
; extension to upper-case.  All other letters   ;
; are set to lower-case. SI must point to the   ;
; file name to set.                             ;
;===============================================;

Fcaps_Set Proc Near
 Lodsb                  ;length
 Or Al,Al               ;check if zero
 Jz Fcsdone             ;jump if so

 Mov Di,Si
 Mov Cl,Al
 Sub Ch,Ch

;----- start of name

Fcscaps Lodsb           ;load
 Cmp Al,'\'             ;check if start of name
 Je Fcsspec
 Cmp Al,':'             ;check if start of name
 Je Fcsspec
 Cmp Al,'.'             ;check if start of extension
 Je Fcsspec
 Char_Upper Al          ;make upper-case
 Stosb
 Loop Fcsloop           ;loop back for next byte CX times
Fcsdone Ret

;----- remaining characters are lower

Fcsloop Lodsb           ;load
 Cmp Al,'\'             ;check if start of name
 Je Fcsspec
 Cmp Al,':'             ;check if start of name
 Je Fcsspec
 Cmp Al,'.'             ;check if start of extension
 Je Fcsspec
 Char_Lower Al          ;make lower-case
 Stosb                  ;store
 Loop Fcsloop           ;loop back for next byte CX times
 Ret

;----- special character

Fcsspec Stosb           ;store
 Loop Fcscaps           ;loop for next byte, make it capital
 Ret
 Endp                   ;Fcaps_Set

;===============================================;
;                   Store_Nam                   ;
; Stores the specified parts of the file name   ;
; at SI to NAME_BUFF as a string (first byte    ;
; is length). The name at SI becomes  the       ;
; default name. The effects for the bits  set   ;
; in AH are:                                    ;
;  1 - Store the drive and path.                ;
;  2 - Store the name (not including ext).      ;
;  3 - Store the extension.                     ;
;  4 - Print results to screen.                 ;
;===============================================;

Store_Nam Proc Near
 Push Di
 Push Si

 Push Ax
 Call Reextract         ;pick apart name
 Pop Ax

 Mov Di,Name_Buff       ;place to put name
 Inc Di                 ;space for length

;----- drive and path

 Test Ah,Bit1           ;test for drive
 Jz Nodrive             ;jump if not

 Mov Si,Def_Pth         ;path
 Store_Str              ;move

;----- name

Nodrive Test Ah,Bit2    ;test for name
 Jz Nooname             ;jump if not

 Mov Si,Def_Nam         ;path
 Store_Str              ;move

;----- extension

Nooname Test Ah,Bit3    ;test for extension
 Jz Nooext              ;jump if not

 Mov Si,Def_Ext         ;path
 Cmp Byte [Si], 0       ;check if no extension
 Je Nooext

 Mov Al,'.'             ;period
 Stosb                  ;store

 Store_Str              ;move

;----- set string size

Nooext Mov Si,Name_Buff ;beginning of storage
 Sub Di,Si              ;length of output + length
 Dec Di                 ;remove length, DI has size
 Mov Cx,Di
 Mov [Si],Cl            ;store size

;----- print to screen

 Test Ah,Bit4           ;test if output to screen
 Jz Naoprn              ;jump if so

 Call Print_Str         ;print string

Naoprn Pop Si
 Pop Di
 Ret
 Endp                   ;Store_Nam

;===============================================;
;                   Name_Type                   ;
; Check if the name in DEF_NAM is something     ;
; special (a device). AL returns the type of    ;
; file.                                         ;
;===============================================;

Name_Type Proc Near
 Mov Di,Offset File_Table ;device table
 Mov Si,Def_Nam         ;name location
 Call Tab_Look          ;table look up
 Jnc Namalt             ;jump if not in table

;----- special file name, device

 Mov Al,Bl
 Ret

;----- other file type (not a device)

Namalt Mov Al,Alt_File  ;alternate type
 Ret

;----- device types

File_Table Label Byte
 Db 3,'Nul'             ;nul device
 Db 3,'Con'             ;console
 Db 3,'Prn'             ;printer
 Db 0
 Endp                   ;Name_Type

