{
    BSD 3-Clause License
    Copyright (c) 2022, Jerome Shidel
    All rights reserved.
}

{ Danger Engine based game, see https://gitlab.com/DOSx86/sge }

{$I DANGER.DEF}
{ DEFINE DEVEL}
program SaysWho;

uses Danger;

var
	Level           : integer;
	Lives			: word;
	PauseMode		: word;
	LevelStr		: String;
	TriesStr		: String;
	DeathStr		: String;
	StartStr		: String;
	GameOverStr		: String;
	HintAStr,
	HintBStr		: String;
	PlayPos			: integer;
	PlayWait		: integer;
	HintWait 		: integer;
	HintFlip		: boolean;
	PlayState		: boolean;
	LastButton		: integer;
	CurrentButton	: integer;

	PlayArea,
	HintArea		: TArea;
	Blocks			: array[0..3] of record
		LowImg,
		HighImg : PImage;
		Place	: TPoint;
		Area	: TArea;
		Freq	: integer;
	end;
	Sequence		: array[0..$8000] of byte;

const
	pmNone			 = 0;
	pmStart			 = 1;
	pmDeath			 = 3;
	pmGameOver		 = 4;
	pmPlayback		 = 5;
	pmListen		 = 6;
	pmWrong			 = 7;

	Background		 = 0;
	TextColor		 = 7;
	StatusColor		 = 15;
	StatusbarColor	 = 250;
	HintColor		 = 8;

	PlayTime		 = 24;
	PlayDelay		 = 8;
	HintMin 		 = 48;
	HintDelay		 = 6;
	StartLives		 = 4;

procedure Initialize;
begin
    AppInfo.Title 		:= 'SaysWho';
    AppInfo.Year 		:= '2022';
    AppInfo.Version 	:= '0.2';
    AppInfo.FontName 	:= '1214N-EN';
    Lives  				:= StartLives;
    Level  				:= -1;
    PauseMode			:= pmNone;
end;

procedure ProcessCommandLine;
var
    I, E : integer;
    Opt : String;
begin
    I := 0;
    while I < ParamCount do begin
        Inc(I);
        Opt := UCase(ParamStr(I));
        if (Opt = '/H') or (Opt = '/?') then begin
            PrintHelp;
        end;
        if Opt = '/BM' then begin
            AppInfo.Driver.Video   := 'BIOSVID.DRV';
            Inc(I);
            if UCase(ParamStr(I)) <> 'LIST' then
                Val(ParamStr(I), AppInfo.VideoMode, E)
            else
                AppInfo.VideoMode:=$ffff;
            Inc(I);
        end else if Opt = '/VGA' then begin
            AppInfo.Driver.Video   := 'VGA386.DRV'; { at present, default }
        end else if Opt = '/LOG' then begin
            {$IFDEF LOGS} Logging := True; {$ENDIF}
        end;
    end;
end;

procedure Prepare;
var
    P : PAsset;
	I : integer;
	H : integer;
begin
    Video^.Fill(0);
    Video^.SetSync(True);
    Video^.SetBuffered(True);
    Randomize;
	LevelStr 	:= ' ' + Trim(RawNLS('LEVEL')) + ' ';
	TriesStr 	:= ' ' + Trim(RawNLS('TRIES')) + ' ';
	DeathStr    := Trim(RawNLS('DEATH'));
	StartStr    := Trim(RawNLS('START'));
	GameOverStr := Trim(RawNLS('GAMEOVER'));
	HintAStr 	:= Trim(RawNLS('HINT'));
	HintBStr	:= '';
	HintFlip 	:= false;
	SetArea(0, GetMaxY - Video^.TextHeight(HintAStr), GetMaxX, GetMaxY, HintArea);
	if (Video^.TextWidth(HintAStr) > GetMaxX) then begin
		I := Pos(',', HintAStr);
		if I > 0 then begin
			HintBStr := Trim(Copy(HintAStr, I + 1, Length(HintAStr)));
			HintAStr := Trim(Copy(HintAStr, 1, I));
		end;
	end;
	HintWait := HintMin + HintDelay * Length(HintAStr);
	H := Video^.TextHeight('') + 1;
	SetArea(0, H, GetMaxX, GetMaxY - H, PlayArea);
	for I := 0 to 3 do begin
	    Blocks[I].LowImg := AssetLoadOrDie(IntStr(I) + 'L.IGG', asDefault, P);
	    Blocks[I].HighImg := AssetLoadOrDie(IntStr(I) + 'H.IGG', asDefault, P);
	    with Blocks[I] do begin
	    	case I of
	    		0 : SetPoint(GetMaxX div 2 - LowImg^.Width * 3 div 2 - 10,
						GetMaxy div 2 - LowImg^.Height div 2, Place);
	    		1 : SetPoint(GetMaxX div 2 - LowImg^.Width div 2,
						GetMaxy div 2 + LowImg^.Height div 2 - 10, Place);
	    		2 : SetPoint(GetMaxX div 2 + LowImg^.Width div 2 + 10,
						GetMaxy div 2 - LowImg^.Height div 2, Place);
	    		3 : SetPoint(GetMaxX div 2 - LowImg^.Width div 2,
						GetMaxy div 2 - LowImg^.Height * 3 div 2 + 10, Place);
	    	end;
		    SetArea(0,0,LowImg^.Width - 1, LowImg^.Height - 1, Area);
		    MoveArea(Place.X, Place.Y, Area);
		    Freq := 200 + I * 50;
		end;
	end;
end;

procedure DrawStats;
var
	A : TArea;
	W, WW : Integer;
begin
	SetArea(0,0,GetMaxX, Video^.TextHeight('') + 1, A);
	Video^.FillArea(A, StatusbarColor);
	W := Video^.TextWidth(LevelStr);
	Video^.PutText(0, 1, LevelStr, TextColor);
	Video^.PutText(W, 1, IntStr(Level + 1), StatusColor);
	W := Video^.TextWidth(TriesStr);
	WW := Video^.TextWidth(IntStr(Lives) + ' ');
	Video^.PutText(GetMaxX - W - WW, 1, TriesStr, TextColor);
	Video^.PutText(GetMaxX - WW, 1, IntStr(Lives - 1), StatusColor);
end;

procedure DrawHint(Redraw : boolean);
begin
	if not Redraw then begin
		Dec(HintWait);
		if HintWait > 0 then exit;
	end;
	Video^.FillArea(HintArea, 0);
	if HintFlip then begin
		Video^.PutText(
			GetMaxX div 2 - Video^.TextWidth(HintBStr) div 2,
			GetMaxY - Video^.TextHeight(HintBStr),
			HintBStr, HintColor);
		if not Redraw then HintWait := HintMin + HintDelay * Length(HintBStr);
	end else begin
		Video^.PutText(
			GetMaxX div 2 - Video^.TextWidth(HintAStr) div 2,
			GetMaxY - Video^.TextHeight(HintAStr),
			HintAStr, HintColor);
		if not Redraw then HintWait := HintMin + HintDelay * Length(HintAStr);
	end;
	if Redraw then exit;
	if HintBStr <> '' then HintFlip := Not HintFlip;
end;

procedure DrawButton(I : integer; Hot : boolean);
begin
	With Blocks[I] do
		case Hot of
			False : Video^.PutImage(LowImg,  Place.X, Place.Y);
			True  : Video^.PutImage(HighImg, Place.X, Place.Y);
		end;
end;

procedure DrawScreen;
var
	I : integer;
begin
	DrawStats;
	DrawHint(True);
	for I := 0 to 3 do
		DrawButton(I, False);
end;

procedure PauseGame(Style : integer);
const
	X : integer = 0;
	Y : Integer = 0;
	I : PImage = nil;
var
	W : integer;
	S : String;
	A : TArea;
begin
	PauseMode := Style;
	case Style of
		pmNone : begin
			if Assigned(I) then begin
				Video^.PutImage(I, X, Y);
				Video^.FreeImage(I);
			end;
			Exit;
		end;
		pmStart 	: S := StartStr;
		pmDeath 	: S := DeathStr;
		pmGameOver 	: S := GameOverStr;
		pmListen	: begin
			LastButton := dmNowhere;
			CurrentButton := dmNowhere;
			Exit;
		end;
	else
		Exit;
	end;
	S := '  ' + S + '  ';
	W := Video^.TextWidth(S);
	Y := GetMaxY div 2 - Video^.TextHeight(S) * 3 div 2;
	X := PlayArea.Left + AreaWidth(PlayArea) div 2 - W div 2;
	SetArea(X, Y, X + W - 1, Y + Video^.TextHeight(S) * 3 - 1, A);
	I := Video^.NewImage(W, Video^.TextHeight(S) * 3);
	Video^.GetImage(I, X, Y);
	Video^.FillArea(A, 0);
	Video^.FrameArea(A, 2, 7,8);
	Video^.PutText(X, Y + Video^.TextHeight(S), S, 15);
end;

procedure StartLevel;
begin
	SetVRTInterval(3);
	Video^.Fill(Background);
	DrawScreen;
	PlayPos   := -1;
	PlayWait  := (PlayDelay + PlayTime);
	PlayState := False;
	PauseGame(pmPlayback);
end;

procedure NextLevel;
begin
	if HintBStr <> '' then begin
		HintAStr := '';
		HintBStr := '';
	end;
	Inc(Level);
	Sequence[Level] := Random(80) and $03;
	StartLevel;
end;

procedure Death;
begin
	Dec(Lives);
	LastButton := dmNowhere;
	CurrentButton := dmNowhere;
	PlayWait  := 0;
	PlayState := False;
	if Lives <= 0 then
		PauseGame(pmGameOver)
	else
		PauseGame(pmDeath);
end;

procedure NewGame;
begin
    Lives  			:= StartLives;
    Level  			:= -1;
    PauseMode		:= pmStart;
    NextLevel;
end;

procedure Playback;
begin
	Dec(PlayWait);
	if PlayWait > 0 then Exit;
	if PlayState = true then begin
		DrawButton(Sequence[PlayPos], False);
		PlayWait := PlayDelay;
		PlayState := False;
		Audio^.Silence;
		{ NoSound; }
	end else begin
		Inc(PlayPos);
		if Level < PlayPos then begin
			PlayPos := 0;
			PauseGame(pmListen);
			Exit;
		end;
		DrawButton(Sequence[PlayPos], True);
		PlayWait := PlayTime;
		PlayState := True;
		Audio^.Sound(Blocks[Sequence[PlayPos]].Freq);
	end;
end;

procedure PlayDeath;
begin
	Dec(PlayWait);
	if PlayWait > 0 then Exit;
	PauseGame(pmNone);
	PlayState := Not PlayState;
	DrawButton(Sequence[PlayPos], PlayState);
	if PlayState then
		PlayWait := PlayTime
	else
		PlayWait := PlayTime div 2;
	PauseGame(pmDeath);
end;

procedure PlayGameOver;
begin
	Dec(PlayWait);
	if PlayWait > 0 then Exit;
	PauseGame(pmNone);
	PlayState := Not PlayState;
	DrawButton(Sequence[PlayPos], PlayState);
	if PlayState then
		PlayWait := PlayTime div 2
	else
		PlayWait := PlayDelay div 2;
	PauseGame(pmGameOver);
	if not PlayState then begin
		Inc(PlayPos);
		if PlayPos > Level then begin
			PlayPos := 0;
			PlayWait := (PlayTime + PlayDelay) * 4;
		end;
	end;
end;

function DirToButton(D : TDirection) : integer;
begin
	case D of
		dmLeft  : DirToButton := 0;
		dmDown  : DirToButton := 1;
		dmRight : DirToButton := 2;
		dmUp    : DirToButton := 3;
	else
		DirToButton := -1;
	end;
end;

procedure Listen;
begin
	if CurrentButton = LastButton then Exit;
	if LastButton <> dmNowhere then begin
		DrawButton(DirToButton(LastButton), False);
		Audio^.Silence;
	end;
	if CurrentButton <> dmNowhere then begin
		if Sequence[PlayPos] <> DirToButton(CurrentButton) then begin
			Death;
			Exit;
		end;
		DrawButton(DirToButton(CurrentButton), True);
		Audio^.Sound(Blocks[DirToButton(CurrentButton)].Freq);
		Inc(PlayPos);
	end else begin
		if PlayPos > Level then NextLevel;
	end;
	LastButton := CurrentButton;
end;

procedure Play;
var
    Event : TEvent;
    LTT : word;
    R, BTest : integer;
begin
	DrawScreen;
    PurgeEvents;
    LTT := VRTimer^;
    PauseGame(pmStart);
    if MousePresent then MouseShow;
    repeat
        Video^.Update;
        While not GetEvent(Event) and (LTT = VRTimer^) do Idle;
        Video^.Prepare;
        if MouseEvent(Event) then begin
            MouseMove(Event.Position);
            if Event.Kind and evMouseClick = evMouseClick then begin
            	case PauseMode of
            		pmStart 		: NextLevel;
            		pmDeath 		: StartLevel;
            		pmGameOver 		: NewGame;
            		pmListen		: begin
            			CurrentButton := dmNowhere;
						for BTest := 0 to 3 do
							if InArea(Event.Position, Blocks[BTest].Area) then
								case BTest of
									0 : CurrentButton := dmLeft;
									1 : CurrentButton := dmDown;
									2 : CurrentButton := dmRight;
									3 : CurrentButton := dmUp;
								end;
            		end;
            	end;
            end else
            if Event.Kind and evMouseRelease = evMouseRelease then begin
				CurrentButton := dmNowhere;
			end;
        end;
        if Event.Kind and evKeyPress = evKeyPress then begin
       		CurrentButton := dmNowhere;
			case PauseMode of
           		pmStart 		: NextLevel;
				pmDeath 		: StartLevel;
				pmGameOver 		: NewGame;
			end;
            case Event.KeyCode of
                kbEscape : Break;
                kb2 : CurrentButton := dmDown;
                kb4 : CurrentButton := dmLeft;
                kb6 : CurrentButton := dmRight;
                kb8 : CurrentButton := dmUp;
            else
            	case Event.Scancode of
            		scDown,  scNumPad2 : CurrentButton := dmDown;
	                scLeft,  scNumPad4 : CurrentButton := dmLeft;
    	            scRight, scNumPad6 : CurrentButton := dmRight;
        	        scUp,	 scNumPad8 : CurrentButton := dmUp;
        	    end;
            end;
        end else
        if Event.Kind and evKeyRelease = evKeyRelease then begin
			CurrentButton := dmNowhere;
        end;
        if LTT <> VRTimer^ then begin
	        LTT := VRTimer^;
	        if LTT and $1 = 0 then Video^.SpriteNextAll(False);
			case PauseMode of
				pmStart 	: begin
					R := Random(1);
					if HintBStr <> '' then DrawHint(false);
				end;
				pmNone 		: NextLevel;
				pmPlayBack  : Playback;
				pmListen 	: Listen;
				pmDeath		: PlayDeath;
				pmGameOver	: PlayGameOver;
			end;
		end;
    until False;
    MouseHide;
	Audio^.Silence;
	Audio^.NoSound;
end;

{$I INTRO.INC}

begin
	Initialize;
    ProcessCommandLine;
    Ignite;
    Prepare;
    {$IFNDEF DEVEL}
    PlayIntro;
    {$ENDIF}
    SetVRTInterval(3);
    Video^.SetSync(True);
    Play;
    {$IFNDEF DEVEL}
    PlayExtro;
    {$ENDIF}
    Extinguish;
end.