/*
 * rcal - a "rolling" calculator
 * Copyright 2018 (C) Mateusz Viste
 *
 * http://rcal.sourceforge.net
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <i86.h>

#include "io.h"

int io_getkey(void) {
  /* INT 21h, AH=08  /  DOS 1+ - CHARACTER INPUT WITHOUT ECHO
   * AL = character read from standard input */
  union REGS regs;
  regs.h.ah = 0x08;
  int86(0x21, &regs, &regs);
  if (regs.h.al == 0) {  /* extended key - poll again */
    regs.h.ah = 0x08;
    int86(0x21, &regs, &regs);
    return(0x100 | regs.h.al);
  }
  return(regs.h.al);
}

void io_getcursorpos(int *row, int *col) {
  /* INT 10h, AH=03h
   * BH = page number
   * Returns:
   *   CH = start scan line
   *   CL = end scan line
   *   DH = row (0 is top)
   *   DL = column (0 is left) */
  union REGS regs;
  regs.h.ah = 0x03;
  regs.h.bh = 0;
  int86(0x10, &regs, &regs);
  *row = regs.h.dh;
  *col = regs.h.dl;
}

void io_setcursorpos(int row, int col) {
  /* INT 10h, AH=02h
   * BH = page number
   * DH = row (0 is top)
   * DL = column (0 is left) */
  union REGS regs;
  regs.h.ah = 0x02;
  regs.h.bh = 0;
  regs.h.dh = row;
  regs.h.dl = col;
  int86(0x10, &regs, &regs);
}

void io_setcursorxpos(int col) {
  int x, y;
  io_getcursorpos(&y, &x);
  io_setcursorpos(y, col);
}

void io_movcursorx(int delta) {
  int x, y;
  io_getcursorpos(&y, &x);
  io_setcursorpos(y, x + delta);
}

void io_printdosstr(char *s) {
  /* INT 21h, AH=09h   DOS 1+ - WRITE STRING TO STANDARD OUTPUT
   * DS:DX -> '$'-terminated string */
  union REGS regs;
  struct SREGS sr;
  regs.h.ah = 0x09;
  sr.ds = FP_SEG(s);
  regs.x.dx = FP_OFF(s);
  int86x(0x21, &regs, &regs, &sr);
}

void io_printdoschar(char c) {
  /* INT 21h, AH=02h  DOS 1+ - WRITE CHARACTER TO STANDARD OUTPUT
   * DL = character to write */
  union REGS regs;
  regs.h.ah = 0x02;
  regs.h.dl = c;
  int86(0x21, &regs, &regs);
}

int io_getcountryparms(void *buf) {
  /* INT 21h, AH=38h  DOS 2.11+ - GET COUNTRY-SPECIFIC INFORMATION
   * AL = 00h get current-country info
   * DS:DX -> buffer for returned info
   * Return:
   *   CF set on error, clear otherwise
   *   AX = error code (02h)
   * Format of country info:
   * 00h    WORD    date format
   * 02h  5 BYTEs   ASCIZ currency symbol string
   * 07h  2 BYTEs   ASCIZ thousands separator
   * 09h  2 BYTEs   ASCIZ decimal separator
   * 0Bh  2 BYTEs   ASCIZ date separator
   * 0Dh  2 BYTEs   ASCIZ time separator
   * 0Fh    BYTE    currency format
   * ...
   * 18h 10 BYTEs   reserved
   */
  union REGS regs;
  regs.x.ax = 0x3800u;
  regs.x.dx = FP_OFF(buf);
  int86(0x21, &regs, &regs);
  return(regs.x.cflag);
}

int io_fcreat(char *f, unsigned short *handle) {
  union REGS r;
  struct SREGS sr;
  r.h.ah = 0x5b; /* DOS 3.0+ CREATE NEW FILE */
  r.x.cx = 0;
  sr.ds = FP_SEG(f);
  r.x.dx = FP_OFF(f);
  int86x(0x21, &r, &r, &sr);
  *handle = r.x.ax;
  return(r.x.cflag);
}

int io_fwrite(unsigned short handle, void *s, unsigned short len) {
  union REGS r;
  struct SREGS sr;
  while (len > 0) {
    r.h.ah = 0x40; /* DOS 2+ WRITE - WRITE TO FILE OR DEVICE */
    r.x.bx = handle;
    r.x.cx = len;
    sr.ds = FP_SEG(s);
    r.x.dx = FP_OFF(s);
    int86x(0x21, &r, &r, &sr);
    if (r.x.cflag != 0) return(-1);
    len -= r.x.ax;
    s = (char *)s + r.x.ax;
  }
  return(0);
}

int io_fclose(unsigned short handle) {
  union REGS r;
  r.h.ah = 0x3e; /* DOS 2+ CLOSE - CLOSE FILE */
  r.x.bx = handle;
  int86(0x21, &r, &r);
  return(r.x.cflag);
}
