/*      mousec.c                            14.03.95       */
/*!
/  --------------------------------------------------------------
/  Copyright (C) 1993: Michael Braun
/                      Kaetinger Muehlenweg 103 A
/                      D-28816 Stuhr
/  --------------------------------------------------------------
/
/    handling of mouse events
/
/
/    Copyright Note:
/    This program was developed on the base of a chapter in
/    the following book. It has been carried on and adapted
/    to the special requirements of mbedit by M. Braun.
/
/       Title    : PC intern 3.0
/       Author   : Michael Tischer
/       Publisher: Data Becker, Duesseldorf
/       ISBN     : 3-89011-591-8
/       1st edition 1992
/
*/

/*== Include-Dateien einbinden =======================================*/

#include "config.h"
#include "global.h"
#include "standard.h"
#include "mousec.h"
#include "mbedit.h"
#include "perform.h"
#include "history.h"
#include "disp_hnd.h"
#include "mon_outp.h"
#include "mbed_sub.h"
#include "kb_input.h"


#define UEBERFLUESSIG 0  /* ist jetzt selbst ueberfluessig ! 09.03.95 */
#define RING_BUFFER   0


#if (!WITH_MOUSE)

/* dummy routine */
void MouEventHandler( int EvFlags, int ButState, int x, int y )
{
   EvFlags;     /* wg. compiler warning */
   ButState;
   x;
   y;

   return;
}  /* MouEventHandler */

#else

extern void far AssHand( void );            /* Extern-Deklaration des */
                                            /* Assembler-Handlers     */
/*== Typedefs ========================================================*/

typedef unsigned char BYTE;               /* wir basteln uns ein Byte */
typedef unsigned long PTRVIEW;               /* Maske fr Maus-Cursor */
typedef struct {                      /* beschreibt einen Mausbereich */
                BYTE x1,             /* Koordinaten der oberen linken */
                     y1,             /* und unteren rechten Ecke des  */
                     x2,             /* spezifizierten Bereichs       */
                     y2;
               } BEREICH;
typedef void (far * MOUHAPTR)( void );   /* Pointer auf Event-Handler */

/*== Konstanten ======================================================*/


/*-- Event-Codes -----------------------------------------------------*/

#define EV_MOU_MOVE      1                             /* Maus bewegt */
#define EV_LEFT_PRESS    2            /* linker Mausknopf niedergedr. */
#define EV_LEFT_REL      4            /* linker Mausknopf losgelassen */
#define EV_RIGHT_PRESS   8           /* rechter Mausknopf niedergedr. */
#define EV_RIGHT_REL    16           /* rechter Mausknopf losgelassen */
#define EV_MOU_ALL      31                        /* alle Maus-Events */

#define KEIN_BEREICH 255           /* Maus-Cursor nicht in Bereich xy */

/*-- Makros ----------------------------------------------------------*/

#define MouGetCol()         (ev_col)        /* liefern Mausposition & */
#define MouGetRow()         (ev_row)        /* -bereich im Moment des */
#define MouGetBereich()     (ev_ber)        /* Event-Eintritts        */
#define MouAvail()          ( mavail )  /* liefert TRUE, wenn Maus v. */
#define MouGetAktCol()      ( moucol )       /* liefern jeweils aktu- */
#define MouGetAktRow()      ( mourow )       /* elle Mausposition und */
#define MouGetAktBer()      ( mouber )       /* aktuellen Mausbereich */
#define MouIsLeftPress()    ( mouevent & EV_LEFT_PRESS )
#define MouIsLeftRel()      ( mouevent & EV_LEFT_REL )
#define MouIsRightPress()   ( mouevent & EV_RIGHT_PRESS )
#define MouIsRightRel()     ( mouevent & EV_RIGHT_REL )
#define MouSetMoveAreaAll() MouSetMoveArea( 0, 0, (BYTE) (tcol-1), \
                                                  (BYTE) (tline-1), 0 );

#define ELVEK(x) ( sizeof(x) / sizeof(x[0]) ) /* Anzahl Elemente in X */

/*-- Makros zur Erstellung der Bit-Maske, die das Erscheinungsbild ---*/
/*-- des Maus-Cursors definiert.                                   ---*/
/*-- Der Aufruf von MouPtrMask lautet beispielsweise:              ---*/
/*--   MouPtrMask( PTRDIFCHAR( 'x' ), PTRINVCOL )                  ---*/
/*-- um den Maus-Cursor als ein kleines X darzustellen, das die    ---*/
/*-- invertierte Farbe des Zeichens trgt, das von ihm berdeckt   ---*/
/*-- wird.                                                         ---*/

#define MouPtrMask( z, f )\
  ( (( (PTRVIEW) f) >> 8 << 24) + ((( PTRVIEW) z) >> 8 << 16) +\
    (((f) & 255) << 8) + ((z) & 255) )

#define PTRSAMECHAR   ( 0x00ff )        /* gleiches Zeichen           */
#define PTRDIFCHAR(z) ( (z) << 8 )      /* anderes Zeichen            */
#define PTRSAMECOL    ( 0x00ff )        /* gleiche Farbe              */
#define PTRINVCOL     ( 0x7777 )        /* Farbe invertiert           */
#define PTRSAMECOLB   ( 0x807f )        /* gleiche Farbe blinkend     */
#define PTRINVCOLB    ( 0xF777 )        /* invertierte Farbe blinkend */
#define PTRDIFCOL(f)  ( (f) << 8 )      /* andere Farbe               */
#define PTRDIFCOLB(f) (((f)|0x80) << 8) /* andere Farbe blinkend      */

#define MOUINT(rin, rout) int86(0x33, &rin, &rout)
#define MOUINTX(rin, rout, sr) int86x(0x33, &rin, &rout, &sr)

/*-- Makros zur Umrechnung zwischen Maus-Koordinaten in Bezug auf den */
/*-- virtuellen Maus-Bildschirm und dem Textbildschirm                */

#define SPALTEN_94  0  /* 1 = 94-Spalten-Modus, 0 = andere Modi (80) */

#if SPALTEN_94

static const int size_tab [] = {
                    8, 8, 8, 8, 8, 8, 8, 8, 8, 8,  /*  0 -  9 */
                    8, 8, 8, 8, 8, 8, 8, 8, 8, 8,  /* 10 - 19 */
                    8, 8, 8, 8, 8, 8, 8, 8, 7, 8,  /* 20 - 29 */
                 /*                ^25      ^28               */
                    8, 8, 8, 8, 7, 8, 8, 8, 8, 8,  /* 30 - 39 */
                 /* ^30         ^34                           */
                    8, 8, 8, 4, 8, 8, 8, 8, 8, 8,  /* 40 - 49 */
                 /*          ^43                              */
                    4, 8, 8, 8, 8, 8, 8, 8, 8, 8,  /* 50 - 59 */
                 /* ^50                                       */
                    4, 8, 8, 8 };                  /* 60 - 63 */
                 /* ^60                                       */

#if 0
#define XTOCOL(x) ( ((x) / ((COLUMNS == 94) ? 7 : 8)) )
#define YTOROW(y) ( ((y) / size_tab[ROWS & 0x3f]) )
#define COLTOX(c) ( ((c) * ((COLUMNS == 94) ? 7 : 8)) + 4 )
#define ROWTOY(r) ( ((r) * size_tab[ROWS & 0x3f])     + 3 )
#else
#define XTOCOL(x) ( ((x) / 8) )
#define YTOROW(y) ( ((y) / size_tab[ROWS & 0x3f]) )
#define COLTOX(c) ( ((c) * 8) + 4 )
#define ROWTOY(r) ( ((r) * size_tab[ROWS & 0x3f])     + 3 )
#endif

#else

#define XTOCOL(x) ( ((x) >> 3) )          /* Rundung ! */   /* X durch 8 */
#define YTOROW(y) ( ((y) >> 3) )                            /* Y durch 8 */
#define COLTOX(c) ( ((c) << 3) + 4)                         /* C mal 8 */
#define ROWTOY(r) ( ((r) << 3) + 4)                         /* R mal 8 */

#endif

/*== globale Variablen ===============================================*/

static BYTE tline,                              /* Anzahl Text-Zeilen */
            tcol,                              /* Anzahl Text-Spalten */
            mavail = FALSE;          /* ist TRUE, wenn Maus verfgbar */

/*-- Maske fr den Standard-Maus-Cursor ------------------------------*/

static PTRVIEW stdptr = MouPtrMask( PTRSAMECHAR, PTRINVCOL );
static int  mouevent = EV_LEFT_REL + EV_RIGHT_REL;     /* Event-Maske */
static BYTE moucol,                   /* Mausspalte (Text-Bildschirm) */
            mourow;                   /* Mauszeile  (Text-Bildschirm) */

static byte center_col = (INIT_COLUMNS / 2);
static byte center_row = (INIT_ROWS / 2);
static byte old_win_row, old_win_col;
static int  event_count, disp_window_active;


/***********************************************************************
*  Funktion         : M o u D e f i n e P t r                          *
**--------------------------------------------------------------------**
*  Aufgabe          : Definiert die Cursor- und die Screen-Maks, die   *
*                     das Erscheinungsbild des Maus-Cursors bestimmen  *
*  Eingabe-Parameter: MASK = die beiden Bit-Masken, vereint zu einem   *
*                            32-Bit-Wert vom Typ UNSIGNED LONG         *
*  Return-Wert      : keiner                                           *
*  Info             : - Die hherwertigen 16-Bit von MASK reprsen-    *
*                       tieren die Screen-Mask, die niederwertigen 16  *
*                       Bit die Cursor-Mask                            *
***********************************************************************/

#pragma check_stack(off)                  /* kein Stack-Checking hier */

void MouDefinePtr( PTRVIEW mask )
{
 static PTRVIEW altercursor = (PTRVIEW) 0;   /* letzter Wert fr MASK */
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 if ( altercursor != mask )  /* Vernderung gegenber letztem Aufruf? */
  {                                                             /* Ja */
   regs.x.ax = 0x000a;          /* Fktnr. fr "Set text pointer type" */
   regs.x.bx = 0;                       /* Software-Cursor einstellen */
   regs.x.cx = (word) mask;                  /* Lo-Word ist AND-Maske */
   regs.x.dx = (word)(mask >> 16);           /* Hi-Word ist XOR-Maske */
   MOUINT(regs, regs);                       /* Maus-Treiber aufrufen */
   altercursor = mask;                       /* neue Bit-Maske merken */
  }
}

/***********************************************************************
*  Funktion         : M o u E v e n t H a n d l e r                    *
**--------------------------------------------------------------------**
*  Aufgabe          : Wird vom Maustreiber ber die Assembler-Routine  *
*                     AssHand aufgerufen, sobald ein Ereignis in Ver-  *
*                     bindung mit der Maus eingetreten ist.            *
*  Eingabe-Parameter: EvFlags  = Event-Maske des Ereignis              *
*                     ButState = Status der Mausknpfe                 *
*                     X, Y     = die aktuelle Position des Maus-Cur-   *
*                                sors, bereits umgerechnet in das Ko-  *
*                                ordinatensystem des Textbildschirms   *
*  Return-Wert      : keiner                                           *
*  Info             : - Diese Funktion ist nur zum Aufruf durch den    *
*                       Maus-Treiber bestimmt und darf nicht von einer *
*                       anderen Funktion aufgerufen werden.            *
***********************************************************************/

#define TEST_MOUSE 0

#if (TEST_MOUSE)
static int x_sav, y_sav;
#endif


void _loadds MouEventHandler( int EvFlags, int ButState, int x, int y )
{
#define LBITS ( EV_LEFT_PRESS | EV_LEFT_REL )
#define RBITS ( EV_RIGHT_PRESS | EV_RIGHT_REL )

static BYTE oldcol, oldrow;

 ButState;    /* wg. compiler warning */

#if (TEST_MOUSE)
 x_sav = x;
 y_sav = y;
#endif

 mouevent &= ~1;                                  /* Bit 0 ausblenden */
 mouevent |= ( EvFlags & 1 );           /* Bit 0 aus EvFlags kopieren */

 if ( EvFlags & LBITS )      /* linker Mausknopf losg. oder niederg.? */
 {                                                             /* Ja */
   mouevent &= ~LBITS;                /* bisherigen Status ausblenden */
   mouevent |= ( EvFlags & LBITS );        /* neuen Status einblenden */
 }

 if ( EvFlags & RBITS )     /* rechter Mausknopf losg. oder niederg.? */
 {                             /* Ja, Bits ausblenden und einblenden */
   mouevent &= ~RBITS;                /* bisherigen Status ausblenden */
   mouevent |= ( EvFlags & RBITS );        /* neuen Status einblenden */
 }

 moucol = (BYTE) XTOCOL(x);         /* Spalte in Textspalten umrechnen */
 mourow = (BYTE) YTOROW(y);           /* Zeile in Textzeilen umrechnen */


 if ((mourow != oldrow) ||   /* ignore minor movements */
     (moucol != oldcol) ||   /* in the pixel position. */
     (EvFlags & LBITS)  ||   /* handle every button event. */
     (EvFlags & RBITS))
    event_count++;

 oldrow = mourow;
 oldcol = moucol;

 return;
}  /* MouEventHandler */


int mouse_event (void)
{
   return (event_count != 0);
}  /* mouse_event */


int mouse_event_handler_c (int repeat)
{
static int old_col, phase;
static struct dostime_t time_start, time_act;
static int msec10_start, msec10_act;
#define TIME_DELAY 20    /* LSB = 10 msec */

int  new_col, delta_line, key, ii;
long new_line, new_index;
#define SLOW_DOWN  1    /* 1: max. mouse speed */
                        /* 2: reduced "    "   */

static int left_button_tab [] =
           {
              0x00,     /* 0: normal mode */
              0x0d,     /* 1: history window */
              0x0d,     /* 2: filename window */
              0x1b,     /* 3: get_line_2_string */
               'E',     /* 4: hex_view */
              0x09,     /* 5: hex_edit */
           };


#if (TEST_MOUSE)
static int  row, column;
static char text [100];
#endif


   if (event_count > 0)
      event_count--;

   key = 0;  /* default for standard text mode */

/* -----------------------------------------------------------------------*/

/* fallunterscheidung */
   if (disp_window_active)
   {
   /* simulate keys cursor up/down, <enter> + <esc> */

      key = KEY_DO_NOTHING;    /* default for display window */

      if (MouIsRightPress())
         key = 0x03;   /* ^C   */

      if (MouIsLeftPress())
      {
         key = left_button_tab [disp_window_active];
      }
   
      if ((disp_window_active <= 2) ||     /* history/filename window */
          (disp_window_active >= 4))       /* hex view/edit  */
      {
         if (mourow > old_win_row)
         {
            delta_line = mourow - old_win_row;
            for (ii = 1 ; ii < (delta_line/SLOW_DOWN) ; ii++)
            {
               set_key_fifo (KEY_DOWN);
            }
            key = KEY_DOWN;
         }
   
         if (mourow < old_win_row)
         {
            delta_line = old_win_row - mourow;
            for (ii = 1 ; ii < (delta_line/SLOW_DOWN) ; ii++)
            {
               set_key_fifo (KEY_UP);
            }
            key = KEY_UP;
         }
      }

      if (disp_window_active >= 3)    /* get_status_line_2_... */
      {                               /* hex editor            */
         if (moucol > old_win_col)
         {
            delta_line = moucol - old_win_col;
            for (ii = 1 ; ii < (delta_line/SLOW_DOWN) ; ii++)
            {
               set_key_fifo (KEY_RIGHT);
            }
            key = KEY_RIGHT;
         }
   
         if (moucol < old_win_col)
         {
            delta_line = old_win_col - moucol;
            for (ii = 1 ; ii < (delta_line/SLOW_DOWN) ; ii++)
            {
               set_key_fifo (KEY_LEFT);
            }
            key = KEY_LEFT;
         }
      }

      MouMovePtr (center_col, center_row);
      old_win_col = moucol;
      old_win_row = mourow;

   /* @@ prevent multiple reads of mouse buttons !! */
      mouevent &= ~(EV_LEFT_PRESS | EV_RIGHT_PRESS);

      return (key);
   }

/* -----------------------------------------------------------------------*/

/* control repetition behaviour (delay time at begin) */
   if (!repeat)
   {
      phase = 0;        /* no repetition at all */
   }
   else
   {
      if (phase == 0)
      {
         _dos_gettime (&time_start);
         msec10_start = (int) time_start.second * 100 +
                        (int) time_start.hsecond;
         phase = 1;     /* begin of time delay */
      }

      if (phase == 1)
      {
         _dos_gettime(&time_act);
         msec10_act = (int) time_act.second * 100 +
                      (int) time_act.hsecond;

      /* before completion of time delay: abort repetition */
         if ((((msec10_act - msec10_start) + 6000) % 6000) >= TIME_DELAY)
            phase = 2;  /* time delay completed */
      }
   }

/* incompleted time delay: abort repetition */
   if (phase == 1)
      return 0;

/* -----------------------------------------------------------------------*/

/* start handling */
   if (MouIsLeftPress())
   {
   /* set cursor to mouse position */
      check_dirty_cursor ();

   /* set row */
                /* diese klammern muessen sein !! */
                /*          v        v            */
      delta_line = mourow - (fc->lrow);
      if (delta_line)
      {
         new_line = fc->line_index + (long) delta_line;
         new_line = max (0L, min (fc->line_anz, new_line));
         new_index= line_to_index (fc->buff_0, 0L,
                                   fc->byte_anz, new_line);
         if (new_index < 0L)       /* behind EOF */
            new_index = fc->byte_anz;  /* set to EOF */
         perform_move (new_index);
      }

   /* set column */
      new_col = moucol + fc->left_col;
      new_col = max (0, new_col);
      new_index = act_index (fc->buff_0, fc->lin_left,
                             fc->byte_anz, new_col);
      perform_move (new_index);
      fc->column = new_col;

   /* scroll screen up/down */
      if (delta_line == 0)
      {
         if (mourow == (BYTE) MIN_ROW)
            perform_key_up (1);

         if (mourow == (BYTE) MAX_ROW)
            perform_key_down (1);
   
         update_entire_window (fc->top_left);
         perform_update (fc->buff_0, fc->byte_anz, fc->left_col);

      /* set cursor */
         if (get_video_active (0))
            set_cursor_to (fc->lrow, REL_COLUMN);
      }


   /* scroll screen left/right */
      if ((BYTE) old_col == moucol)
      {
         if (moucol >= (BYTE) (COLUMNS-1))
            fc->left_col++;
   
         if (moucol <= 0)
            fc->left_col--;
   
         fc->left_col = max (0, fc->left_col);
      }
      old_col = moucol;


   /* show screen, set cursor */
      refresh_1_window ();
      if (get_video_active (0))
         set_cursor_to (fc->lrow, REL_COLUMN);
   }  /* if left button pressed */


#if (TEST_MOUSE)
   /* status line 1 missbrauchen */
      get_cursor_pos (&row, &column);
      sprintf (text, " <%3d,%3d> <%2d,%2d> ",
                       x_sav, y_sav, moucol, mourow);
      set_cursor_to (BOT_ROW, 0);
      out_string (text);
      set_cursor_to (row, column);
#endif


/* set flag for check_dirty_cursor() in mbedit.c */
   set_mouse_moved ();

   return key;
}  /* mouse_event_handler_c */


#pragma check_stack              /* alten Zustand im Hinblick auf das */
#pragma check_stack              /* Stack-Checking wieder herstellen  */



/***********************************************************************
*  Funktion         : M o u I S e t E v e n t H a n d l e r            *
**--------------------------------------------------------------------**
*  Aufgabe          : Installiert einen Event-Handler, der beim Ein-   *
*                     tritt eines bestimmten Maus-Ereignisses vom      *
*                     Maus-Treiber aufgerufen wird.                    *
*  Eingabe-Parameter: EVENT = Bit-Maske, die das Ereignis spezifi-     *
*                             ziert, bei dessen Eintritt der Event-    *
*                             Handler aufgerufen werden soll.          *
*                     PTR   = Pointer auf den Maus-Handler             *
*  Return-Wert      : keiner                                           *
*  Info             : - EVENT kann durch Oder-Verknpfung der ver-     *
*                       schiedenen Konstanten wie z.B. EV_MOU_MOVE     *
*                       oder EV_LEFT_PRESS aus der Datei gebildet      *
*                       werden                                         *
***********************************************************************/

static void MouISetEventHandler( unsigned event, MOUHAPTR ptr )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */
 struct SREGS sregs;           /* Segmentregister fr Interruptaufruf */

 regs.x.ax = 0x000C;                /* Fktnr. fr "Set Mouse Handler" */
 regs.x.cx = event;                              /* Event-Maske laden */
 regs.x.dx = FP_OFF( ptr );             /* Offsetadresse des Handlers */
 sregs.es  = FP_SEG( ptr );            /* Segmentadresse des Handlers */
 MOUINTX( regs, regs, sregs );               /* Maus-Treiber aufrufen */
}

/***********************************************************************
*  Funktion         : M o u I G e t X                                  *
**--------------------------------------------------------------------**
*  Aufgabe          : Ermittelt die (Text-) Spalte, in der sich der    *
*                     Maus-Cursor befindet.                            *
*  Eingabe-Parameter: keine                                            *
*  Return-Wert      : die Maus-Spalte in Bezug auf den Textbildschirm  *
***********************************************************************/

static BYTE MouIGetX( void )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 regs.x.ax= 0x0003;               /* Fktnr.: fr "Get mouse position" */
 MOUINT( regs, regs );                       /* Maus-Treiber aufrufen */
 return (BYTE) XTOCOL( regs.x.cx );  /* Spalte umrechnen und zurckl. */
}

/***********************************************************************
*  Funktion         : M o u I G e t Y                                  *
**--------------------------------------------------------------------**
*  Aufgabe          : Ermittelt die (Text-) Zeile, in der sich der     *
*                     Maus-Cursor befindet.                            *
*  Eingabe-Parameter: keine                                            *
*  Return-Wert      : die Maus-Zeile in Bezug auf den Textbildschirm   *
***********************************************************************/

static BYTE MouIGetY( void )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 regs.x.ax= 0x0003;               /* Fktnr.: fr "Get mouse position" */
 MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */
 return (BYTE) YTOROW(regs.x.dx);     /* Zeile umrechnen und zurckl. */
}


/***********************************************************************/

int mouse_get_left_button (void)
{
   return MouIsLeftPress();
}

/***********************************************************************
*  Funktion         : M o u S h o w M o u s e                          *
**--------------------------------------------------------------------**
*  Aufgabe          : Maus-Cursor auf dem Bildschirm anzeigen.         *
*  Eingabe-Parameter: keine                                            *
*  Return-Wert      : keiner                                           *
*  Info             : Die Aufrufe von MouHidemMouse() und MouShow-     *
*                     Mouse() mssen ausbalanciert sein, damit sie     *
*                     Wirkung zeigen.                                  *
***********************************************************************/

static int mouse_on_off_status;

void MouShowMouse( void )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 if (mouse_on_off_status == 0)
 {
    mouse_on_off_status++;

    regs.x.ax = 0x0001;                   /* Fktnr.: fr "Show Mouse" */
    MOUINT(regs, regs);                      /* Maus-Treiber aufrufen */
 }
}

/***********************************************************************
*  Funktion         : M o u H i d e M o u s e                          *
**--------------------------------------------------------------------**
*  Aufgabe          : Maus-Cursor vom dem Bildschirm entfernen.        *
*  Eingabe-Parameter: keine                                            *
*  Return-Wert      : keiner                                           *
*  Info             : Die Aufrufe von MouHidemMouse() und MouShow-     *
*                     Mouse() mssen ausbalanciert sein, damit sie     *
*                     Wirkung zeigen.                                  *
***********************************************************************/

void MouHideMouse( void )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 if (mouse_on_off_status == 1)
 {
    mouse_on_off_status--;
    regs.x.ax = 0x0002;                    /* Fktnr. fr "Hide Mouse" */
    MOUINT(regs, regs);                      /* Maus-Treiber aufrufen */
 }
}

/***********************************************************************
*  Funktion         : M o u S e t M o v e A r e a                      *
**--------------------------------------------------------------------**
*  Aufgabe          : Definiert den Bildschirmbereich, innerhalb dessen*
*                     sich der Maus-Cursor bewegen darf.               *
*  Eingabe-Parameter: x1, y1 = Koordinaten der oberen linken Ecke      *
*                     x2, y2 = Koordinaten der unteren rechten Ecke    *
*  Return-Wert      : keiner                                           *
*  Info             : - Beide Angaben beziehen sich auf den Textbild-  *
*                       schirm und nicht auf den virtuellen Grafik-    *
*                       bildschirm des Maustreibers                    *
***********************************************************************/

#define STACK_SIZE 3

struct WORK_AREA
   {
      byte x1, y1, x2, y2;
      byte save_row, save_col;
      int  win_flag;
   };

static struct WORK_AREA work_area [STACK_SIZE];
static int stack_ind;


void MouSetMoveArea( BYTE x1, BYTE y1, BYTE x2, BYTE y2,
                     int window_flag )
{
/* window_flag = 0 : normal text editing */
/*               1 : inside history window */
/*               2 : inside wildname window */
/*               3 : inside get_line_2_string */
/*               4 : inside hex view/edit */
/*              -1 : return to next lower level */

union REGS regs;                /* Prozessorregs fr Interruptaufruf */

/* handle display windows (history, wildname) */  
   center_col = (byte) ((x1 + x2) / 2);
   center_row = (byte) ((y1 + y2) / 2);

   if (window_flag == 0)
   {
      work_area[stack_ind].save_row = mourow;
      work_area[stack_ind].save_col = moucol;

      work_area[stack_ind].x1 = x1;
      work_area[stack_ind].y1 = y1;
      work_area[stack_ind].x2 = x2;
      work_area[stack_ind].y2 = y2;

      work_area[stack_ind].win_flag = window_flag;
   }

   if (window_flag > 0)   /* save old cursor position */
   {
      work_area[stack_ind].save_row = mourow;
      work_area[stack_ind].save_col = moucol;

      stack_ind++;
      stack_ind = min (stack_ind, (STACK_SIZE-1));

      work_area[stack_ind].x1 = x1;
      work_area[stack_ind].y1 = y1;
      work_area[stack_ind].x2 = x2;
      work_area[stack_ind].y2 = y2;

      work_area[stack_ind].win_flag = window_flag;

      moucol = center_col;
      mourow = center_row;
   }

   if (window_flag < 0)   /* restore old cursor position */
   {
      stack_ind--;
      stack_ind = max (stack_ind, 0);

      mourow = work_area[stack_ind].save_row;
      moucol = work_area[stack_ind].save_col;

      x1 = work_area[stack_ind].x1;
      y1 = work_area[stack_ind].y1;
      x2 = work_area[stack_ind].x2;
      y2 = work_area[stack_ind].y2;
   }

   disp_window_active = work_area[stack_ind].win_flag;


/* set hardware registers */
   regs.x.ax = 0x0008;              /* Fktnr. fr "Set vertical Limits" */
   regs.x.cx = ROWTOY( y1 );                /* Umrechnung in virtuellen */
   regs.x.dx = ROWTOY( y2 );                /* Mausbildschirm           */
   MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */
   regs.x.ax = 0x0007;            /* Fktnr. fr "Set horizontal Limits" */
   regs.x.cx = COLTOX( x1 );                /* Umrechnung in virtuellen */
   regs.x.dx = COLTOX( x2 );                /* Mausbildschirm           */
   MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */

   MouMovePtr (moucol, mourow);

   return; 
}

/***********************************************************************
*  Funktion         : M o u S e t S p e e d                            *
**--------------------------------------------------------------------**
*  Aufgabe          : Legt das Verhltnis zwischen der Lnge einer     *
*                     Mausbewegung und der daraus resultierenden Be-   *
*                     wegung des Maus-Cursors fest.                    *
*  Eingabe-Parameter: - XSPEED = Geschwindigkeit horizontal            *
*                     - YSPEED = Geschwindigkeit vertikal              *
*  Return-Wert      : keiner                                           *
*  Info             : - Beide Angaben beziehen sich auf die Einheit    *
*                       Mickey / 8 Pixel.                              *
***********************************************************************/

void MouSetSpeed( int xspeed, int yspeed )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 regs.x.ax = 0x000f;           /* Fktnr. "Set mickeys to pixel ratio" */
 regs.x.cx = xspeed;
 regs.x.dx = yspeed;
 MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */
}

/***********************************************************************
*  Funktion         : M o u M o v e P t r                              *
**--------------------------------------------------------------------**
*  Aufgabe          : Bewegt den Maus-Cursor an eine bestimmte Bild-   *
*                     schirmposition.                                  *
*  Eingabe-Parameter: - COL = neue Bildschirmspalte                    *
*                     - ROW = neue Bildschirmzeile                     *
*  Return-Wert      : keiner                                           *
*  Info             : - Beide Angaben beziehen sich auf den Textbild-  *
*                       schirm und nicht auf den virtuellen Grafik-    *
*                       bildschirm des Maustreibers                    *
***********************************************************************/

static void MouMovePtr( BYTE col, BYTE row )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */


 regs.x.ax = 0x0004;           /* Fktnr. "Set mouse pointer position" */
 regs.x.cx = COLTOX( moucol = col );  /* Koordinaten umrechnen und in */
 regs.x.dx = ROWTOY( mourow = row );  /* globalen Variablen speichern */
 MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */


 old_win_col = moucol;
 old_win_row = mourow;
}


/***********************************************************************
*  Funktion         : M o u S e t D e f a u l t P t r                  *
**--------------------------------------------------------------------**
*  Aufgabe          : Definiert das Erscheinungsbild des Maus-Cursors  *
*                     fr Bildschirmbereiche, die nicht mit Hilfe von  *
*                     MouDefBereich definiert wurden.                  *
*  Eingabe-Parameter: STANDARD = Bit-Maske fr den Standard-Maus-Cursor*
*  Return-Wert      : keiner                                           *
***********************************************************************/

void MouSetDefaultPtr( PTRVIEW standard )
{
 stdptr = standard;              /* Bit-Maske in globaler Var. merken */

 /*-- befindet sich die Maus im Augenblick in keinem Bereich, wird ---*/
 /*-- direkt auf das neue Erscheinungsbild umgeschaltet            ---*/


  MouDefinePtr( standard );                                   /* Nein */
}

/***********************************************************************
*  Funktion         : M o u E n d                                      *
**--------------------------------------------------------------------**
*  Aufgabe          : Beendet die Arbeit mit den Funktionen des Mousec-*
*                     Moduls.                                          *
*  Eingabe-Parameter: keine                                            *
*  Return-Wert      : keiner                                           *
*  Info             : Die Funktion wird bei der Beendigung eines Pro-  *
*                     gramms automatisch aufgerufen, sofern zuvor      *
*                     MouInstall aufgerufen wurde.                     *
***********************************************************************/

void MouEnd( void )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 MouHideMouse();              /* Maus-Cursor vom Bildschirm entfernen */
 regs.x.ax = 0;                            /* Reset des Maus-Treibers */
 MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */

}

/***********************************************************************
*  Funktion         : M o u I n i t                                    *
**--------------------------------------------------------------------**
*  Aufgabe          : Leitet die Arbeit mit dem Mousec-Modul ein und   *
*                     initialisiert die verschiedenen Variablen        *
*  Eingabe-Parameter: Spalten, = die Auflsung des Textbildschirms     *
*                     Zeilen                                           *
*  Return-Wert      : TRUE, wenn eine Maus installiert ist,sonst FALSE *
*  Info             : Diese Funktion mu als erste Funktion aus diesem *
*                     Modul aufgerufen werden.                         *
***********************************************************************/

BYTE MouInit( BYTE spalten, BYTE zeilen )
{
 union REGS regs;                /* Prozessorregs fr Interruptaufruf */

 tline = zeilen;                     /* Anzahl Zeilen- und Spalten in */
 tcol  = spalten;                    /* globalen Variablen speichern  */

/* atexit( MouEnd ); */              /* am Programmende MouEnd aufrufen */

 /*-- Puffer fr Maus-Bereiche allokieren und fllen -----------------*/


 regs.x.ax = 0;                       /* Mouse-Treiber initialisieren */
 MOUINT(regs, regs);                         /* Maus-Treiber aufrufen */
 if ( regs.x.ax != 0xffff )              /* Maus-Treiber installiert? */
  return FALSE;                                               /* Nein */

 MouSetMoveAreaAll();                      /* Bewegungsbereich setzen */


 /*-- den Assembler-Event-Handler "AssHand" installieren -------------*/
 MouISetEventHandler( EV_MOU_ALL, (MOUHAPTR) AssHand );

 return mavail = TRUE;                        /* Maus ist installiert */
}


/***********************************************************************
*                   S T A R T U P    P R O G R A M M                   *
***********************************************************************/

int MouStartup (int columns, int rows)
{
   if ( MouInit((BYTE) columns, (BYTE) rows) ) /* Mouse-Modul initialisieren */
   {                 /* alles o.k., es ist ein Maustreiber installiert */
  
      MouSetDefaultPtr( MouPtrMask( PTRSAMECHAR, PTRINVCOL) );

      return 0;                          /* OK-Code an DOS zurckliefern */
   }
   else                 /* keine Maus oder kein Maustreiber installiert */
   {
      return 1;                       /* Fehlercode an DOS zurckliefern */
   }
}  /* MouStartup */

#endif

/* -FF-  */

/* Modification History */
/* 10.12.92 - copied from disc of mentioned book */
/* 11.12.92 - a whole bunch of of modifications/adaptions for mbedit */
/* 12.12.93 - handling of history windows */
/* 14.12.93 - handling of status line 2 */
/* 26.09.94 - MouSetMoveArea(): window_flag = 4/5 (hex view/edit) */
/* 27.09.94 - type cast(s) wg. compiler warnings (microsoft c 6.0) */
/* 09.03.95 - Handling of other resolutions than 80 x 25 (not yet o.k.) */
/* 14.03.95 - more tests for higher resolutions (not yet o.k.) */
