\PEDE.XPL	20-Nov-2003
\Centipede Arcade Game
\ by Richard Ottosen, Bill Bailey and Loren Blaney

\Revision History:
\04-Nov-1981: Original ran on Apple II in text mode
\20-Nov-2003: Added graphics and modified for PC, LB.

inc	c:\cxpl\codesi;

\The 320x200 graphic screen (mode $13) is is divided into 8x8-pixel cells,
\ giving a column and line Width and Height of 40x25. All graphic objects
\ (sprites) are mapped into one (or two) of these 8x8 cells. A 40x24 play
\ field array (Field) represents all but the bottom line of the screen.

def	Width= 40, Height= 25,	\screen dimensions in 8x8 pixel cells
	Length= 12,		\initial number of segments in centipede
	\Warning: Width and Length can't be changed unless PlayRound is changed
	Tail= Length-1,		\index to centipede's last segment
	HeadTime= 30,		\delay for injecting heads near bottom
	MushIni= Width*Height/24, \initial number of mushrooms
	PlayIni= 3,		\initial number of players
	SpeedIni= 9,		\initial speed of game (60ths of a second)
	SpeedMax= 5,		\limit maximum speed after 40000 points
	SpiLimIni= Height-12;	\initial spider limit (line)

def	Right= Width-1,		\right-most (8-pixel wide) column of screen
	Center= Width/2,	\center column of screen
	Bottom= Height-1,	\bottom-most line number of screen (for scores)
	BotLimit= Bottom-1,	\lower limit of all objects on play field
	TopLimit= BotLimit-4;	\upper limit of player & upward moving centipede

def	Blank, Head, Segment,	\objects that go into play field
	Player, Shell,
	Mush, Mush3, Mush2, Mush1,
	SpiderL, SpiderR, Flea,
	Explo;

def	Esc= $1B;		\Esc key exits program

def	Black, Blue, Green, Cyan, Red, Magenta, Brown, White,	\attribute colors
	Gray, LBlue, LGreen, LCyan, LRed, LMagenta, Yellow, BWhite; \ EGA palette

int	Armed,			\flag: player is armed with shell
	Bitten,			\flag: something bit the player (he's dead)
	CpuReg,			\CPU register array for soft (BIOS) interrupts
	FleaDrop,		\flag: flea is dropping (watch out!)
	FleaMad,		\flag: flea is mad cuz it's been shot once
	FleaX, FleaY,		\flea's field coordinates
	FleaWas,		\what the flea moved on top of (restores mushes)
	FullMoon,		\timer for injecting heads near bottom
	II,			\scratch for Main
	Players,		\total number of players remaining
	PlayX, PlayY,		\player's field coordinates
	ScoreL,			\current total score (low part)
	ScoreH,			\high-order portion of score (multiples of 10000)
	HitBottom,		\flag: centipede has reached BotLimit
	HiScoreL,		\highest score in game series (low part)
	HiScoreH,		\high order portion (in 10000's)
	SegCnt,			\current number of live segments (incl. heads)
	ShellX, ShellY,		\shell's field coordinates
	SpiBias,		\pushes spider right or left (+/-1)
	SpiLimit,		\upper limit of spider (line)
	SpiX, SpiY,		\spider's field coordinates (left side)
	SpiXDir, SpiYDir,	\direction spider is currently moving
	Vol,			\flag: sound volume is on
	Wave;			\number of wave being played (starts at 0)

seg char Screen(1);		\320x200 video graphic screen ($A000)

char	Field(Width, Height-1);	\the play field

int	Seg(Length),		\centipede segment type: Head, Segment, or Blank
	SegX(Length),		\segment's (or head's) field coordinates
	SegY(Length),
	SegXDir(Length),	\X direction of a segment (left=-1, right=+1)
	SegYDir(Length);	\Y direction of a segment (up=-1, down=+1)

\sprites (8x8 pixel graphic images):
char	SpBlank(2+8*8),		\first 2 bytes hold width and height in pixels
	SpHead(8, 2+8*8),	\centipede's head (array of 8 for leg positions)
	SpSeg(8, 2+8*8),
	SpPlay(2+8*8),
	SpShell(2+8*8),
	SpMush(2+8*8),
	SpMush3(2+8*8),
	SpMush2(2+8*8),
	SpMush1(2+8*8),
	SpSpiL0(2+8*8),
	SpSpiR0(2+8*8),
	SpSpiL1(2+8*8),
	SpSpiR1(2+8*8),
	SpFlea0(2+8*8),
	SpFlea1(2+8*8),
	SpExplo0(2+8*8),
	SpExplo1(2+8*8);



func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES);	\call software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES;  \(unused arguments need not be passed)
begin
CpuReg(0):= AX;
CpuReg(1):= BX;
CpuReg(2):= CX;
CpuReg(3):= DX;
CpuReg(6):= BP;
CpuReg(9):= DS;
CpuReg(11):= ES;
Softint(Int);
return CpuReg(0);		\return AX register
end;	\CallInt



func	GetKey;			\get character from keyboard (wait if necessary)
int	SC, Ch;			\this is a low-level routine with no echo,
begin				\ no Ctrl-C, and no flashing cursor.
SC:= CallInt($16, $0000);	\function $00
Ch:= SC & $00FF;
if Ch = 0 then Ch:= -(SC>>8);	\return non-ASCII chars as negative scan code
return Ch;
end;	\GetKey

\=============================== MOUSE ROUTINES ================================

func	OpenMouse;		\initializes mouse; returns 'false' if it fails
begin				\pointer is set to center of screen but is hidden
CallInt($21, $3533);		\make sure mouse vector ($33) points to something
if CpuReg(1)=0 & CpuReg(11)=0 then return false;
return CallInt($33, $0000);	\reset mouse; return 'false' if failure
end;	\OpenMouse



func	GetMousePosition(N); \return position of specified mouse coordinate
int	N;	\0 = X coordinate; 1 = Y coordinate
\For video modes $0-$E and $13 the maximum coordinates are 639x199, minus
\ the size of the pointer. For modes $F-$12 the coordinates are the same as
\ the pixels. For 80-column text modes divide the mouse coordinates by 8 to
\ get the character cursor position.
begin
CallInt($33, $0003);
return if N then CpuReg(3) else CpuReg(2);
end;	\GetMousePosition



func	GetMouseButton;		\returns non zero if either mouse button is down
begin				\ also checks for keyboard commands
if ChkKey then				\Esc key aborts program
	case GetKey of
	 ^S,^s:	Vol:= ~Vol;		\toggle sound on/off
	  Esc:	begin
		SetVid(3);		\restore normal text mode
		port($43):= $36;	\restore normal timer mode (3)
		port($40):= $FF;	\set to maximum count
		port($40):= $FF;	\ (low byte first)
		exit;
		end
	other	[];

CallInt($33, $0003);
return CpuReg(1) & $03;
end;	\GetMouseButton



proc	MoveMouse(X, Y);	\move mouse pointer to X,Y
int	X, Y;
CallInt($33, $0004, 0, X, Y);

\===============================================================================

proc	Delay(N);		\delay N sixtieths of a second
int	N;			\(negative N does 0 delay)
int	I;
begin
for I:= 1, N do
	begin
	while port($3DA) & $08 do;		\wait for no vertical retrace
	repeat until port($3DA) & $08;		\wait for vertical retrace
	end;
end;	\Delay



func	GetTime;		\read system timer's 8253 counter
begin				\each count is 838 ns (= 1 / 1.19E6 Hz)
port($43):= 0;	\latch counter 0
return not(port($40) + port($40)<<8);
end;	\GetTime



proc	Noise(Cy, Per);		\emit a sound on the beeper speaker
int	Cy, Per;		\number of cycles, period in 10 ns intervals
int	Cnt, I;
int	T0;
begin
\Each count is 838 ns. Multiplying by 12 gives the period in 10 ns intervals.
Cnt:= 12*Per;
if Vol then
	begin
	port($61):= port($61) ! $03;		\enable speaker and timer
	port($43):= $B6;			\set channel 2 for mode 3

	port($42):= Cnt;
	port($42):= Cnt>>8;
	end;

\delay approximately the same as the original Apple II Sound intrinsic
for I:= 1, Cy do
	begin
	T0:= GetTime;
	repeat until GetTime-T0 > Cnt;
	end;

port($61):= port($61) & ~$03;			\turn off speaker
end;	\Noise

\===============================================================================

proc	IntoutDbl(High, Low);	\display double precision integer
int	High, Low;
begin
if High \#0\ then
	begin
	Intout(6, High);
	if Low <1000 then Chout(6, ^0);
	if Low <100 then Chout(6, ^0);
	if Low <10 then Chout(6, ^0);
	end;
Intout(6, Low);
end;	\IntoutDbl



proc	CycleColors;		\cycle 32 shades of 7 colors
int	I, J,
	R, G, B,
	R0, G0, B0;
begin
for I:= 0, 31 do				\for 32 shades...
	begin
	port($3C7):= I+32;			\save first color
	R0:= port($3C9);
	G0:= port($3C9);
	B0:= port($3C9);

	for J:= 1, 6 do				\shift 6 colors down
		begin
		port($3C7):= I+(J+1)*32;
		R:= port($3C9);
		G:= port($3C9);
		B:= port($3C9);

		port($3C8):= I+J*32;
		port($3C9):= R;
		port($3C9):= G;
		port($3C9):= B;
		end;

	port($3C8):= I+7*32;			\set last color to first color
	port($3C9):= R0;
	port($3C9):= G0;
	port($3C9):= B0;
	end;
end;	\CycleColors



proc	DrawSprite(X0, Y0, S, Mirror);
\draw a sprite onto the 320x200 graphic screen
int	X0, Y0;		\pixel coordinates of upper-left corner of sprite
char	S;		\address of sprite data
int	Mirror;		\flag: mirror image: reverse left and right
int	X, Y, K, Y320, W, H;
begin
W:= S(0);		\get width and height (in pixels)
H:= S(1);
K:= 2;
for Y:= Y0, Y0+H-1 do
	begin
	Y320:= Y * 320;
	if Mirror then 
	    for X:= -(X0+W-1), -X0 do
		begin
		Screen(0, Y320-X):= S(K);
		K:= K + 1;
		end
	else
	    for X:= X0, X0+W-1 do
		begin
		Screen(0, Y320+X):= S(K);
		K:= K + 1;
		end;
	end;
end;	\DrawSprite



proc	DrawObj(X, Y, Obj, Ex);	\draw object given Field coordinates
int	X, Y,		\8x8 pixel cell coordinates
	Obj,		\object to draw
	Ex;		\extra modifier (segment direction, etc.)
int	Mirror;
char	Sp;
begin
Mirror:= false;
case Obj of
  Head:		[Sp:= addr SpHead(Ran(8), 0);   Mirror:= Ex];
  Segment:	[Sp:= addr SpSeg(Ran(8), 0);    Mirror:= Ex];
  Player:	Sp:= SpPlay;
  Shell:	Sp:= SpShell;
  Mush:		Sp:= SpMush;
  Mush3:	Sp:= SpMush3;
  Mush2:	Sp:= SpMush2;
  Mush1:	Sp:= SpMush1;
  SpiderL:	Sp:= if Ex then SpSpiL0 else SpSpiL1;
  SpiderR:	Sp:= if Ex then SpSpiR0 else SpSpiR1;
  Flea:		[Sp:= if Y&1 then SpFlea0 else SpFlea1;   Mirror:= Ex];
  Explo:	Sp:= if Ran(2) then SpExplo0 else SpExplo1
other	Sp:= SpBlank;

DrawSprite(X<<3, Y<<3, Sp, Mirror);
end;	\DS

\-------------------------------------------------------------------------------

proc	ShowPlayers;		\show the waiting players
int	I;
begin
for I:= 2, 12 do
	DrawObj(I+1, Bottom, if I<=Players then Player else Blank, 0)
end;	\ShowPlayers



proc	SetUp(X, Y, Obj, Ex);	\set object into Field array and display it
int	X, Y, Obj, Ex;
begin
if X>=0 & X<=Right & Y>=0 & Y<Height then
	begin			\X,Y coordinate is on the screen
	Field(X, Y):= Obj;
	DrawObj(X, Y, Obj, Ex);
	end;
end;	\SetUp

\-------------------------------------------------------------------------------

proc	StartSpider;		\start the spider off the screen
begin
SpiBias:= Ran(2);		\randomly pick left or right direction of motion
if SpiBias = 0 then SpiBias:= -1;	\+1 or -1
SpiX:= if SpiBias < 0 then Right+Width/2 else -Width/2;
SpiXDir:= SpiBias;
end;	\StartSpider

\-------------------------------------------------------------------------------

proc	BumpScore(Points);	\increment score by Points and display it
int	Points;
int	I;
begin
ScoreL:= ScoreL + Points;
if ScoreL >= 10000 then
	begin
	ScoreH:= ScoreH + 1;
	ScoreL:= ScoreL - 10000;
	Players:= Players + 1;
	ShowPlayers;
	Delay(2);	\pause between last score's sound and new player sound
	for I:= 0, 1 do [Noise(25, 191);   Noise(25, 143)];
	end;
Cursor(Width-18, Bottom);   Attrib(Yellow);   IntoutDbl(ScoreH, ScoreL);
end;	\BumpScore

\-------------------------------------------------------------------------------

proc	ScoreHit;		\bump score if shell hit something
int	I, Points;



proc	FlashNum(X, Y, N);	\flash number of points (N) on screen
int	X, Y, N;
int	J;
begin
if X > Right-2 then X:= Right -2;		\don't go off edge (spider)
if X < 0 then X:= 0;
Cursor(X, Y);   Attrib(White);   Intout(6, N);
for J:= 100, 150 do Noise(3, Ran(J));		\make splat sound
Delay(6);
SetUp(X,   Y, Blank, 0);	\remove anything the score may have overwritten
SetUp(X+1, Y, Blank, 0);
SetUp(X+2, Y, Blank, 0);
end;	\FlashNum



proc	ScoreSeg;		\hit a segment or head
int	J;
begin
\find the hit segment (or head):
loop for J:= 0, Tail do
	if ShellX=SegX(J) & ShellY=SegY(J) & Seg(J)#Blank then quit;
SetUp(SegX(J), SegY(J), Blank, 0);		\remove it
if Seg(J) = Head then SetUp(SegX(J)+SegXDir(J), SegY(J), Mush, 0)
	\(make it a mushroom shifted over one position)
else SetUp(SegX(J), SegY(J), Mush, 0);
Seg(J):= Blank;
\grow a new head at the next segment:
if J<Tail & Seg(J+1)=Segment then Seg(J+1):= Head;
SegCnt:= SegCnt -1;
end;	\ScoreSeg



begin	\ScoreHit
Points:= 0;			\default points
case Field(ShellX, ShellY) of
  Mush:		begin
    		for I:= 2, 50 do Noise(2, Ran(I));
		SetUp(ShellX, ShellY, Mush3, 0);
		end;
  Mush3:	begin
    		for I:= 2, 50 do Noise(2, Ran(I));
		SetUp(ShellX, ShellY, Mush2, 0);
		end;
  Mush2:	begin
    		for I:= 2, 50 do Noise(2, Ran(I));
		SetUp(ShellX, ShellY, Mush1, 0);
		end;
  Mush1:	begin
		Points:= 1;
    		for I:= 2, 50 do Noise(2, Ran(I));
		SetUp(ShellX, ShellY, Blank, 0);
		end;
  Segment:	begin
		Points:= 10;
    		for I:= 50, 100 do Noise(3, Ran(I));
		ScoreSeg;
		end;
  Head:		begin
		Points:= 100;
    		for I:= 50, 100 do Noise(3, Ran(I));
		ScoreSeg;
		end;
  SpiderL, SpiderR:
		begin
		case (PlayY-SpiY+1) >> 1 of
		  1:	Points:= 900;
		  2:	Points:= 600
		other	Points:= 300;
    		FlashNum(SpiX, SpiY, Points);
		SetUp(ShellX-1, ShellY, Blank, 0);	\remove dead spider
		SetUp(ShellX, ShellY, Blank, 0);
		SetUp(ShellX+1, ShellY, Blank, 0);
		StartSpider;
		end;
  Flea:		begin
		if FleaMad then
			begin
			Points:= 200;
	    		for I:= 100, 150 do Noise(3, Ran(I));
			SetUp(ShellX, ShellY, Blank, 0);
			FleaDrop:= false;
			end
		else	FleaMad:= true;
		end
other		[];
BumpScore(Points);
end;	\ScoreHit

\-------------------------------------------------------------------------------

proc	KillPlayer;		\something bit the player (i.e. you're dead)
int	I, J, X, Y;
begin
for J:= 0, 8 do			\you're dead, player (awful noise)
	begin			\make sure player, and possibly spider is gone
	SetUp(PlayX-1, PlayY, Explo, 0);
	SetUp(PlayX,   PlayY, Explo, 0);
	SetUp(PlayX+1, PlayY, Explo, 0);
	for I:= 0, 10 do Noise(1, Ran(J*150));
	end;
SetUp(PlayX-1, PlayY, Blank, 0); \remove dead player from field
SetUp(PlayX,   PlayY, Blank, 0);
SetUp(PlayX+1, PlayY, Blank, 0);
Players:= Players -1;		 \one less player

\restore shot-up mushrooms:
for X:= 0, Right do
    for Y:= 0, BotLimit do
	case Field(X, Y) of
	  Mush1, Mush2, Mush3:
		begin
		BumpScore(5);
		SetUp(X, Y, Mush, 0);
    		for I:= 2, 3 do Noise(2, Ran(I));
		Delay(12);
		end
	other	[];

Delay(40);			\pause for effect
end;	\KillPlayer

\-------------------------------------------------------------------------------

proc	MoveFlea;		\move the flea
int	X, Y, I;
begin
if Wave = 0 then return;	\(give the guy a break to start with)
if FleaDrop then
	begin
	if FleaY < BotLimit then
		begin
		if FleaWas = Blank then		\leave mushrooms in wake
		    SetUp(FleaX, FleaY, if Ran(Height/6) then Blank else Mush, 0)
		\replace what the flea hopped on except if it is a moving object
		else	case FleaWas of
			  Head, Segment, Player, Shell, SpiderL, SpiderR:
				SetUp(FleaX, FleaY, Blank, 0)
			other SetUp(FleaX, FleaY, FleaWas, 0);
		FleaY:= FleaY + 1;
		FleaWas:= Field(FleaX, FleaY);	\save what the flea is on
		SetUp(FleaX, FleaY, Flea, FleaX<Center);
		if FleaWas = Player then Bitten:= true
		else if FleaWas = Shell then 
			[ScoreHit;   Armed:= true];
		if FleaMad then Noise(5, FleaY*3+20)
		else		Noise(2, FleaY*30+200);
		end
	else	begin
		FleaDrop:= false;		\hit bottom
		SetUp(FleaX, FleaY, Blank, 0);	\eradicate flea
		end;
	end
else	begin
	\count the objects (mushrooms) on the bottom 5 rows
	I:= 0;
	for Y:= TopLimit, BotLimit do
	    for X:= 0, Right do
		if Field(X,Y) # Blank then I:= I + 1;
	\if the objects (mushrooms) are less than threshold then drop the flea
	if I < 5+Wave/2 then
		begin
		FleaDrop:= true;
		FleaMad:= false;
		FleaX:= Ran(Width);   FleaY:= 0;
		FleaWas:= Field(FleaX, FleaY);
		end;
	end;
end;	\MoveFlea

\-------------------------------------------------------------------------------

proc	MoveSpider;		\bounce the spider around
int	X, Y;
begin
X:= SpiX;   Y:= SpiY;

if X<=0 ! X>=Right then				\spider enters at the top line
	[Y:= SpiLimit;   SpiXDir:= SpiBias;   SpiYDir:= 1];

if Ran(10) = 0 then SpiXDir:= Ran(3)-1+SpiBias;	\10% of the time change horz dir
if SpiXDir < 0 then X:= SpiX -1		  	\1/3 of time move horizontally
else if SpiXDir > 0 then X:= SpiX + 1;		\2 1 0   1 0-1   0-1-2   2 1 0

if Ran(10) = 0 then SpiYDir:= Ran(5) - 2;	\10% of the time change vert dir
if SpiYDir <= 0 then Y:= Y -1			\SpiYDir = -2, -1, 0, 1, or 2
else if SpiYDir > 0 then Y:= Y + 1;

if Y > BotLimit then
	[Y:= BotLimit -1;   SpiYDir:= -SpiYDir]
else if Y < SpiLimit then
	[Y:= SpiLimit + 1;   SpiYDir:= 1];

\erase what the spider was on
\(spider eats mushrooms; moving objects restore themselves)
SetUp(SpiX,   SpiY, Blank, 0);
SetUp(SpiX+1, SpiY, Blank, 0);

SpiX:= X;   SpiY:= Y;				\actually move spider

SetUp(SpiX, SpiY, SpiderL, SpiYDir<=0);
SetUp(SpiX+1, SpiY, SpiderR, SpiYDir<=0);

if SpiX+1>=0 & SpiX<=Right then
	Noise(2, (SpiY-SpiLimit)*10+100);

if SpiY = PlayY then
    if SpiX=PlayX ! SpiX+1=PlayX then
	Bitten:= true;

\if spider went 1/2 way off the screen then reverse its direction
if SpiX > Right+Width/2 then SpiBias:= -1
else if SpiX < -Width/2 then SpiBias:= 1;
end;	\MoveSpider

\-------------------------------------------------------------------------------

proc	InjectHead;		\inject a new head at bottom to prevent boredom
int	I, Zombie;
begin
if FullMoon > HeadTime then 		\the zombies emerge
    begin
    \if a head reached the bottom (and is going back up) then inject a head.
    for I:= 0, Tail do
        if SegYDir(I)<0 & Seg(I)#Blank then HitBottom:= true;
    Zombie:= -1;			\start with no dead bodies
    if HitBottom then
	for I:= 0, Tail do		\find a dead segment to revive
	    if Seg(I) = Blank then Zombie:= I;
    if Zombie >= 0 then			\we found a live one (sort of)
	begin
	Seg(Zombie):= Head;
	SegXDir(Zombie):= if Ran(2) then 1 else -1;
	SegYDir(Zombie):= 1;
	SegX(Zombie):= if SegXDir(Zombie) = 1 then 0 else Right;
	SegY(Zombie):= TopLimit -1;
	SegCnt:= SegCnt + 1;		\the zombie flesh lives...
	end;
    FullMoon:= 0;			\now for a period of darkness
    end
else FullMoon:= FullMoon + 1;
end;	\InjectHead

\-------------------------------------------------------------------------------

proc	MoveCent;		\move centipede
int	I, X0, Y0, X1, Y1;


	proc	Turn;		\turn the segment down (or up) and around
	begin
	\move the segment down (or up) one row and reverse horizontal direction
	Y1:= Y0 + SegYDir(I);			\make tenative vertical move
	if Y1 > BotLimit then SegYDir(I):= -1	\if too far then reverse dir
	else if Y1 < TopLimit then SegYDir(I):= +1;
	SegY(I):= Y0 + SegYDir(I);		\make actual vertical move
	SegXDir(I):= -SegXDir(I);
	end;	\Turn


begin	\MoveCent
for I:= 0, Tail do
    begin
    if Seg(I) # Blank then			\don't move dead segments
	begin
	X0:= SegX(I);   Y0:= SegY(I);		\get initial position
	X1:= X0 + SegXDir(I);			\make tenative horizontal move

	if X1 < 0 then Turn			\if screen limit then turn
	else if X1 > Right then Turn
	else	case Field(X1, Y0) of		\bounce segment off mushrooms
		  Blank, Player, Shell, SpiderL, SpiderR, Flea:
		  	SegX(I):= X1		\ and other segments
		other Turn;

	SetUp(X0, Y0, Blank, 0);		\remove segment from old location
	SetUp(SegX(I), SegY(I), Seg(I), SegXDir(I)<0); \ & display it in new loc

	if SegX(I)=PlayX & SegY(I)=PlayY then Bitten:= true;
	end;
    end;
end;	\MoveCent

\-------------------------------------------------------------------------------

proc	MoveShell;		\move the shell if it has been fired
begin
if Armed then
	begin
	if GetMouseButton then			\fire!
		begin
		ShellX:= PlayX;   ShellY:= PlayY-1;
		Armed:= false;
		if Field(ShellX, ShellY) # Blank then
			[ScoreHit;   Armed:= true]
		else SetUp(ShellX, ShellY, Shell, 0);
		end;
	end
else	begin					\shell is in the air
	ScoreHit;				\did something move into shell?
	SetUp(ShellX, ShellY, Blank, 0);	\remove shell from old location
	if ShellY < 1 then Armed:= true		\off top of screen--missed
	else	begin
		ShellY:= ShellY -1;		\move shell up
		if Field(ShellX, ShellY) # Blank then		\hit something
			[ScoreHit;   Armed:= true]
		else SetUp(ShellX, ShellY, Shell, 0);
		end;
	end;
end;	\MoveShell

\-------------------------------------------------------------------------------

proc	MovePlayer;		\move the player
int	MX, MY, X, Y, S;
begin
if Bitten then return;		\don't move bitten player

\make tenative move of one character cell toward mouse pointer:
MX:= GetMousePosition(0) *Width /640;		\(0 thru Width-1)
X:= PlayX;					\make tenative move toward mouse
if MX>PlayX & PlayX<Right then X:= PlayX+1;
if MX<PlayX & PlayX>0     then X:= PlayX-1;

MY:= TopLimit + GetMousePosition(1)*5/200;	\(0 thru 4)
Y:= PlayY;					\BotLimit-TopLimit+1 = 5
if MY>PlayY & PlayY<BotLimit then Y:= PlayY+1;
if MY<PlayY & PlayY>TopLimit then Y:= PlayY-1;

S:= Field(X, Y);				\get contents of target location
if S=Head ! S=Segment ! S=SpiderL ! S=SpiderR ! S=Flea then Bitten:= true;
if S#Mush & S#Mush1 & S#Mush2 & S#Mush3 then	\don't move over mushrooms
	begin
	SetUp(PlayX, PlayY, Blank, 0);		\remove player from old position
	SetUp(X, Y, Player, 0);			\display player in new position
	PlayX:= X;   PlayY:= Y;			\save player's coordinates
	end
else	MoveMouse(PlayX*16, (PlayY-TopLimit)*40);
   \move mouse position to X,Y to remove springiness (assumes Bottom = Height-1)
end;	\MovePlayer

\-------------------------------------------------------------------------------

proc	PlayRound;		\play one round (until player is bitten & dies)
int	SegXTbl, SegXDirTbl,
	I, Looper,		\makes some things happen faster than others
	ModWave;		\wave number, modulo 12 (0..11)
begin
\initialize centipede segment arrays:     Table(ModWave, Segment)
\ (assumes Length = 12 and Width = 40)
\		Segment (incl. head) -->		      ModWave
SegXTbl:=	\					      -------
      [ [28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39],	\0
	[27, 10,  9,  8,  7,  6,  5,  4,  3,  2,  1,  0],	\1
	[10, 18, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39],	\2
	[35, 24, 12,  8,  7,  6,  5,  4,  3,  2,  1,  0],	\3
	[ 8, 17, 24, 29, 32, 33, 34, 35, 36, 37, 38, 39],	\4
	[34, 27, 20, 14,  9,  6,  5,  4,  3,  2,  1,  0],	\5
	[ 5, 12, 19, 24, 27, 32, 34, 35, 36, 37, 38, 39],	\6
	[34, 28, 23, 18, 15, 10,  7,  4,  3,  2,  1,  0],	\7
	[ 0,  5, 10, 15, 20, 24, 28, 32, 36, 37, 38, 39],	\8
	[37, 32, 27, 23, 19, 15, 11,  8,  5,  2,  1,  0],	\9
	[ 0,  4,  8, 11, 14, 17, 22, 27, 30, 34, 38, 39],	\10
	[39, 34, 30, 27, 24, 20, 18, 14, 10,  7,  3,  0] ];	\11
SegXDirTbl:=
      [ [-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],	\0
	[-1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1],	\1
	[-1,  1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1],	\2
	[-1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1],	\3
	[ 1, -1, -1,  1, -1, -1, -1, -1, -1, -1, -1, -1],	\4
	[ 1,  1, -1,  1,  1,  1,  1,  1,  1,  1,  1,  1],	\5
	[ 1, -1,  1, -1,  1, -1, -1, -1, -1, -1, -1, -1],	\6
	[ 1, -1, -1,  1,  1,  1, -1, -1,  1,  1,  1,  1],	\7
	[ 1,  1,  1, -1, -1, -1,  1, -1, -1, -1, -1, -1],	\8
	[ 1,  1,  1,  1, -1,  1,  1, -1, -1,  1,  1,  1],	\9
	[ 1,  1, -1, -1, -1,  1,  1,  1, -1,  1, -1, -1],	\10
	[-1, -1, -1, -1, -1, -1, -1,  1,  1,  1,  1,  1] ];	\11

ShowPlayers;					\show remaining players

PlayX:= Center;   PlayY:= BotLimit;		\set up player
MoveMouse(PlayX*16, (PlayY-TopLimit)*40);
\move mouse position to remove springiness (assumes Bottom = Height-1)

SetUp(PlayX, PlayY, Player, 0);
Bitten:= false;
Armed:= true;

StartSpider;
SpiLimit:= SpiLimIni;

FleaDrop:= false;				\don't drop flea yet

repeat	begin					\repeat until player is bitten
	Cursor(0, Bottom);   Attrib(Yellow);   Intout(6, Wave+1);

	\hatch centipede
	ModWave:= Rem(Wave/12);
	for I:= 0, Tail do
		begin
		SegX(I):= SegXTbl(ModWave, I);
		SegY(I):= 0;				\top row
		SegXDir(I):= SegXDirTbl(ModWave, I);
		SegYDir(I):= 1;				\move downward
		Seg(I):= if I <= ModWave then Head else Segment;
		end;
	for I:= 0, Tail do SetUp(SegX(I), SegY(I), Seg(I), SegXDir(I)<0);
	SegCnt:= Length;
	HitBottom:= false;
	FullMoon:= 0;

	loop	begin			\loop until centipede is dead
		SpiLimit:= SpiLimIni + ScoreH;
		if SpiLimit > TopLimit then SpiLimit:= TopLimit;

		I:= SpeedIni - ScoreH;	\speed up game as score increases
		if I < SpeedMax then I:= SpeedMax;
		for Looper:= 0, I do
			begin		\this loop is run 60 times per second
			Delay(1);	\it's the surrounding loop that varies
			MovePlayer;	\ in speed according to ScoreH
			MoveShell;
			if Bitten then [KillPlayer;   quit];
			if SegCnt <= 0 then
				[Wave:= Wave+1;   CycleColors;   quit];
			if Looper = I/2 then
				begin
				if SegCnt = 1 then MoveCent;
				if FleaMad then MoveFlea;
				if ScoreH then MoveSpider;
				end;
			end;
		MoveCent;
		InjectHead;
		MoveSpider;
		MoveFlea;
		end;
	end;
until Bitten;

\clean up the carnage:
for I:= 0, Tail do				\remove centipede
    if Seg(I) # Blank then
	SetUp(SegX(I), SegY(I), Blank, 0);

if not Armed then				\remove shell from flight
	SetUp(ShellX, ShellY, Blank, 0);

if FleaDrop then				\remove flea
	SetUp(FleaX, FleaY, Blank, 0);

SetUp(SpiX, SpiY, Blank, 0);			\remove spider
SetUp(SpiX+1, SpiY, Blank, 0);

SetUp(PlayX, PlayY, Blank, 0);			\remove player
end;	\PlayRound

\-------------------------------------------------------------------------------

proc	LoadSprites;		\Load sprite images from .BMP file
int	Han,			\file handle
	Width, Height,		\dimensions of entire image in .BMP file, pixels
	R, G, B,		\red, green, blue
	X, Y,			\screen coordinates (pixels)
	Y320,			\Y * 320
	I, T;			\index and tempory scratch


	proc	LoadSp(X0, Y0, W, H, Sp);	\Load a sprite
	int	X0, Y0,		\coordinates in Screen to get sprite from
		W, H;		\width and height (pixels)
	char	Sp;		\sprite to load
	int	K, X, Y;
	begin
	Sp(0):= W;
	Sp(1):= H;
	K:= 2;
	for Y:= Y0, Y0+H-1 do
	    for X:= X0, X0+W-1 do
		begin
		Sp(K):= Screen(0, X+Y*320);
		K:= K + 1;
		end;
	end;	\LoadSp


begin	\LoadSprites
\read in a 256-color .BMP file
Trap(false);			\don't trap "file not found" error; we'll do it
Han:= FOpen("PEDE.BMP", 0);	\open file for input
if Geterr \#0\ then
	begin
	SetVid(3);
	Text(0, "PEDE.BMP file not found.
WinZip users: Try CheckOut.
");
	exit;
	end;
Trap(true);			\turn traps back on
FSet(Han, ^I);			\set device 3 to handle
Openi(3);

for Y:= 0, 17 do X:= Chin(3);	\skip unused header info
Width:= Chin(3) + Chin(3)<<8;	\0..32764 (ample range)
for Y:= 0, 2-1 do X:= Chin(3);	\skip
Height:= Chin(3) + Chin(3)<<8;	\0..32767 (ample range)
for Y:= 24, 53 do X:= Chin(3);	\skip

port($3C8):= 0;			\set color registers
for I:= 0, 255 do
	begin
	B:= Chin(3)>>2;
	G:= Chin(3)>>2;
	R:= Chin(3)>>2;
	T:= Chin(3);
	port($3C9):= R;
	port($3C9):= G;
	port($3C9):= B;
	end;

\load .BMP image onto the video screen
for Y:= -(Height-1), 0 do	\.BMP files are upside down!
	begin
	Y320:= 320 * -Y;
	for X:= 0, Width-1 do
		Screen(0, X+Y320):= Chin(3);
	end;
FClose(Han);			\close handle so it can be used again

\			 X,  Y, W, H, Sprite
for I:= 0, 7 do LoadSp(8*I,  0, 8, 8, addr SpSeg(I, 0));
for I:= 0, 7 do LoadSp(8*I,  8, 8, 8, addr SpHead(I, 0));
LoadSp(32, 24, 8, 8, SpBlank);
LoadSp( 0, 16, 8, 8, SpPlay);
LoadSp( 8, 16, 8, 8, SpShell);
LoadSp( 0, 24, 8, 8, SpMush);
LoadSp( 8, 24, 8, 8, SpMush3);
LoadSp(16, 24, 8, 8, SpMush2);
LoadSp(24, 24, 8, 8, SpMush1);
LoadSp( 0, 32, 8, 8, SpSpiL0);
LoadSp( 8, 32, 8, 8, SpSpiR0);
LoadSp(16, 32, 8, 8, SpSpiL1);
LoadSp(24, 32, 8, 8, SpSpiR1);
LoadSp( 0, 40, 8, 8, SpFlea0);
LoadSp( 8, 40, 8, 8, SpFlea1);
LoadSp( 0, 48, 8, 8, SpExplo0);
LoadSp( 8, 48, 8, 8, SpExplo1);
end;	\LoadSprites

\-------------------------------------------------------------------------------

proc	PlayGame;		\play one game (until all players die)
int	X, Y, I;
begin
LoadSprites;			\load sprites and set up color registers

\show introductory screen:
Clear;
X:= Center -15;   Y:= Height/2 -9;
Attrib(Yellow);
Cursor(Center-16, Y+8);    Text(6, " New player every 10000 points ");
Attrib(LRed);
Cursor(Center-16, Y+11);   Text(6, "     Hit Esc to exit game      ");
Attrib(LCyan);
Cursor(Center-16, Y+13);   Text(6, "     S key controls sound      ");
Attrib(LGreen);
Cursor(Center-16, Y+15);   Text(6, "     Click mouse to start      ");

\show results of previous game:
Cursor(5, Bottom-1);   Attrib(White);   Text(6, "Wave");
Cursor(5, Bottom);   Attrib(Yellow);   Intout(6, Wave+1);
Cursor(Width-23, Bottom-1);   Attrib(White);   Text(6, "Score ");
Cursor(Width-23, Bottom);     Attrib(Yellow);   IntoutDbl(ScoreH, ScoreL);
Cursor(Width-10, Bottom-1);   Attrib(White);   Text(6, "High ");
Cursor(Width-10, Bottom);     Attrib(Yellow);   IntoutDbl(HiScoreH, HiScoreL);

loop	begin			\wiggle the cute little centipede feet
	for I:=  0, 12 do DrawObj(X+I, Y, Segment, 0);
	DrawObj(X+13, Y, Head, 0);
	DrawObj(X+15, Y, Head, 1);
	for I:= 16, 28 do DrawObj(X+I, Y, Segment, 1);

	DrawObj(X, Y+1, Segment, 0);   DrawObj(X+28, Y+1, Segment, 1);
	DrawObj(X, Y+2, Segment, 0);
	Cursor(X+1, Y+2);   Attrib(BWhite);
	Text(6, "  P E D E I C I D E   1.0  ");
	DrawObj(X+28, Y+2, Segment, 1);
	DrawObj(X, Y+3, Segment, 0);   DrawObj(X+28, Y+3, Segment, 1);

	for I:=  0, 13 do DrawObj(X+I, Y+4, Segment, 0);
	for I:= 14, 28 do DrawObj(X+I, Y+4, Segment, 1);

	for I:= 0, 9 do
		begin
		Delay(1);
		if GetMouseButton then quit;	\click mouse...
		end;
	end;

\unveil screen full of mushooms:
for Y:= 0, BotLimit-1 do
    begin
    for X:= 0, Width-1 do
	SetUp(X, Y, if Ran(Width*BotLimit) > MushIni then Blank else Mush, 0);
    Delay(2);
    end;
for X:= 0, Width-1 do
    SetUp(X, BotLimit, Blank, 0);

Wave:= 0;
ScoreL:= 0;  ScoreH:= 0;
ScoreH:= 0;
Players:= PlayIni;

ShowPlayers;					\show the waiting players

\show the score:
Cursor(Width-24, Bottom);   Attrib(White);   Text(6, "Score ");
Attrib(Yellow);   IntoutDbl(ScoreH, ScoreL);
Cursor(Width-11, Bottom);   Attrib(White);   Text(6, "High ");
Attrib(Yellow);   IntoutDbl(HiScoreH, HiScoreL);

while Players > 0 do PlayRound;

if ScoreH>HiScoreH ! (ScoreH=HiScoreH & ScoreL>HiScoreL) then
	[HiScoreL:= ScoreL;   HiScoreH:= ScoreH];

for II:= 1, 12 do Noise(9, 227+19*II);		\depressing sound--you lose!
Delay(60);

while GetMouseButton do;			\wait until buttons are released
end;	\PlayGame

\===============================================================================

begin	\Main
CpuReg:= Getreg;
if not OpenMouse then
	[Text(0, "This program requires a mouse.");   exit];

Screen(0):= $A000;
SetVid($13);			\40x25 text mode

\The system timer (timer 0) is normally set to mode 3, as defined by IBM.
\ Some BIOSes, Win3.1 and WinXP have a bug that instead sets this timer to
\ mode 2. This does not change the interrupt rate, but it does change the
\ rate that the internal counter decrements. Mode 3 decrements by 2, and
\ mode 2 decrements by 1. Since the internal counter is read by this code,
\ it is essential that the rate be correct. Of course WinXP (in its infinite
\ wisdom) does not allow reprogramming the mode, so the non-standard (but
\ more logical) mode 2 is used here.

port($43):= $34;		\timer 0 mode 2
port($40):= $FF;		\set 16-bit countdown timer to maximum count
port($40):= $FF;		\ (low byte first)

Vol:= true;
Wave:= 0;			\(PlayGame shows these initial values)
ScoreL:= 0;   ScoreH:= 0;
HiScoreL:= 0;   HiScoreH:= 0;

loop PlayGame;			\(Esc key exits program)
end;	\Main
