{
    BSD 3-Clause License
    Copyright (c) 2022, Jerome Shidel
    All rights reserved.
}

{ Danger Engine based game, see https://gitlab.com/DOSx86/sge }

{$I DANGER.DEF}
{ DEFINE DEVEL}
program BlockDrop;

uses Danger;

const
	MaxVRTValue		 = 6;
	MinVRTValue		 = 3;
	MaxStepRate		 = 4;
	MaxBlock 	     = 36;
	MaxColumns		 = 14;
	MaxRows			 = 12;
	MinDrops		 = 15;
	DropGrow		 = 5;
	MaxDrops		 = 100;
	MinDropWait		 = MaxRows div 2;
	MaxDropCount     = 4;

	ScreenBackground = 253;
	PlayBackground	 = 17;
	TextBackground	 = 253;
	TextColor		 = 30;
	StatusBackground = 254;
	StatusColor 	 = 14;
	StatusFrameColor = 1;
	PreviewFrameColor= 251;
	PreviewSize		 = 3;
	PreviewRows		 = 20;

	pmNone			 = 0;
	pmStart			 = 1;
	pmPlay			 = 2;
	pmDeath			 = 3;
	pmGameOver		 = 4;
	pmDropping		 = 5;

	baNone			 = 0;
	baNormal		 = 1;

var
	Frame  		: array[0..2] of PImage;
	Blocks 		: array[0..MaxBlock] of PImage;
	NullBlock   : PImage;
	Life		: PImage;

	VRTValue	: word;
	VRTWant		: word;
	StepRate	: word;
	Level  		: integer;
	Columns		: integer;
	Drops		: integer;
	Rows		: integer;
	Lives  		: integer;
	Score  		: LongInt;
	BlockColors : integer;
	NextFall	: integer;
	PauseMode	: integer;
	DropWait	: integer;
	DropDelay   : integer;
	DropCount	: integer;
	DropLimit   : integer;
	PauseWait   : integer;
	PauseDelay	: integer;
	PlayArea	: TArea;
	TextArea	: TArea;
	PreviewArea : TArea;
	ScoreStr	: String;
	LevelStr	: String;
	LivesStr	: String;
	ElapsedStr	: String;
	PauseStr	: String;
	DeathStr	: String;
	StartStr	: String;
	GameOverStr : String;

	BlockStaged		: array [0..MaxColumns - 1, 0..MaxDrops - 1] of byte;
	BlockFalling	: array [0..MaxColumns - 1] of integer;
	BlockActive		: array [0..MaxColumns - 1] of byte;
	ColumnDone		: array [0..MaxColumns - 1] of boolean;
	BlockGrid,
	TestGrid		: array [0..MaxRows - 1, 0..MaxColumns - 1] of byte;

const
	BlockOrder : array[0..MaxBlock - 1] of byte = (
		1,2,4, 		16,34,
		17,18,20,	32,

		1,2,4,16,
		15,31,
		14,30,
		12,28,
		33,

		1,2,3,4,
		1,2,3,4,

		14,15,14,15,
		14,15,

		35,36
	);

procedure Initialize;
begin
    AppInfo.Title 	:= 'BlockDrop';
    AppInfo.Year 	:= '2022';
    AppInfo.Version := '0.1';
    Level  			:= 0;
    Lives  			:= 3;
    Score  			:= 0;
    VRTWant 		:= MinVRTValue;
    PauseMode		:= pmNone;
end;

procedure ProcessCommandLine;
var
    I, E : integer;
    Opt : String;
begin
    I := 0;
    while I < ParamCount do begin
        Inc(I);
        Opt := UCase(ParamStr(I));
        if (Opt = '/H') or (Opt = '/?') then begin
            PrintHelp;
        end;
        if UCase(ParamStr(I)) <> '/L' then begin
			Val(ParamStr(I), Level, E);
            if Level > 0 then Dec(Level);
        end;
        if Opt = '/BM' then begin
            AppInfo.Driver.Video   := 'BIOSVID.DRV';
            Inc(I);
            if UCase(ParamStr(I)) <> 'LIST' then
                Val(ParamStr(I), AppInfo.VideoMode, E)
            else
                AppInfo.VideoMode:=$ffff;
            Inc(I);
        end else if Opt = '/VGA' then begin
            AppInfo.Driver.Video   := 'VGA386.DRV'; { at present, default }
        end else if Opt = '/LOG' then begin
            {$IFDEF LOGS} Logging := True; {$ENDIF}
        end;
    end;
end;

procedure Prepare;
var
    P : PAsset;
	I : integer;
begin
	VRTValue := MinVRTValue;
    Video^.Fill(0);
    Video^.SetSync(False);
    Video^.SetBuffered(True);
    Randomize;
    for I := 0 to Sizeof(Frame) div Sizeof(PImage) - 1 do
	    Frame[I] := AssetLoadOrDie('FRAME' + ZPad(I, 3) + '.IGG', asDefault, P);
    for I := 0 to MaxBlock do
	    Blocks[I] := AssetLoadOrDie('BLOCK' + ZPad(I, 3) + '.IGG', asDefault, P);
	NullBlock   := Video^.NewImage(Blocks[0]^.Width, Blocks[0]^.Height);
	Video^.ImageFill(NullBlock, PlayBackground);
	ScoreStr 	:= RawNLS('SCORE');
	LevelStr 	:= RawNLS('LEVEL');
	LivesStr   	:= RawNLS('LIVES');
	ElapsedStr 	:= RawNLS('ELAPSED');
	PauseStr    := RawNLS('PAUSE');
	DeathStr    := RawNLS('DEATH');
	StartStr    := RawNLS('START');
	GameOverStr := RawNLS('GAMEOVER');
end;

procedure DrawTextFrame(Line, Lines : integer; Color : integer);
var
	Area : TArea;
begin
	Line := Line * (Video^.TextHeight('') + 1);
	SetArea(TextArea.Left - 1, TextArea.Top + Line - 1, TextArea.Right + 1,
		TextArea.Top + Line + (Video^.TextHeight('') + 1) * Lines, Area);
	Video^.FrameArea(Area, 1, Color, Color);
end;

procedure DrawTextLn(Line : integer; S : String; Color : integer);
var
	Area : TArea;
begin
	Line := Line * (Video^.TextHeight(S) + 1);
	SetArea(TextArea.Left, TextArea.Top + Line, TextArea.Right,
		TextArea.Top + Line + Video^.TextHeight(S), Area);
	While (Video^.TextWidth(S) > AreaWidth(TextArea)) and (Length(S) > 0) do
		S[0] := Chr(Ord(S[0])-1);
	Video^.FillArea(Area, StatusBackground);
	Video^.PutText(
		TextArea.Left + AreaWidth(TextArea) div 2 - Video^.TextWidth(S) div 2,
		Area.Top + 1, S, Color);
end;

procedure DrawText;
begin
	Video^.FillArea(TextArea, TextBackground);
	DrawTextLn(1, ScoreStr, TextColor);
	DrawTextLn(2, IntStr(Score), StatusColor);
	DrawTextFrame(1, 2, StatusFrameColor);
	DrawTextLn(4, LevelStr, TextColor);
	DrawTextLn(5, IntStr(Level), StatusColor);
	DrawTextFrame(4, 2, StatusFrameColor);
	DrawTextLn(7, LivesStr, TextColor);
	DrawTextLn(8, IntStr(Lives), StatusColor);
	DrawTextFrame(7, 2, StatusFrameColor);
end;

procedure DrawBoard;
var
	I : integer;
begin
	Video^.FillArea(PlayArea, PlayBackground);
	for I := 0 to (PlayArea.Bottom - PlayArea.Top) div Frame[1]^.Height do begin
		Video^.PutImage(Frame[1],
			PlayArea.Left - Frame[1]^.Width,
			PlayArea.Top + I * Frame[1]^.Height);
		Video^.PutImage(Frame[1],
			PlayArea.Right + 1,
			PlayArea.Top + I * Frame[1]^.Height);
	end;
	for I := 0 to (PlayArea.Right - PlayArea.Left) div Frame[0]^.Width do
		Video^.PutImage(Frame[0],
			PlayArea.Left + Frame[0]^.Width * I,
			PlayArea.Bottom + 1);
	Video^.PutImage(Frame[2],
		PlayArea.Left - Frame[2]^.Width,
		PlayArea.Bottom + 1);
	Video^.PutImage(Frame[2],
		PlayArea.Right + 1,
		PlayArea.Bottom + 1);
end;

procedure DrawPreviewArea;
var
	Area : TArea;
	X, Y, C : integer;
begin
	with PreviewArea do
		SetArea(Left - 1, Top - 1, Right + 1, Bottom + 1, Area);
	Video^.FrameArea(Area, 1, PreviewFrameColor, PreviewFrameColor);
	Video^.FillArea(PreviewArea, PlayBackground);
	for Y := 0 to PreviewRows - 1 do
		for X := 0 to Columns - 1 do begin
			with PreviewArea do
				SetArea(Left + X * PreviewSize, Bottom - Y * PreviewSize - PreviewSize + 2,
					Left + X * PreviewSize + PreviewSize - 2,
					Bottom - Y * PreviewSize, Area);
			Video^.FillArea(Area, BlockStaged[X, Y] and $0f);
		end;
end;

procedure DrawStagedRow;
var
	I : integer;
begin
	for I := 0 to Columns - 1 do
		if BlockStaged[I, 0] > 0 then
			Video^.PutImage(
				Blocks[BlockStaged[I, 0]],
				PlayArea.Left + NullBlock^.Width * I,
				PlayArea.Top - NullBlock^.Height
				)
		else { only happens in death }
		if BlockActive[I] > 0 then
			Video^.PutImage(
				Blocks[BlockActive[I]],
				PlayArea.Left + NullBlock^.Width * I,
				PlayArea.Top - NullBlock^.Height
				);
end;

procedure DrawLevel;
begin
	Video^.Fill(ScreenBackground);
	Rows	:= (GetMaxY - Frame[0]^.Height - NullBlock^.Height div 2) div
		NullBlock^.Height;
	SetArea(Frame[1]^.Width,
		GetMaxY - Frame[0]^.Height - NullBlock^.Height * Rows,
		Frame[1]^.Width + NullBlock^.Width * Columns - 1,
		GetMaxY - Frame[0]^.Height - 1,
		PlayArea);
	SetArea(GetMaxX - Video^.TextWidth('ELAPSED') - 2, 0, GetMaxX - 2, GetMaxY,
		TextArea);
	SetArea(0, 0, Columns * PreviewSize - 2, PreviewRows * PreviewSize - 1, PreviewArea);
	MoveArea((TextArea.Left - 2) div 2 - AreaWidth(PlayArea) div 2,
		PlayArea.Top, PlayArea);
	MoveArea(
		TextArea.Left + AreaWidth(TextArea) div 2 - AreaWidth(PreviewArea) div 2,
		GetMaxY - AreaHeight(PreviewArea) - Video^.TextHeight('') div 2, PreviewArea);
	DrawBoard;
	DrawText;
	DrawPreviewArea;
	DrawStagedRow;
end;

procedure FallNext;
begin
	Dec(DropWait);
	If DropWait > 0 then exit;
	DropWait := 0;
	if DropCount >= DropLimit then Exit;
	Inc(NextFall);
	if NextFall >= Columns then
		NextFall := 0;
	if ColumnDone[NextFall] then Exit;
	if BlockActive[NextFall] <> 0 then exit;
	BlockActive[NextFall] := BlockStaged[NextFall, 0];
	BlockFalling[NextFall] := - NullBlock^.Height;
	BlockStaged[NextFall,0] := 0;
	ColumnDone[NextFall] := BlockActive[NextFall] = 0;
	Inc(DropCount);
	DropWait := DropDelay;
end;

procedure ConfigLevel;
begin
	DropLimit   := (Level div 5) + 1;
	Columns 	:= 5 + Level;
	Drops   	:= MinDrops + (Level - 1) * DropGrow;
	BlockColors := 3 + (Level div 2) * 2;
	VRTValue	:= MinVRTValue + Level div 10;
	case Level of
		1 	    : StepRate := 1;
		2,3,4,5 : StepRate := 2;
	else
		StepRate := 4;
	end;
	PauseDelay  := 9 * VRTValue;
	PauseWait   := PauseDelay;
	DropDelay := MaxRows * 8 div StepRate;
	if DropLimit > MaxDropCount then DropLimit := MaxDropCount;
	if Columns > MaxColumns then Columns := MaxColumns;
	if Drops > MaxDrops then Drops := MaxDrops;
	if BlockColors > MaxBlock then BlockColors := MaxBlock;
	if VRTValue > MaxVRTValue then VRTValue := MaxVRTValue;
	if StepRate > MaxStepRate then StepRate := MaxStepRate;

	if VRTWant >= VRTValue then
		SetVRTInterval(VRTWant)
	else
		SetVRTInterval(VRTValue);
end;

procedure PauseGame(Style : integer);
const
	X : integer = 0;
	Y : Integer = 0;
	I : PImage = nil;
var
	W : integer;
	S : String;
	A : TArea;
begin
	if Style = pmDropping then exit;
	if Style = pmNone then begin
		if Assigned(I) then begin
			Video^.PutImage(I, X, Y);
			Video^.FreeImage(I);
		end;
	end else begin
		PauseWait := PauseDelay;
		case Style of
			pmStart 	: S := StartStr;
			pmPlay 		: S := PauseStr;
			pmDeath 	: S := DeathStr;
			pmGameOver 	: S := GameOverStr;
		end;
		S := '  ' + S + '  ';
		W := Video^.TextWidth(S);
		Y := GetMaxY div 2 - Video^.TextHeight(S) * 3 div 2;
		X := PlayArea.Left + AreaWidth(PlayArea) div 2 - W div 2;
		SetArea(X, Y, X + W - 1, Y + Video^.TextHeight(S) * 3 - 1, A);
		I := Video^.NewImage(W, Video^.TextHeight(S) * 3);
		Video^.GetImage(I, X, Y);
		Video^.FillArea(A, 0);
		Video^.FrameArea(A, 2, 7,8);
		Video^.PutText(X, Y + Video^.TextHeight(S), S, 15);
	end;
	PauseMode := Style;
end;

procedure StartLevel;
const
	BX : word = 0;
var
	X, Y, O, R : integer;
begin
	ConfigLevel;
	NextFall 	:= -1;
	DropCount   := 0;
	DropWait 	:= 0;

	FillChar(BlockStaged, SizeOf(BlockStaged), 0);
	FillChar(BlockFalling, SizeOf(BlockFalling), 0);
	FillChar(BlockActive, SizeOf(BlockActive), 0);
	FillChar(BlockGrid, SizeOf(BlockGrid), 0);
	FillChar(ColumnDone, SizeOf(ColumnDone), False);

	for Y := 0 to Drops - 1 do
		for X := 0 to Columns - 1 do begin
			repeat
				R := BlockOrder[Random(BlockColors)];
				O := Level + 2;
				if R = 34 then begin
					if Level < 4 then begin
						R := 16;
						O := O * 2;
					end else begin
						O := (20 - level) * 2;
						if O < 5 then O := 5;
					end;
				end;
			until (R < 16) or (Random(O) = 0);
			{ if R = 1 then R := 33; }
			BlockStaged[X, Y] := R;
		end;

	DrawLevel;
	FallNext;
	PauseMode := pmNone;
	PauseGame(pmStart);
end;

procedure FallFeed(Column : integer);
var
	Area : TArea;
begin
	if BlockStaged[Column, 1] <> 0 then begin
		Move(BlockStaged[Column, 1], BlockStaged[Column, 0], MaxDrops - 1);
		BlockStaged[Column, MaxDrops - 1] := 0;
		with PreviewArea do
			SetArea(
				Left + Column * PreviewSize, Top,
				Left + Column * PreviewSize + PreviewSize - 2, Bottom,
				Area);
		Video^.ShiftArea(Area, dmDown, PreviewSize, PlayBackground);
		Inc(Area.Top);
		Area.Bottom := Area.Top + PreviewSize - 2;

		Video^.FillArea(Area, BlockStaged[Column, PreviewRows - 1] and $0f);
	end else begin
		BlockStaged[Column, 0] := 0;
		with PreviewArea do
			SetArea(
				Left + Column * PreviewSize, Bottom - PreviewSize + 2,
				Left + Column * PreviewSize + PreviewSize - 2,
				Bottom,
				Area);
		Video^.FillArea(Area, 0);
	end;
end;

procedure NextLevel;
begin
	Inc(Level);
	StartLevel;
end;

procedure Death;
begin
	PauseWait := PauseDelay * 2;
	Dec(Lives);
	DrawStagedRow;
	if Lives <= 0 then
		PauseGame(pmGameOver)
	else
		PauseGame(pmDeath);
end;

procedure NewGame;
begin
    Level  			:= 0;
    Lives  			:= 3;
    Score  			:= 0;
    PauseMode		:= pmStart;
    NextLevel;
end;

procedure RedrawFalling(Column : integer);
var
	X, Y : integer;
begin
	if BlockActive[Column] <> 0 then begin
		X := PlayArea.Left + Column * NullBlock^.Width;
		Y := PlayArea.Top + BlockFalling[Column];
		Video^.PutImage(Blocks[BlockActive[Column]], X, Y);
	end;
end;

procedure FallController;
var
	I, X, Y, N : integer;
begin
	for I := 0 to Columns - 1 do begin
		if BlockActive[I] <> 0 then begin
			if (BlockFalling[I] < 0) and (BlockGrid[0, I] <> 0) then begin
				Death;
				Exit;
			end;
			if (BlockStaged[I,0] = 0) then FallFeed(I);
			X := PlayArea.Left + I * NullBlock^.Width;
			Y := PlayArea.Top + BlockFalling[I];
			Video^.PutImage(NullBlock, X, Y);
			if (BlockFalling[I] <= 0) and (Y > 0) then
				Video^.PutImage(Blocks[BlockStaged[I,0]],
					X, Y - NullBlock^.Height);
			Inc(Y, StepRate);
			Inc(BlockFalling[I], StepRate);
			Video^.PutImage(Blocks[BlockActive[I]], X, Y);
			N := (BlockFalling[I]) div NullBlock^.Height;
			if (Y + NullBlock^.Height > PlayArea.Bottom) or
			((BlockGrid[N + 1, I] <> 0) and (BlockFalling[I] = N * NullBlock^.Height)) then begin
				BlockGrid[N, I] := BlockActive[I];
				BlockActive[I] := 0;
				Dec(DropCount);
			end;
		end;
	end;
	FallNext;
end;

function LevelDone : boolean;
var
	I : integer;
	R : boolean;
begin
	R := True;
	for I := 0 to Columns - 1 do
		R := R and (BlockStaged[I,0] = 0) and (BlockActive[I] = 0);
	LevelDone := R;
end;

procedure BlockPop(X, Y : integer);
begin
	BlockGrid[Y, X] := 0;
	Video^.PutImage(NullBlock, X * NullBlock^.Width + PlayArea.Left,
		Y * NullBlock^.Height + PlayArea.Top);
end;

function FloodPop(X, Y, C : integer) : boolean;
var
	N : integer;
	V : LongInt;

	procedure Flood(X, Y : integer);
	begin
		if TestGrid[Y, X] >= 32 then exit;
		if TestGrid[Y, X] and $f <> C and $f then exit;
		Inc(N);
		Inc(V, TestGrid[Y, X]);
		TestGrid[Y, X] := 0;
		Flood(X - 1, Y);
		Flood(X + 1, Y);
		Flood(X, Y - 1);
		Flood(X, Y + 1);
	end;

begin
	V := 0;
	FloodPop := False;
	N := 0;
	TestGrid := BlockGrid;
	Flood(X, Y);
	if N < 3 then exit;
	Inc(Score, V + (V * Level div 2) * N);
	for X := 0 to Columns - 1 do
		for Y := 0 to Rows - 1 do
			if TestGrid[Y, X] = 0 then
				BlockPop(X, Y);

	FloodPop := True;
end;

function DropDown : boolean;
var
	D, P : boolean;
	X, Y, XX, YY : integer;
begin
	D := False;
	for Y := Rows - 1 downto 1 do begin
		for X := 0 to Columns - 1 do begin
			if (BlockGrid[Y, X] = 0) and  (BlockGrid[Y - 1, X] <> 0) then begin
				D := True;
				for YY := Y downto 0 do begin
					if YY > 0 then
						BlockGrid[YY, X] := BlockGrid[YY - 1, X]
					else
						BlockGrid[YY, X] := 0;
					if BlockGrid[YY, X] > 0 then
						Video^.PutImage(Blocks[BlockGrid[YY, X]],
							X * NullBlock^.Width + PlayArea.Left,
							YY * NullBlock^.Height + PlayArea.Top)
					else
						Video^.PutImage(NullBlock,
							X * NullBlock^.Width + PlayArea.Left,
							YY * NullBlock^.Height + PlayArea.Top);
					end;
			end;
		end;
		if D then begin
			for X := 0 to Columns - 1 do
				RedrawFalling(X);
			Break;
		end;
	end;
	DropDown := D;
end;

procedure Click(Event : TEvent);
var
	I : integer;
	X, Y, XX, YY : integer;
	C : integer;
	V : LongInt;
begin
	if not InArea(Event.Position, PlayArea) then Exit;
	X := (Event.Position.X - PlayArea.Left) div NullBlock^.Width;
	Y := (Event.Position.Y - PlayArea.Top) div NullBlock^.Height;
	if BlockGrid[Y, X] = 0 then exit;
	if BlockGrid[Y, X] = 34 then exit;
	C := BlockGrid[Y, X];
	V := C;
	if C > 15 then
		BlockPop(X, Y);
	if C = 35 then begin { insta death }
		Death;
		Exit;
	end else if C = 36 then begin { bonus life }
		Inc(Lives);
		DrawTextLn(8, IntStr(Lives), StatusColor);
	end else if C = 32 then begin { row clear }
		for I := 0 to Columns - 1 do begin
			Inc(V, BlockGrid[Y, I]);
			BlockPop(I, Y);
		end;
	end else if C = 33 then begin { column clear }
		for I := 0 to Rows - 1 do begin
			Inc(V, BlockGrid[I, X]);
			BlockPop(X, I);
		end;
	end else if C = 16 then begin { standard bomb }
		for XX := -2 to 2 do
			for YY := -2 to 2 do begin
				if (X + XX < 0) or (Y + YY < 0) or (X + XX >= Columns)
				or (Y + YY >= Rows) then Continue;
				if (Abs(XX) = 2) and (Abs(YY) = 2) then continue;
				Inc(V, BlockGrid[Y + YY, X + XX]);
				BlockPop(X + XX, Y + YY);
			end;
	end else if (C > 16) and (C < 32) then begin { color bomb }
		for XX := 0 to Columns - 1 do
			for YY := 0 to Rows - 1 do
				if (BlockGrid[YY, XX] < 32)
				and (BlockGrid[YY, XX] and $f = C and $f) then begin
					Inc(V, BlockGrid[YY, XX]);
					BlockPop(XX, YY);
				end;
	end else if (C < 16) then begin { normal block }
		if not FloodPop(X, Y, C) then exit;
		V := 0;
	end;
	Inc(Score, V + (V * Level div 2));
	DrawTextLn(2, IntStr(Score), StatusColor);
	PauseMode := pmDropping;
end;

procedure Play;
var
    Event : TEvent;
    LTT : word;
begin
    NextLevel;
    PurgeEvents;
    LTT := VRTimer^;
    MouseShow;
    repeat
        Video^.Update;
        While not GetEvent(Event) and (LTT = VRTimer^) do Idle;
        Video^.Prepare;
        if PauseWait > 0 then Dec(PauseWait);
        if MouseEvent(Event) then begin
            MouseMove(Event.Position);
            if Event.Kind and evMouseClick = evMouseClick then begin
            	if PauseMode = pmNone then
            		Click(Event)
            	else if PauseWait = 0 then
					case PauseMode of
						pmStart, pmPlay : PauseGame(pmNone);
						pmDeath 		: StartLevel;
						pmGameOver 		: NewGame;
					end;
            end;
        end;
        if Event.Kind and evKeyPress = evKeyPress then begin
            case Event.KeyCode of
            	kbSpace  : begin
            		if PauseWait = 0 then
						case PauseMode of
							pmNone 			: PauseGame(pmPlay);
							pmStart, pmPlay : PauseGame(pmNone);
							pmDeath 		: StartLevel;
							pmGameOver 		: NewGame;
						end;
            	end;
                kbEscape : Break;
			    { IFDEF DEVEL}
			    kbTab : if PauseMode = pmStart then NextLevel;
			    kbPlus, kbEqual : begin
			    	if VRTWant < MaxVRTValue then begin
			    		Inc(VRTWant);
			    		if VRTWant >= VRTValue then
					    	SetVRTInterval(VRTWant);
				    end;
			    end;
			    kbMinus, kbUnderScore : begin
			    	if VRTWant > MinVRTValue then begin
			    		Dec(VRTWant);
			    		if VRTWant >= VRTValue then
					    	SetVRTInterval(VRTWant);
				    end;
			    end;
			    { ENDIF}
            end;
        end;
        if LTT <> VRTimer^ then begin
	        LTT := VRTimer^;
	        if LTT and $3 = 0 then Video^.SpriteNextAll(False);
	        if PauseMode = pmDropping then begin
	        	 if not DropDown then begin
	        		PauseGame(pmNone);
	        	end;
	        end else
			if PauseMode = pmNone then begin
				FallController;
				if LevelDone then NextLevel;
			end;
		end;
    until False;
    MouseHide;
end;

{$I INTRO.INC}

begin
	Initialize;
    ProcessCommandLine;
    RequireMouse;
    Ignite;
    Prepare;
    {$IFNDEF DEVEL}
    PlayIntro;
    {$ENDIF}
    Video^.SetSync(False);
    Play;
    {$IFNDEF DEVEL}
    PlayExtro;
    {$ENDIF}
    Extinguish;
end.