#define LUA_COMPAT_ALL
#include "../src/lauxlib.h"



/*  Definition of 3D vector.  */
typedef struct vector_3D_t {
  double x;
  double y;
  double z;
} vector_3D;


/*  Function defining a new vector.  */
static int define(lua_State *L)
{
  vector_3D *vec3D;

  /*  Check if there are valid vector components on the stack.  */
  if (lua_isnumber(L, 1) && lua_isnumber(L, 2) && lua_isnumber(L, 3))
  {
    /*  Allocate a space for a new vector.  */
    vec3D = (vector_3D *)lua_newuserdata(L, sizeof(vector_3D));

    /*  Read vector components from stack.  */
    vec3D->x = lua_tonumber(L, 1);
    vec3D->y = lua_tonumber(L, 2);
    vec3D->z = lua_tonumber(L, 3);

    /*  Assign userdata to metatable.  */
    luaL_getmetatable(L, "meta_table.vector_3D");
    lua_setmetatable(L, -2);  

    return 1; /*  Number of elements returned on the stack.  */
  }
  else
    return luaL_error(L, "error defining new vector_3D");
}


/*  Add two vectors.  */
static int add(lua_State *L)
{
  vector_3D *sum;

  /*  Check if there are valid vectors on the stack.  */
  vector_3D *a = luaL_checkudata(L, 1, "meta_table.vector_3D");
  vector_3D *b = luaL_checkudata(L, 2, "meta_table.vector_3D");
  
  /*  Compute result */
  sum = (vector_3D *)lua_newuserdata(L, sizeof(vector_3D));
  sum->x = a->x + b->x;
  sum->y = a->y + b->y;
  sum->z = a->z + b->z;  
  
  /*  Assign userdata to metatable.  */
  luaL_getmetatable(L, "meta_table.vector_3D");
  lua_setmetatable(L, -2);  

  return 1; /*  Number of elements returned on the stack.  */
}


/*  Generate string representation of the vector.  */
static int tostring(lua_State *L)
{
  /*  Check if there are valid vector components on the stack.  */
  vector_3D *vec3D = luaL_checkudata(L, 1, "meta_table.vector_3D");
  
  /*  Push result on stack.  */
  lua_pushfstring(L, "(%f, %f, %f)", vec3D->x, vec3D->y, vec3D->z);
  
  return 1; /*  Number of elements returned on the stack.  */
}


/* Table entries that will be accessible by lua.  */
/* Array of paires of function names and function pointers.  */
static const struct luaL_Reg vector_3D_func[] = {
  {"define", define},  /*  Define new vector.  */
  {NULL, NULL}         /*  End of register.  */
};


/* Meta table entries.  */
/* Array of paires of function names and function pointers.  */
static const struct luaL_Reg vector_3D_meta[] = {
  {"__add", add},           /* Metamethode zum addieren */
  {"__tostring", tostring}, /* Metamethode um Stringreprsentation zu erzeugen */
  {NULL, NULL}              /*  End of register.  */
};


/* Will be called when library is opened.  */
int luaopen_libtest3(lua_State *L)
{
  /*  Create a meta table with name "meta_table.vector_3D" in the registery.  */
  luaL_newmetatable(L, "meta_table.vector_3D");
  
  /*  Create entries in meta table.  */
  luaL_register(L, NULL, vector_3D_meta);
  
  /*  Create entries for the vector operation table (accessible from lua).  */
  luaL_register(L, "vec3D", vector_3D_func);
  
  return 1;
}
