////////////////////////////////////////////////////////////////////////////////
//
//	Dynamic Loading System - Linker
//
//	Copyright (c) 2003 Point Mad, Lukas Lipka. All rights reserved.
//
////////////////////////////////////////////////////////////////////////////////
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <coff.h>
#include <time.h>
#include <conio.h>

#include "../include/dynld.h"
#include "../include/zlib.h"

typedef struct TResEntrie {

	char Name[32];
	unsigned long DataType;
	unsigned long DataEncode;
	unsigned long DataSize;

} TResEntrie;

typedef struct TResHead {

	unsigned long Magic;
	unsigned long FormatVersion;
	unsigned long Entries;

} TResHead;

l_bool WriteString ( FILE *f, l_text s )
{
	l_ushort length = strlen(s);

	fwrite(&length, 1, 2, f);
	fwrite(s, 1, length, f);
	return 1;
}

int main(int args, char **argv)
{
	char Cmd[2048] = { 0, 0, 0 };
	FILE *in,*out, *sc;

	FILHDR  CoffHead;
	SCNHDR *CoffSections;
	RELOC   CoffReloc;
	SYMENT *CoffSymbols;
	char   *CoffStrings;
	l_ulong CoffStringsSize;

	TDynLdHeader  DynHead;
	void        *DynData;
	TDynLdReloc   DynReloc;
	char        *DynName = "(unknown)";
	char         Compress = 0;
	unsigned char d = 0;
	char *RessourceFile = 0;


	l_ulong i;



	if ( args > 1 ) {
		unsigned char i;
		for(i=1; argv[i]; i++) {
			if (!strcmp(argv[i], "-C")) {
				Compress = 1;
				d++;
			} else if (!strncmp(argv[i], "-R",2)) {
				RessourceFile = strdup(argv[i]+2);
				d++;
			}
		}
	}



	memset(&DynHead, 0, sizeof(TDynLdHeader));


	printf("DynLd Linker [0.5.2.0]\n");
	printf("Copyright (c) Point Mad, Lukas Lipka. All rights reserved.\n");
	printf("\n");

	if ( args < 3 + d )
	{
		printf("Usage: DYNLDLNK [-C] [-Rfile] <destination> <source> [source2] [source3] [...]\n");
		printf("\t[-C]           Enable Compression (Powered by ZLIB)\n");
		printf("\t[-Rfile]       Include ressource file 'file' (DMS res files only)\n");
		printf("\t<destination>  Executable (.APP) / Library (.DL) to generate\n");
		printf("\t<sources>      COFF file/s\n");

		return 1;
	}

	/**
	*	Print compilation info
	*/
	printf("Compilation information\n");
	printf("-----------------------\n");
	printf("  Input:     %s\n", argv[2 + d]);
	printf("  Output:    %s\n", argv[1 + d]);
	if ( RessourceFile )
	printf("  Ressource: %s\n", RessourceFile);

	in = fopen(argv[2 + d], "rb");

	if ( !in )
	{
		printf("/!\\ Error: Input file do not exists.\n");

		return 1;
	}

	fread(&CoffHead, 1, FILHSZ, in);

	if ( CoffHead.f_nscns != 1 || args > 3 ) // Multiple sections... Call LD to make a common one
	{
		fclose(in);

		if ( !(sc = fopen("dynld.ld","rt") ) )
		{
			sc = fopen("dynld.ld","wt");

			if ( sc )
			{
				fprintf(sc ,"OUTPUT_FORMAT(\"coff-go32\")\n\nFORCE_COMMON_ALLOCATION\n\nSECTIONS {\n  .text : {\n\t*(.text)\n\t*(.data)\n\t*(.bss)\n\t*(COMMON)\n  }\n}\n");

				fclose(sc);
				printf("(i) Script - dynld.ld - generated.\n");
			}
			else
			{
				printf("/!\\ Error : Unable to generate dynld.ld script\n");

				return 1;
			}
		}
		else fclose(sc);

		strcpy(Cmd,"ld -X -r -o _tmp.o ");

		for(i = 2 + d; argv[i]; i++)
		{
			strcat(Cmd,argv[i]);
			strcat(Cmd," ");
		}

		strcat(Cmd," -T dynld.ld");
		printf("(i) Calling LD (linker)\n");
		// -S skipped as LD crash

		if ( system(Cmd) )
		{
			printf("/!\\ Error : LD have not return correct code...\n");

			return 1;
		}

		in = fopen("_tmp.o","rb");

		if ( !in )
		{
			printf("/!\\ Error: Temporary file do not exists.\n");

			return 1;
		}

		fread(&CoffHead, 1, FILHSZ, in);
	}

	out = fopen(argv[1+d],"wb+");

	if ( !out )
	{
		printf("/!\\ Error: Unable to open output file.\n");

		return 1;
	}

	//printf("Debug Informations\n");
	//printf("------------------\n");

	//printf("  Loading COFF Sections\n");
	// Load Coff Sections
	fseek(in, CoffHead.f_opthdr, SEEK_CUR);
	CoffSections = (SCNHDR *) malloc(SCNHSZ*CoffHead.f_nscns);
	fread(CoffSections, CoffHead.f_nscns, SCNHSZ, in);

	//printf("  Loading COFF Data\n");
	// Load Coff Data
	DynHead.Size = CoffSections[0].s_paddr+CoffSections[0].s_size;
	DynData = (void*)malloc(DynHead.Size);
	memset(DynData, 0, DynHead.Size);
	fseek(in, CoffSections[0].s_scnptr, SEEK_SET);
	fread(DynData, 1, CoffSections[0].s_size, in);

	//printf("  Loading COFF Symbols\n");
	// Load Coff Symbols
	fseek(in, CoffHead.f_symptr, SEEK_SET);
	CoffSymbols = (SYMENT *) malloc(SYMESZ*CoffHead.f_nsyms);
	fread(CoffSymbols, CoffHead.f_nsyms, SYMESZ, in);

	//printf("  Loading COFF Strings\n");
	// Load Coff Strings
	fread(&CoffStringsSize, 1, 4, in);
	CoffStrings = (char *) malloc(CoffStringsSize-4);
	memset(CoffStrings, 0, 4);
	fread(CoffStrings+4, 1, CoffStringsSize-4, in);

	//printf("  Write Output Header\n");
	// Write DynHeader
	DynHead.Magic = ULONG_ID('D','n','L','d');
	DynHead.FileFormatVersion = DYNLDVERSION;
	DynHead.Importations = 0;
	DynHead.Time = time(0);
	//DynHead.Type = DynLdTypeOld;
	DynHead.MainOffset  = NoneOffset;
	DynHead.CloseOffset = NoneOffset;
	DynHead.LibsOffset = NoneOffset;
	fwrite(&DynHead, 1, sizeof(TDynLdHeader), out);


	// Look for Key Data
	for (i=0;i<CoffHead.f_nsyms;i++)
	{
		char *name, nam8[9];

		if (CoffSymbols[i].e.e.e_zeroes)
		{
			memcpy(nam8, CoffSymbols[i].e.e_name, 8);
			nam8[8] = 0;
			name = nam8;
		}
		else name = CoffStrings+CoffSymbols[i].e.e.e_offset;
		//printf("  %d : 0x%06x : %s\n",i,CoffSymbols[i].e_value,name);

		if ( name[0] != '.' /*&& CoffSymbols[i].e_value*/ )
		{
			if ( !strcmp(name+1,"LibMain") ) DynHead.MainOffset = CoffSymbols[i].e_value;
			if ( !strcmp(name+1,"Main") )
			{
				DynHead.Type = DYNLD_TYPEAPP;
				DynHead.MainOffset = CoffSymbols[i].e_value;
				printf("  Threading App\n");
			}
			if ( !strcmp(name+1,"nUID") )  memcpy(&DynHead.UID,DynData+CoffSymbols[i].e_value,12);
			if ( !strcmp(name+1,"AppName") ) DynName = (char*)DynData+CoffSymbols[i].e_value;
			if ( !strcmp(name+1,"Close") )  DynHead.CloseOffset = CoffSymbols[i].e_value;
			if ( !strcmp(name+1,"NeededLibs") )  DynHead.LibsOffset = CoffSymbols[i].e_value;

			if ( !strcmp(name+1,"AppVersion") )  memcpy(&DynHead.FileVersion,DynData+CoffSymbols[i].e_value,4);
			//printf("  %d : 0x%06x : %s\n",i,CoffSymbols[i].e_value,name+1);
		}

    i += CoffSymbols[i].e_numaux;
  }

  if ( DynHead.MainOffset == NoneOffset ) {
	  printf("\nWARNING : No Main() is defined.\n");
	  printf("Example :\n");
    printf("#include\"kernel.h\"\n");
    printf("l_bool Main (l_text Args) {\n");
    printf("  return true;\n");
    printf("}\n");
  }

	if ( DynHead.CloseOffset == NoneOffset && DynHead.Type != DYNLD_TYPEAPP )
	{
		printf("\nWARNING : No Close() is defined.\n");
		printf("Example :\n");
		printf("#include\"kernel.h\"\n");
		printf("void Close ( void ) {\n");
		printf("}\n");
	}

	//Write Application name to file
  WriteString(out, DynName);

  //Write Data to File (after haeder and informations text)
  if ( Compress ) {
	  void *Temp;
	  DynHead.Compression = 0x01;
	  DynHead.OriginalSize = DynHead.Size;
	  DynHead.Size = DynHead.OriginalSize+DynHead.OriginalSize/5+12;
	  Temp = (void*)malloc(DynHead.Size);
	  compress(Temp,&DynHead.Size,DynData,DynHead.OriginalSize);
	  fwrite(Temp, 1, DynHead.Size, out);
	  free(Temp);
  } else
    fwrite(DynData, 1, DynHead.Size, out);


  // Write Inportaions
  for (i=0;i<CoffHead.f_nsyms;i++) {
    char *name, nam8[9];
    if (CoffSymbols[i].e.e.e_zeroes) {
      memcpy(nam8, CoffSymbols[i].e.e_name, 8);
      nam8[8] = 0;
      name = nam8;
    } else
      name = CoffStrings+CoffSymbols[i].e.e.e_offset;

    if ( name[0] != '.' && !CoffSymbols[i].e_scnum ) {
      WriteString(out, name+1);
      fwrite(&i, 1, 4, out);
      DynHead.Importations++;
      //printf("  %d : External : %s\n",i,name+1);
    }
    i += CoffSymbols[i].e_numaux;
  }

  {
    int r;
    //printf("  Relocations   Type     Addr     Symndx\n");
    fseek(in, CoffSections[0].s_relptr, SEEK_SET);
    for (r=0;r<CoffSections[0].s_nreloc;r++) {
      fread(&CoffReloc, 1, RELSZ, in);
      DynReloc.Type = CoffReloc.r_type;
      DynReloc.Address = CoffReloc.r_vaddr;
      DynReloc.Symbol = CoffReloc.r_symndx;
      fwrite(&DynReloc, 1, sizeof(TDynLdReloc), out);
      DynHead.Relocations++;
      //printf("    Relocation %d %d %d\n",CoffReloc.r_type,CoffReloc.r_vaddr,CoffReloc.r_symndx);
    }
  }

  fgetpos(out,&(DynHead.RessourceOffset));

  if ( RessourceFile ) {
  	FILE *r = fopen(RessourceFile,"rb");
	  if ( r ) {
			TResHead Head;
			unsigned long Entries;
			TResEntrie E;
			void *buffer;
			fread(&Head,sizeof(TResHead),1,r);
			if ( Head.Magic == ULONG_ID('D','M','S','R') ) {
			  if ( Head.FormatVersion == ULONG_ID(0,0,0,1) ) {
				  Entries = Head.Entries;
				  while ( Entries ) {
					  fread(&E,sizeof(TResEntrie),1,r);
					  fwrite(&E,sizeof(TResEntrie),1,out);
					  buffer = malloc(E.DataSize);
					  fread(buffer,E.DataSize,1,r);
					  fwrite(buffer,E.DataSize,1,out);
					 	Entries--;
				  }
				  DynHead.RessourceEntries = Head.Entries;
			  }
		  }
  		fclose(r);
		}
	}

  printf("\nApplication Informations\n");
  printf("------------------------\n");
  printf("  MainOffset      : 0x%08x\n",DynHead.MainOffset);
  printf("  CloseOffset     : 0x%08x\n",DynHead.CloseOffset);
  printf("  UID             : '%s'\n",&DynHead.UID);
  printf("  Version         : 0x%08x\n",DynHead.FileVersion);
  printf("  Name            : '%s'\n",DynName);
  printf("  Relocations     : %d\n",DynHead.Relocations);
  printf("  Imports         : %d\n",DynHead.Importations);
  if ( Compress ) {
  printf("  Data Size       : %d\n",DynHead.OriginalSize);
  printf("  Compressed Size : %d\n",DynHead.Size);
  } else
  printf("  Data Size       : %d\n",DynHead.Size);
  printf("  Ressource       : %d Entrie(s)\n",DynHead.RessourceEntries);
  fseek(out, 0, SEEK_SET);
  fwrite(&DynHead, 1, sizeof(TDynLdHeader), out);

  free(CoffStrings);
  free(CoffSymbols);
  free(DynData);
  free(CoffSections);


  fclose(in);
  fclose(out);
  remove("_tmp.o");

	return 0;
}
////////////////////////////////////////////////////////////////////////////////
