////////////////////////////////////////////////////////////////////////////////
//
//  Treeview Widget - Core File
//
//	(c) Copyright 2003,2004 Point Mad, Lukas Lipka. All rights reserved.
//
////////////////////////////////////////////////////////////////////////////////

#include "kernel.h"
#include "widget.h"
#include "treeview.h"
#include "textbox.h"

////////////////////////////////////////////////////////////////////////////////
l_ulong	AppVersion = ULONG_ID(0,0,0,1);
l_char	AppName[] = "Treeview Widget";
l_uid	nUID = "treeview";
l_uid NeededLibs[] = { "widget", "skin","textbox","menu","" };

PSkin SkinTreeview = 0;

////////////////////////////////////////////////////////////////////////////////
PTreeItem AddTreeItem ( PTreeItem o, l_text Caption, p_bitmap Icon,void  (*GenerateSubs)(PTreeItem,void*), void*Args ) {
	PTreeItem i = malloc(sizeof(TTreeItem));
	if ( !i ) return NULL;
	memset(i,0,sizeof(TTreeItem));

	i->Caption = TextDup(Caption);
	i->Icon = Icon;
	i->GenerateSubs = GenerateSubs;
	i->Args = Args;
	i->Parent = o;

	if ( o->Last ) {
		i->Next = o->Last->Next;
		i->Prev = o->Last;
		o->Last->Next->Prev = i;
		o->Last->Next = i;
		o->Last = i;
  } else {
	  i->Next = i;
	  i->Prev = i;
	  o->Last = i;
  }
	return i;
}
////////////////////////////////////////////////////////////////////////////////
void DrawTreeItem ( p_bitmap Out, PTreeItem o, TRect Abs, l_int *x, l_int *y, l_char Flags ,l_char Delta, l_char DeltaY, l_ulong *n, l_ulong ss,l_int SelColor ) {

	RectAssign(&o->BufAbs,*x,*y,(*x)+text_length(default_font,o->Caption)+Delta+2,(*y)+18);

	if ( _RectOverlay(o->BufAbs,Abs) ) {

		if ( o->Last && (Flags & 1) ) {
			rect(Out,o->BufAbs.a.x+3, o->BufAbs.a.y+3, o->BufAbs.a.x+11, o->BufAbs.a.y+11,makecol(128,128,128));
			hline(Out,o->BufAbs.a.x+5,o->BufAbs.a.y+7,o->BufAbs.a.x+9,makecol(0,0,0));
		  if ( !(o->Flags & TIF_EXPENDED) )	vline(Out,o->BufAbs.a.x+7,o->BufAbs.a.y+5,o->BufAbs.a.y+9,makecol(0,0,0));
		}

		if ( o->Icon && (Flags & 2) ) {
			draw_sprite(Out,o->Icon,o->BufAbs.a.x+Delta-16,o->BufAbs.a.y+1);
		}

	  if ( o->Flags & TIF_SELECTED ) {
	    rectfill(Out, Delta+o->BufAbs.a.x+2, o->BufAbs.a.y+DeltaY, o->BufAbs.b.x, o->BufAbs.a.y+DeltaY+text_height(default_font), SelColor);
	    textout(Out, default_font, o->Caption, Delta+3+o->BufAbs.a.x, o->BufAbs.a.y+DeltaY, makecol(255,255,255));
	  } else
	    textout(Out, default_font, o->Caption, Delta+3+o->BufAbs.a.x, o->BufAbs.a.y+DeltaY, makecol(0,0,0));
	}

	if (Flags & 2)
		(*y) += 19;
	else
		(*y) += text_height(default_font)+1;


	(*n)++;


	if ( o->Last && ( o->Flags & TIF_EXPENDED )  ) {
		PTreeItem a = o->Last->Next;
		PTreeItem b = a;
	  (*x) += 16;
		do {
			DrawTreeItem(Out,a,Abs,x,y,Flags,Delta,DeltaY,n,ss,SelColor);
			a = a->Next;
		} while ( a != b );
		(*x) -= 16;
	}
}
////////////////////////////////////////////////////////////////////////////////
void TreeViewDraw ( PWidget o, p_bitmap buffer, PRect w )
{
	l_int x = o->Absolute.a.x-SLLWDG(o)->Pos.x, y = o->Absolute.a.y-(SLLWDG(o)->Pos.y);
	l_char Delta = 0, Flags = 0,DeltaY = 0;
	l_int SelColor = ( o->Flags & WFFocused ) ? makecol(0,0,255) : makecol(215,215,215);
	
	
	//rectfill(buffer, w->a.x, w->a.y, w->b.x, w->b.y, makecol(255,255,255));
	DrawSkin(buffer, SkinTreeview, o->Absolute.a.x, o->Absolute.a.y, o->Absolute.b.x, o->Absolute.b.y);

	Flags = 3;

	if ( Flags == 3 )
		Delta = 32;
	else
		Delta = 16;

	if ( Flags & 2 )	DeltaY = (18-text_height(default_font))/2;


	if ( TREEVIEW(o)->Items->Last  ) {
		PTreeItem a = TREEVIEW(o)->Items->Last->Next;
		PTreeItem b = a;
		l_ulong n = 0;

		do {
			DrawTreeItem(buffer,a,*w,&x,&y,Flags,Delta,DeltaY,&n, SLLWDG(o)->Pos.y,SelColor );

			a = a->Next;
		} while ( a != b );

  }

}
////////////////////////////////////////////////////////////////////////////////
void CalculateMaxEx ( PTreeItem o, l_long *n, l_long Delta, l_long *x, l_long *mx ) {

	*mx = max ( *mx, (*x)+text_length(default_font,o->Caption)+Delta+4 );

	(*n)++;
	if ( o->Last && ( o->Flags & TIF_EXPENDED )  ) {
		PTreeItem a = o->Last->Next;
		PTreeItem b = a;
		(*x) += 16;
		do {
			CalculateMaxEx(a,n,Delta,x,mx);
			a = a->Next;
		} while ( a != b );
		(*x) -= 16;
	}
}
////////////////////////////////////////////////////////////////////////////////
l_ulong CalculateMax ( PTreeView o ) {
	PTreeItem a = o->Items->Last->Next;
	PTreeItem b = a;
	l_long n = 0;
	l_long x = 0;
	l_long mx = 0;


	do {
		CalculateMaxEx(a,&n,32,&x,(l_ulong*)&mx );
		a = a->Next;
	} while ( a != b );

	ScrollBarSetMaxY(SLLWDG(o),max(0,(((l_long)n*19)-((WIDGET(o)->Absolute.b.y-WIDGET(o)->Absolute.a.y))-1)));
	ScrollBarSetMaxX(SLLWDG(o),max(0,(mx-(WIDGET(o)->Absolute.b.x-WIDGET(o)->Absolute.a.x))));
	
	
	//*******
	if ( SLLWDG(o)->Pos.x > SLLWDG(o)->Max.x  ) ScrollBarSetPosX(SLLWDG(o),SLLWDG(o)->Max.x);
	if ( SLLWDG(o)->Pos.y > SLLWDG(o)->Max.y  ) ScrollBarSetPosY(SLLWDG(o),SLLWDG(o)->Max.y);
	//*******
}
////////////////////////////////////////////////////////////////////////////////
void FreeTreeItem ( PTreeItem o ) {
	if ( o->Caption ) free(o->Caption);
	if ( o->Last ) {
		PTreeItem a = o->Last->Next;
		PTreeItem b = a, n = a;
		do {
			n = a->Next;
			FreeTreeItem(a);
			a = n;
		} while ( a != b );
	}
	free(o);
}
////////////////////////////////////////////////////////////////////////////////
void EmptySubTreeItems (PTreeItem o ) {
	if ( o->Last ) {
		PTreeItem a = o->Last->Next;
		PTreeItem b = a, n = a;
		do {
			n = a->Next;
			FreeTreeItem(a);
			a = n;
		} while ( a != b );
	}
	o->Last = NULL;
}
////////////////////////////////////////////////////////////////////////////////
void TreeViewFreeEx ( PWidget o ) {

	if ( TREEVIEW(o)->Items ) FreeTreeItem(TREEVIEW(o)->Items);


}
////////////////////////////////////////////////////////////////////////////////
l_bool FoundItemOnPoint ( PTreeItem o, TPoint p, PTreeItem *f ) {

	if ( PointInRect(p,o->BufAbs) ) {
	  *f = o;
	  return true;
	}

	if ( o->Last && ( o->Flags & TIF_EXPENDED ) ) {
		PTreeItem a = o->Last->Next;
		PTreeItem b = a;
		do {
			if ( FoundItemOnPoint(a,p,f) ) return true;
			a = a->Next;
		} while ( a != b );
	}

	return false;

}
////////////////////////////////////////////////////////////////////////////////
void TreeviewScrollBarRecalculate( PTreeView o ) {

	CalculateMax(o);

}
////////////////////////////////////////////////////////////////////////////////
void TreeviewExpendItem ( PTreeView o, PTreeItem i ) {

	if ( i->GenerateSubs ) {
		//EmptySubTreeItems(i);
		i->GenerateSubs(i,i->Args);
	}
	i->Flags |= TIF_EXPENDED;

	CalculateMax(o);

	WidgetDraw(WIDGET(o), NULL);

	WidgetDraw(WIDGET(SLLWDG(o)->Vertical), NULL);
  WidgetDraw(WIDGET(SLLWDG(o)->Horizontal), NULL);


}

////////////////////////////////////////////////////////////////////////////////
void TreeviewUnExpendItem ( PTreeView o, PTreeItem i ) {

	i->Flags &= ~TIF_EXPENDED;

	CalculateMax(o);

	WidgetDraw(WIDGET(o), NULL);

	WidgetDraw(WIDGET(SLLWDG(o)->Vertical), NULL);
  WidgetDraw(WIDGET(SLLWDG(o)->Horizontal), NULL);

}
////////////////////////////////////////////////////////////////////////////////
void TreeViewSelectItem ( PTreeView o, PTreeItem i ) {

	if ( o->Sel != i ) {
		if ( o->Sel ) {
			o->Sel->Flags &= ~TIF_SELECTED;
		}
	}

  i->Flags |= TIF_SELECTED;
  o->Sel = i;

	WidgetDraw(WIDGET(o), NULL);
	ScrollBarSetPosX(SLLWDG(o),i->BufAbs.a.x-WIDGET(o)->Absolute.a.x+SLLWDG(o)->Pos.x);
	ScrollBarSetPosY(SLLWDG(o),i->BufAbs.a.y-WIDGET(o)->Absolute.a.y+SLLWDG(o)->Pos.y);
	WidgetDraw(WIDGET(o), NULL);
	ScrollBarRedraw(SLLWDG(o));
}
////////////////////////////////////////////////////////////////////////////////
void TreeViewSelectItemAndTell ( PTreeView o, PTreeItem i ) {

	if ( o->Sel != i ) {
		if ( o->Sel ) {
			o->Sel->Flags &= ~TIF_SELECTED;
		}
	}

  i->Flags |= TIF_SELECTED;
  o->Sel = i;

	WidgetDraw(WIDGET(o), NULL);
	ScrollBarSetPosX(SLLWDG(o),i->BufAbs.a.x-WIDGET(o)->Absolute.a.x+SLLWDG(o)->Pos.x);
	ScrollBarSetPosY(SLLWDG(o),i->BufAbs.a.y-WIDGET(o)->Absolute.a.y+SLLWDG(o)->Pos.y);
	WidgetDraw(WIDGET(o), NULL);
	ScrollBarRedraw(SLLWDG(o));
	
	if ( TREEVIEW(o)->OnSelMsg )
		WidgetSendSEvent(WIDGET(o), EV_MESSAGE, o->OnSelMsg, o->Sel);

	
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetFirstFromByCharEx ( PTreeItem i, PTreeItem from, l_char c, l_bool *ok ) {
	PTreeItem r = NULL;
	if ( i->Caption ) if ( ToLower(*i->Caption) == c && *ok ) return i;
	if ( i == from ) *ok = true;
	if ( i->Last && ( i->Flags & TIF_EXPENDED )  ) {
		PTreeItem a = i->Last->Next;
		PTreeItem b = a;
		do {
			if ( r = TreeViewGetFirstFromByCharEx(a,from,c,ok) ) return r; 
			a = a->Next;
		} while ( a != b );
	}
	return NULL;	
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetFirstByCharEx ( PTreeItem i, l_char c ) {
	PTreeItem r = NULL;
	if ( i->Caption ) if ( ToLower(*i->Caption) == c ) return i;
	if ( i->Last && ( i->Flags & TIF_EXPENDED )  ) {
		PTreeItem a = i->Last->Next;
		PTreeItem b = a;
		do {
			if ( r = TreeViewGetFirstByCharEx(a,c) ) return r; 
			a = a->Next;
		} while ( a != b );
	}
	return NULL;	
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetNextEx ( PTreeItem i, PTreeItem who, l_bool *ok ) {
	PTreeItem r = NULL;
	if ( *ok ) return i;
	if ( i == who ) *ok = true;
	if ( i->Last && ( i->Flags & TIF_EXPENDED )  ) {
		PTreeItem a = i->Last->Next;
		PTreeItem b = a;
		do {
			if ( r = TreeViewGetNextEx(a,who,ok) ) return r; 
			a = a->Next;
		} while ( a != b );
	}
	return NULL;	
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetPrevEx ( PTreeItem i, PTreeItem who, PTreeItem *prev ) {
	PTreeItem r = NULL;
	if ( i == who ) return *prev;
	*prev = i;
	if ( i->Last && ( i->Flags & TIF_EXPENDED )  ) {
		PTreeItem a = i->Last->Next;
		PTreeItem b = a;
		do {
			if ( r = TreeViewGetPrevEx(a,who,prev) ) return r; 
			a = a->Next;
		} while ( a != b );
	}
	return NULL;	
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetNext ( PTreeView o, PTreeItem ii ) {
	if ( ii ) {
		l_bool ok = false;
		PTreeItem i = TreeViewGetNextEx(o->Items,ii,&ok);
		if ( !i ) i = ii;
		return i;
	} else if ( o->Items->Last )
		return o->Items->Last->Next;
	return NULL;
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetPrev ( PTreeView o, PTreeItem ii ) {
	if ( ii ) {
		PTreeItem tmp = NULL;
		PTreeItem i = TreeViewGetPrevEx(o->Items,ii,&tmp);
		if ( i == o->Items || !i ) i = ii;
		return i;
	} else if ( o->Items->Last )
		return o->Items->Last->Next;
	return NULL;
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetByCharFrom ( PTreeView o, PTreeItem ii, l_uchar c ) {
	l_bool ok = !(ii);
	PTreeItem i = TreeViewGetFirstFromByCharEx(o->Items,ii,c,&ok);
	if ( !i ) i = TreeViewGetFirstByCharEx(o->Items,c);
	if ( !i ) i = ii;
	return i;
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeViewGetLast ( PTreeView o ) {
	PTreeItem i = TREEVIEW(o)->Items->Last;
	if ( !i ) return NULL;
	while ( i->Last && ( i->Flags & TIF_EXPENDED ) ) {
		i = i->Last;	
	}	
	return i;
}
////////////////////////////////////////////////////////////////////////////////
l_bool TreeViewEventHandler ( PWidget o, PEvent Ev ) {
	
	if (Ev->Type == EV_KEYBOARD) {
		if ( SCANCODE(Ev,KEY_DOWN) ) {
			PTreeItem i = TreeViewGetNext(TREEVIEW(o),TREEVIEW(o)->Sel);
			if ( i ) 
				TreeViewSelectItemAndTell(TREEVIEW(o),i);
		} else if ( SCANCODE(Ev,KEY_UP) ) {
			PTreeItem i = TreeViewGetPrev(TREEVIEW(o),TREEVIEW(o)->Sel);
			if ( i ) 
				TreeViewSelectItemAndTell(TREEVIEW(o),i);
		} else if ( SCANCODE(Ev,KEY_HOME) ) {
			if ( TREEVIEW(o)->Items->Last ) 
				TreeViewSelectItemAndTell(TREEVIEW(o),TREEVIEW(o)->Items->Last->Next);
		} else if ( SCANCODE(Ev,KEY_END) ) {
			PTreeItem i = TreeViewGetLast(TREEVIEW(o));
			if ( i ) 
				TreeViewSelectItemAndTell(TREEVIEW(o),i);
		} else if ( SCANCODE(Ev,KEY_ENTER) ) {
			if ( TREEVIEW(o)->Sel ) {
				if ( TREEVIEW(o)->Sel->Flags & TIF_EXPENDED )
					TreeviewUnExpendItem(TREEVIEW(o),TREEVIEW(o)->Sel);
				else
					TreeviewExpendItem(TREEVIEW(o),TREEVIEW(o)->Sel);
			}
		} else if ( !Ev->Keyb.Shift ) {
			l_char sel = ToLower(ASCII(Ev));
			PTreeItem i = TreeViewGetByCharFrom(TREEVIEW(o),TREEVIEW(o)->Sel,sel);
			if ( i ) 
				TreeViewSelectItemAndTell(TREEVIEW(o),i);
		}	
	} else if (Ev->Type == EV_MESSAGE) {
 
		if ( Ev->Message == WEvNotifyLostFocus ) {
			WidgetDraw(o, NULL);
			return true;
		}
		if ( Ev->Message == WEvNotifyFocused ) {
			WidgetDraw(o, NULL);
			return true;
		}
		if ( Ev->Message == WEvNotifySameFocus ) {
			
			return true;
		}
			
		if ( Ev->Message == MSG_NOTIFY_AUTO_RESIZE ) {
			CalculateMax(TREEVIEW(o));
			return true;
		}
		
		
	} else if ( Ev->Type == EV_MOUSE ) {
	
		if ( Ev->Message == WEvMouseLDClk ) {
			PTreeItem i = NULL;
			l_bool SendMsg = 1;



			if ( FoundItemOnPoint(TREEVIEW(o)->Items,Mouse->State.p,&i) ) {

				if ( i->Flags & TIF_EXPENDED )
					TreeviewUnExpendItem(TREEVIEW(o),i);
				else
					TreeviewExpendItem(TREEVIEW(o),i);
			}
			return true;
		} else if ( Ev->Message == WEvMouseLDown || Ev->Message == WEvMouseRDown ) {
			PTreeItem i = NULL;
			l_bool SendMsg = 1;



			if ( FoundItemOnPoint(TREEVIEW(o)->Items,Mouse->State.p,&i) ) {

				if ( TREEVIEW(o)->Sel != i ) {
					if ( TREEVIEW(o)->Sel ) {
						TREEVIEW(o)->Sel->Flags &= ~TIF_SELECTED;
						WidgetDraw(o, &(TREEVIEW(o)->Sel->BufAbs));
					}
				} else
				  SendMsg = 0;

			  i->Flags |= TIF_SELECTED;
			  WidgetDraw(o, &(i->BufAbs));
			  TREEVIEW(o)->Sel = i;


			} else if ( TREEVIEW(o)->Sel ) {
				TREEVIEW(o)->Sel->Flags &= ~TIF_SELECTED;
				WidgetDraw(o, &(TREEVIEW(o)->Sel->BufAbs));
				TREEVIEW(o)->Sel = NULL;
			}

			if ( TREEVIEW(o)->OnSelMsg && SendMsg  )
					WidgetSendSEvent(o, EV_MESSAGE, TREEVIEW(o)->OnSelMsg, TREEVIEW(o)->Sel);

			if ( Ev->Message == WEvMouseRDown ) {
				if ( TREEVIEW(o)->GenerateItemMenu ) {
					if ( TREEVIEW(o)->Sel ) {
						PMenu m = TREEVIEW(o)->GenerateItemMenu(TREEVIEW(o),TREEVIEW(o)->Sel);
						if ( m ) {
							PopUpMenu(o->AppOwner,Mouse->State.p,m,o,0);
							// FreeMenu(m); Function TBD
						}
					}
				} else if ( TREEVIEW(o)->OnMenuMsg ) {
					if ( TREEVIEW(o)->Sel )
						WidgetSendSEvent(o, EV_MESSAGE, TREEVIEW(o)->OnMenuMsg, TREEVIEW(o)->Sel);
				}
			}
			return true;
		}


		
	}
	return ScrollWidgetEventHandler(o,Ev);
}
////////////////////////////////////////////////////////////////////////////////
void Scroll ( PScrollWidget o, TPoint p ) {

	// To prevent blicking, force temporaly double buffer !

	if ( WIDGET(o)->Flags & WFForceBuffer )
		WidgetDraw(WIDGET(o),NULL);
	else {
		WIDGET(o)->Flags |= WFForceBuffer;
		WidgetDraw(WIDGET(o),NULL);
		WIDGET(o)->Flags &= ~WFForceBuffer;
	}


}
////////////////////////////////////////////////////////////////////////////////
l_text TreeViewRenameItemBox ( PTreeView t, PTreeItem i ) {

	PTextbox tb;
	TRect 		r;
	l_text ret = NULL;

	RectAssign(&r,i->BufAbs.a.x+32,i->BufAbs.a.y+2,WIDGET(t)->Absolute.b.x-32,i->BufAbs.b.y-2);
	tb = CreateTextbox(&Me,r,TBF_EDITABLE);
	InsertWidget(DeskTop, WIDGET(tb));
	TextBoxSetTextEx(tb,i->Caption);

	tb->SelPos = tb->Text;
	tb->SelPosEnd = tb->Text+TextLen(i->Caption);
	WidgetDraw(WIDGET(tb), NULL);

	tb->ValidMsg = MSG_OK;


	WidgetSetFocus(WIDGET(tb));

	if ( WidgetExecute(WIDGET(tb)) != MSG_CANCEL ) {
		if ( TextCaseCompare(tb->Text,i->Caption) )
			ret = TextDup(tb->Text);
	}
	WidgetDispose(WIDGET(tb));

	return ret;

}
////////////////////////////////////////////////////////////////////////////////
void TreeViewRenameItem ( PTreeView o, PTreeItem i, l_text Caption ) {

	free(i->Caption);
	i->Caption = TextDup(Caption);

	WidgetDraw(WIDGET(o), NULL);

	WidgetDraw(WIDGET(SLLWDG(o)->Vertical), NULL);
  WidgetDraw(WIDGET(SLLWDG(o)->Horizontal), NULL);
}
////////////////////////////////////////////////////////////////////////////////
void TreeViewDeleteItem ( PTreeView o, PTreeItem i ) {
	PTreeItem p = i->Parent;

	if ( o->Sel == i ) {
		if ( i->Next == i ) {
			if ( i->Parent == o->Items )
				o->Sel = NULL;
			else
				o->Sel = i->Parent;
		} else
			o->Sel = i->Next;

		if ( o->Sel ) o->Sel->Flags |= TIF_SELECTED;
	}

  if ( i->Next == i ) {
	  p->Last = NULL;
		if ( p->Flags & TIF_EXPENDED ) p->Flags &= ~TIF_EXPENDED;
  } else {
	  i->Next->Prev = i->Prev;
	  i->Prev->Next = i->Next;
	  if ( p->Last == i ) p->Last = i->Prev;
  }


	WidgetDraw(WIDGET(o), NULL);
	WidgetDraw(WIDGET(SLLWDG(o)->Vertical), NULL);
  WidgetDraw(WIDGET(SLLWDG(o)->Horizontal), NULL);
}

////////////////////////////////////////////////////////////////////////////////
PTreeView InitTreeView( PTreeView b, PApplication App, TRect r, l_ulong OnSelMsg )
{

	b->Items = malloc(sizeof(TTreeItem));
	memset(b->Items,0,sizeof(TTreeItem));

	b->Items->Flags |= TIF_EXPENDED;

	b->OnSelMsg = OnSelMsg;

	InitScrollWidget(SLLWDG(b), App, r, 3);

	WIDGET(b)->Draw = &TreeViewDraw;
	WIDGET(b)->EventHandler = &TreeViewEventHandler;
	WIDGET(b)->FreeEx = &TreeViewFreeEx;
	WIDGET(b)->Flags |= WFFocusable;

	SLLWDG(b)->Scrolled = &Scroll;
	SLLWDG(b)->Delta.y = 19;
	SLLWDG(b)->Delta.x = 16;
	
	return b;
}
////////////////////////////////////////////////////////////////////////////////
PTreeView CreateTreeView( PApplication App, TRect r, l_ulong OnSelMsg )
{
	PTreeView b = malloc(sizeof(TTreeView));

	if (!b) return NULL;

	//r.b.x -= 10;

	memset(b, 0, sizeof(TTreeView));

	InitTreeView(b,App,r,OnSelMsg);
	
	return b;
}
////////////////////////////////////////////////////////////////////////////////
l_bool LibMain ( l_text Args )
{
	SkinTreeview = GetSkinItem("Treeview");

	APPEXPORT(CreateTreeView);
	APPEXPORT(EmptySubTreeItems);
	APPEXPORT(AddTreeItem);
	APPEXPORT(TreeviewExpendItem);
	APPEXPORT(TreeviewUnExpendItem);
	APPEXPORT(TreeViewRenameItemBox);
	APPEXPORT(TreeViewSelectItem);
	APPEXPORT(TreeViewRenameItem);
	APPEXPORT(TreeViewDeleteItem);

	APPEXPORT(TreeviewScrollBarRecalculate);
	APPEXPORT(InitTreeView);
	APPEXPORT(TreeViewEventHandler);	
	return true;
}
////////////////////////////////////////////////////////////////////////////////
void Close (void)
{

}
////////////////////////////////////////////////////////////////////////////////
