/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/***********************************************************************
 * This program saves a copy of its input to a file and prints a copy to
 * the standard output.
 *
 * Author: James Hall
 *
 * 2008-07-03: DOS-ified the code by using the INT21H functions.
 * 			   Only compiler I have is OpenWatcom, so am not sure
 * 			   if it's portable compiler-wise - AJM
 * 2008-07-01: Added append and ignore-interrupts functionality by
 *             porting GNU Coreutils code - AJM
 */

#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "getopt.h"
#include "tee.h"

/* prototypes */
int main (int, char **);
static int outtee(int, const char **);
static void usage(void);

char params;		/* User defined parameters */
char mode;			/* File's access permissions, either read or append */

extern int errno;
extern int opterr;
extern char * opt_unknown_arg;

int
main (int argc, char **argv)
{
	int i;
	int optc;

	opterr = 1;			/* Allow error messaging in getopt() */

	if (kittenopen("TEE") <= 0)
		fprintf(stderr, 
			PROGNAME ": Error loading kitten file.\r\n"
			"     Please check your LANG and NLSPATH environment variables.\r\n"
			"     Defaulting to EN.\r\n");

	opt_unknown_arg = ERR_UNKNOWN_OPT;

	while ((optc = getopt(argc, argv, "AI?")) != -1) {
		switch (optc) {
			case 'A':
				params |= APPEND;
				break;
			case 'I':
				params |= INTERRUPT;
				break;
			case '?':
				usage();
		}
	}

	if (params & INTERRUPT)
		signal(SIGINT, SIG_IGN);

	mode = (params & APPEND) ? MODE_APPEND : MODE_WRITE;

	/* Tee the input */
	return outtee(argc - optind, (const char **) &argv[optind]);
}

int
outtee(int numFiles, const char **files)
{
	int i;
	char status;
	char buffer[BUFSIZ];
	int * handles;			/* file handles */
	unsigned bytesRead;
	unsigned bytesWritten;

	status = EXIT_SUCCESS;
	memset(buffer, '\0', sizeof buffer);

	if ((handles = (int *) calloc(numFiles + 1, sizeof (int))) == NULL) {
		PRINT_ERR_ALLOC;
		exit(1);		/* Unable to continue */
	}

	/* Move all files up one in array to make room for STDOUT */
	for (i = numFiles; i >= 1; --i)
		files[i] = files[i - 1];

	handles[0] = 1;						/* STDOUT */

	for (i = 1; i <= numFiles; ++i) {
		if (mode == MODE_WRITE) {
			/* Overwrite the file if one exists */
			if (_dos_creat(files[i], _A_NORMAL, &handles[i])) {
				PRINT_ERR_OPEN(files[i]);
				exit(1);
			}
		} else if (mode == MODE_APPEND) {
			/* Only create a new file if one does not exist */
			if (_dos_creatnew(files[i], _A_NORMAL, &handles[i])
					&& errno != EEXIST) {
				PRINT_ERR_OPEN(files[i]);
				exit(1);
			}
			/* If file already exists, open it */
			if (errno == EEXIST) {
				if (_dos_open(files[i], O_WRONLY, &handles[i])) {
					PRINT_ERR_OPEN(files[i]);
					exit(1);
				}
			}
		}

		if (mode == MODE_APPEND)
			lseek(handles[i], 0, SEEK_END);
	}

	for (;;) {
		/* Read from STDIN, when you hit EOF, bytesRead will equal 0 */
		if (_dos_read(0, buffer, BUFSIZ, &bytesRead) || bytesRead == 0)
			break;

		/* Write to file(s) */
		for (i = 0; i <= numFiles; ++i) {
			if (_dos_write(handles[i], buffer, bytesRead, &bytesWritten)
					|| bytesRead != bytesWritten) {
				PRINT_ERR_WRITE(files[i]);
				status = EXIT_FAILURE;
			}
		}
	}

	for (i = 1; i <= numFiles; ++i) {
		if (handles[i] && _dos_close(handles[i])) {
			PRINT_ERR_CLOSE(files[i]);
			status = EXIT_FAILURE;
		}
	}

	free(handles);

	return status;
}

void 
usage (void)
{
	printf( PROGNAME " " VERSION ": %s\r\n"
			"%s: " PROGNAME " [/A] [/I] %s\r\n\r\n"
			"  /A    %s\r\n"
			"  /I    %s\r\n"
			"  /?    %s\r\n"
			"\r\n",
			MSG_DESCRIPTION, MSG_USAGE_STR, MSG_FILE_STR,
			MSG_A_OPT, MSG_I_OPT, MSG_HELP_OPT
	);
	exit (1);
}
