-- main.e - File hiding/retrieving - HIP 2.0
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file contains routines for hiding data and retrieving it.
-- They take care of the header and encryption.

constant include_name = "main.e"

include crc32.e
include encrypt.e
include scatter.e
include truerand.e
include fileutil.e

constant HIP_ID = "HP" & 20

constant FLAGS_CRC = 1


-- Order of fields in the hidden data

-- Header (3 bytes)
-- Flags (1 byte)
-- Uncompressed data size (4 bytes)
-- Compression method (1 byte)
-- Compressed data size (if compression != none) (4 bytes) (not certain yet)
-- File name (1 length byte + contents)
-- Extra fields (2 length bytes + contents)
-- Data
-- CRC-32 (if present)

atom fn, bytes_handled, data_size, write_crc

global function total_bits (   atom compressed_size,
			       atom compressed,
			       atom write_crc_,
			       atom fn_len,
			       atom ef_len
			       )

    atom total_bytes

    total_bytes = 8 + 8 +       -- IV and salt
		 3 + 1 + 4 +    -- Header (3 bytes) + Flags (1 byte) + Size (4 bytes)
		 1 + 4 * compressed +  -- Compression method + compressed size (if compressed)
		 fn_len + 1 + -- File name (variable) + file name length (1 byte)
		 ef_len + 2 + -- Extra fields (variable) + extra fields size (2 bytes)
		 compressed_size +           -- File size
		 4 * write_crc_    -- CRC (4 bytes), if present

    return total_bytes * 8
end function

global function init_writing (
			      atom data_length,      -- Picture size
			      atom fn_,              -- File to be written in picture
			      atom write_crc_,       -- Boolean: write CRC?

			      sequence file_name,    -- Name of the file
			      sequence extra_fields, -- Optional extra fields
			      atom enc_algorithm,    -- Encryption algorithm to be used
			      sequence key,          -- Key to be used
			      atom max_capacity)     -- Maximum allowed number of bits

    atom fn_size, flags, ef_len
    sequence iv, salt

    fn = fn_
    write_crc = write_crc_

    -- Get file size
    fn_size = file_size (fn)

    ef_len = length(extra_fields)

    -- Compute number of bytes to write
    if total_bits ( fn_size, 0, write_crc, length(file_name),
	length (extra_fields)) > max_capacity

    then return ERROR_FILETOOBIG

    end if

    -- Get random Initialization vector and salt

    salt = get_random_data (8)
    iv = get_random_data (8)

    init_scatter_write (data_length)         -- Initialize scattering

    -- Write random salt
    set_scatter_key (key)                    -- Set key
    write_string (salt)                     -- Write salt

    key = salt & key                        -- Salt key

    set_scatter_key (key)                    -- Set key

    write_string (iv)                       -- Write IV

    -- Initialize encryption
    init_encryption (enc_algorithm, key, iv)

    -- Generate flags byte
    flags = 0
    if write_crc then flags = flags + FLAGS_CRC end if


    -- Write header
    write_string (encrypt_data(HIP_ID & flags))    -- Write ID and flags
    write_string (encrypt_data(int_to_bytes (fn_size)))   -- Write uncompressed size
    write_string (encrypt_data({0}))              -- Compression method, uncompressed

    -- No compression, so do not write compressed size

    write_string (encrypt_data( length(file_name) & file_name))    -- File name
    write_string (encrypt_data( {and_bits (ef_len, #FF), floor (ef_len / #100) })) -- Extra fields length
    write_string (encrypt_data( extra_fields ))                    -- Extra fields

    -- Initialize CRC
    init_crc ()

    if seek(fn, 0) then return ERROR_FILEERROR end if


    bytes_handled = 0
    data_size = fn_size

    return 0
end function

constant bytes_per_call =  2048 * 2

global function go_write ()
    atom last_block, bytes_to_read
    sequence block

    if bytes_handled + bytes_per_call < data_size then
	-- Not last block
	last_block = 0
	bytes_to_read = bytes_per_call

    else    -- Last block
	last_block = 1
	bytes_to_read = data_size - bytes_handled
    end if

    -- Read block
    block = read_bytes (fn, bytes_to_read)
    if length (block) != bytes_to_read then    -- Unexpected EOF
	return ERROR_FILEERROR
    end if

    update_crc (block)      -- Update CRC

    write_string (encrypt_data (block))  -- Encrypt and write block

    bytes_handled = bytes_handled + bytes_to_read

    if last_block then

	if write_crc then
	    write_string (encrypt_data (get_crc()))
	end if

	-- All the bytes have been read
	return INFO_LASTBLOCK
    end if

    return 0
end function


global function get_hidden_info (
			      atom enc_algorithm,    -- Encryption algorithm, 0 = auto
			      sequence key,          -- Key to be used
			      atom max_capacity)     -- Maximum allowed number of bits

    atom flags, size, fn_len, ef_len, compression
    sequence salt, iv, id, decrypted, file_name, extra_fields


    -- Write random salt
    set_scatter_key (key)                    -- Set key
    salt = read_string (8)                  -- Read salt

    key = salt & key                        -- Salt key

    set_scatter_key (key)                    -- Set key

    iv = read_string (8)                    -- Read IV

    id = read_string (length( HIP_ID) )
    if enc_algorithm = 0 then
	-- Identify algorithm
	enc_algorithm = identify_encryption_algorithm (key, iv, id, HIP_ID)
	if enc_algorithm = 0 then           -- No algorithm worked
	    return ERROR_NOHIDDENFILE
	end if

    end if

    -- Initialize decryption
    init_encryption (enc_algorithm, key, iv)

    if compare (decrypt_data (id), HIP_ID) then
	return ERROR_NOHIDDENFILE
    end if

    -- Get flags byte
    decrypted = decrypt_data (read_string (1))
    flags = decrypted[1]

    if and_bits (flags, FLAGS_CRC) then
	write_crc = 1
    else write_crc = 0 end if

    if and_bits (flags, #FE) then   -- Reserved bit set, error
	return ERROR_UNSUPPORTED
    end if

    -- Uncompressed size
    size = bytes_to_int (decrypt_data (read_string (4) ))

    -- Read compression method
    decrypted = decrypt_data (read_string (1))
    compression = decrypted[1]

    if compression != 0 then        -- Compressed file, unsupported yet
	return ERROR_UNSUPPORTED
    end if

    -- We will only reach this point when compression = none, so there is
    -- no need to read compressed length

    -- Read file name length
    decrypted = decrypt_data (read_string (1))
    fn_len = decrypted[1]

    file_name = decrypt_data (read_string (fn_len))  -- Read file name

    -- Read extra fields length
    decrypted = decrypt_data (read_string (2))
    ef_len = decrypted[1] + decrypted[2] * #100

    -- Extra fields can be rather big, so let's test again

    if total_bits ( size, 0, write_crc, fn_len, ef_len ) > max_capacity then
	-- The contents could not fit
	return ERROR_CORRUPTED
    end if

    -- Read extra fields
    extra_fields = decrypt_data (read_string (ef_len))

    -- Everything OK
    data_size = size
    bytes_handled = 0

    -- Return information
    return {file_name, size, enc_algorithm, write_crc}
end function

global procedure init_reading (atom fn_)

    -- Open file
    fn = fn_

    -- Initialize CRC
    init_crc ()
end procedure

global function go_read ()
    atom last_block, bytes_to_read
    sequence block

    if bytes_handled + bytes_per_call < data_size then
	-- Not last block
	last_block = 0
	bytes_to_read = bytes_per_call

    else    -- Last block
	last_block = 1
	bytes_to_read = data_size - bytes_handled
    end if

    -- Read block
    block = decrypt_data(read_string (bytes_to_read))

    update_crc (block)      -- Update CRC

    puts (fn, block)        -- Write block

    bytes_handled = bytes_handled + bytes_to_read


    if last_block then

	if write_crc then
	    block = decrypt_data ( read_string (4) )
	    if compare (block, get_crc()) then
		return ERROR_CORRUPTED
	    end if
	end if

	-- All the bytes have been written, close file

	return INFO_LASTBLOCK
    end if

    return 0

end function

global function completion_status ()
    return bytes_handled / data_size
end function

global function get_erase_data (atom data_length, object pwd)
    sequence data
    if sequence (pwd) then
	-- password given, erase salt for the given password
	init_scatter_write (data_length)    -- initialize writing
	set_scatter_key (pwd)               -- set password
	write_string (get_random_data (8))  -- overwrite salt with random data
	data = get_scattered_data ()        -- get data
	end_scatter ()                      -- free scatter.e resources
	return data                         -- done

    else
	-- no password given, erase data from the whole picture
	return make_erase_data ( data_length )
    end if
end function

