;Wolfware Assembler
;Copyright (c) 1985-1991 Eric Tauck. All rights reserved.

;===============================================;
;                   Reg_Look                    ;
; Register look up. Checks if the operand at    ;
; SI is a valid register. If so the carry is    ;
; set and AX returns the type, BX returns the   ;
; value, and CX returns the size.               ;
;===============================================;

Reg_Look Proc Near
 Cmp Byte [Si],2        ;check if length two
 Je Chkreg              ;jump if so, check
 Clc
 Ret

;----- check for most common registers

Chkreg
 Mov Ax,[Si+1]          ;get first two bytes
 Cmp Ax,'AX'            ;check for common registers
 Je Regax
 Cmp Ax,'AL'
 Je Regal
 Cmp Ax,'BX'
 Je Regbx
 Cmp Ax,'CX'
 Je Regcx
 Cmp Ax,'DX'
 Je Regdx
 Cmp Ax,'SI'
 Je Regsi
 Cmp Ax,'DI'
 Je Regdi
 Jmps Chkoreg

;----- AX register

Regax
 Mov Ax,Reg Or Acum     ;type
 Mov Bx,000b * 8        ;value
 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- AL register

Regal
 Mov Ax,Reg Or Acum     ;type
 Mov Bx,000b * 8        ;value
 Mov Cx,S8bit           ;size
 Stc
 Ret

;----- BX register

Regbx
 Mov Ax,Reg             ;type
 Mov Bx,011b * 8        ;value
 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- CX register

Regcx
 Mov Ax,Reg             ;type
 Mov Bx,001b * 8        ;value
 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- DX register

Regdx
 Mov Ax,Reg             ;type
 Mov Bx,010b * 8        ;value
 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- SI register

Regsi
 Mov Ax,Reg             ;type
 Mov Bx,110b * 8        ;value
 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- DI register

Regdi
 Mov Ax,Reg             ;type
 Mov Bx,111b * 8        ;value
 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- check for other registers

Chkoreg
 Mov Dl, 100b Or Bit7   ;set bit pattern for next register
 Cmp Ax,'AH'            ;check for other registers
 Je Regfoun
 Mov Dl, 001b Or Bit7
 Cmp Ax,'CL'
 Je Regfoun
 Mov Dl, 101b Or Bit7
 Cmp Ax,'CH'
 Je Regfoun
 Mov Dl, 011b Or Bit7
 Cmp Ax,'BL'
 Je Regfoun
 Mov Dl, 111b Or Bit7
 Cmp Ax,'BH'
 Je Regfoun
 Mov Dl, 010b Or Bit7
 Cmp Ax,'DL'
 Je Regfoun
 Mov Dl, 110b Or Bit7
 Cmp Ax,'DH'
 Je Regfoun
 Mov Dl, 101b
 Cmp Ax,'BP'
 Je Regfoun
 Mov Dl, 100b
 Cmp Ax,'SP'
 Je Regfoun
 Mov Dl, 11b Or Bit8
 Cmp Ax,'DS'
 Je Regfoun
 Mov Dl, 00b Or Bit8
 Cmp Ax,'ES'
 Je Regfoun
 Mov Dl, 10b Or Bit8
 Cmp Ax,'SS'
 Je Regfoun
 Mov Dl, 01b Or Bit8
 Cmp Ax,'CS'
 Je Regfoun

;----- not a register

 Clc
 Ret

;----- is reg, set values

Regfoun
 Mov Bl, Dl             ;lower three bits
 And Bl, Not(Bit7 or Bit8) ;clear special bits
 Shl Bl
 Shl Bl
 Shl Bl                 ;adjust to proper position
 Sub Bh, Bh             ;clear upper bits

 Test Dl,Bit8           ;check if segment register
 Jnz Setsr

 Mov Ax,Reg             ;type
 Test Dl,Bit7           ;check if normal 8 bit reg.
 Jnz Seter

;----- normal 16 bit register

 Mov Cx,S16bit          ;size
 Stc
 Ret

;----- 8 bit register

Seter
 Mov Cx,S8bit           ;size
 Stc
 Ret

;----- segment register

Setsr
 Mov Ax,Seg             ;type
 Mov Cx,S16bit          ;size
 Stc
 Ret
 Endp                   ;Reg_Look

;===============================================;
;                   Num_Look                    ;
; Number look up. Checks if the operand at SI   ;
; is a valid number. If so the carry is set     ;
; and AX returns the type, BX returns the       ;
; value, and CX returns the size. Note that     ;
; anything starting with a digit 0 to 9 is      ;
; considered a number.                          ;
;===============================================;

Num_Look Proc Near
 Mov Al,[Si+1]          ;first byte
 Cmp Al,'0'             ;check if too low
 Jb Notnum
 Cmp Al,'9'             ;check if too high
 Ja Notnum
 Jmps Startnum

;----- not number, exit

Notnum
 Clc                    ;clear carry, not found
 Ret

;----- must be number

Startnum
 Cmp Byte [Si],1        ;check if length one
 Je Singdig             ;jump if so

 Push Si                ;save source
 Lodsb                  ;length
 Mov Bl,Al
 Sub Bh,Bh              ;BX has length

 Mov Al,[Si+Bx-1]       ;load last byte
 Cmp Al,'H'             ;check if H for hexadecimal
 Je Hexidec             ;jump if so
 Cmp Al,'B'             ;check if B for binary
 Je Binary              ;jump if so

;----- decimal number

 Mov Cx,10              ;base 10
 Jmps Checksin

;----- single digit (decimal) number

Singdig
 Mov Ax,Immed Or True_Const ;type
 Mov Bl,[Si+1]          ;digit
 Sub Bl,'0'             ;make it a number
 Sub Bh,Bh
 Cmp Pres_Si,' '        ;check if any sign
 Jne Singsi             ;jump if so
 Mov Cx,S8bit Or S16bit ;set size
 Stc
 Ret

Singsi
 Sub Cx,Cx              ;clear size
 Call Num_Set           ;set size
 Stc
 Ret

;----- binary number

Binary
 Dec Bx                 ;remove H
 Mov Cx,2               ;base 2
 Jmps Checksin

;----- hexidecimal number

Hexidec
 Dec Bx                 ;remove B
 Mov Cx,16              ;base 16

;----- interpret number

Checksin
 Sub Ax,Ax              ;initial sum
 Mov Dx,Ax              ;set to zero for multiply
 Xchg Cx,Bx             ;switch values
 Mov Di,Offset Hexbase  ;characters to scan

;----- loop for each digit

Digloop
 Push Cx
 Push Di
 Push Ax

 Lodsb                  ;load character
 Mov Cx,Bx              ;base
 Repne
 Scasb                  ;scan for match
 Jne Numerr2            ;jump if no match, error

 Inc Cx
 Sub Cx,Bx              ;negative value
 Neg Cx                 ;value

 Pop Ax                 ;old sum
 Mul Ax,Bx              ;times base
 Jc Numerr              ;jump if carry, too big
 Add Ax,Cx              ;new sum
 Jc Numerr              ;jump if carry, too big

Digecont
 Pop Di
 Pop Cx
 Loop Digloop           ;loop if digits remaining

;----- successful, all digits added in

 Mov Bx,Ax              ;value
 Mov Ax,Immed Or True_Const ;type
 Sub Cx,Cx              ;no size (size set later)
 Pop Si
 Call Num_Set           ;set number size
 Stc                    ;set carry, found
 Ret

;----- number too big

Numerr
 Push Ax
 Push Si
 Mov Bp,Sp              ;stack
 Mov Si,[Bp+8]          ;original source
 Mov Ax,0425h           ;error 37
 Call Error             ;error routine
 Sub Dx,Dx              ;reinitialize DX
 Pop Si
 Pop Ax
 Jmps Digecont          ;continue

;----- invalid number

Numerr2
 Add Sp,6               ;throw away three values on stack
 Pop Si                 ;restore source
 Mov Dl,Al              ;invalid character
 Mov Ax,2487h           ;error 135
 Call Error             ;error routine
 Mov Ax,Immed           ;type
 Sub Bx,Bx              ;value
 Mov Cx,Bx              ;no size (size set later)
 Call Num_Set           ;set number size
 Stc
 Ret
 Endp                   ;Num_Look

;===============================================;
;                    Num_Set                    ;
; Number size and value set. Calculates the     ;
; size and value of a number based on the sign  ;
; (if any) in PRES_SI. The original type,       ;
; value, and size should be in AX, BX, and CX.  ;
; Expects SI to be pointing at the operand in   ;
; case of error.                                ;
;===============================================;

Num_Set Proc Near
 Mov Dl,Pres_Si         ;sign
 Cmp Dl,' '             ;unsigned
 Je Numsus

 Mov Pres_Si,' '        ;reset sign
 Cmp Dl,'-'             ;signed minus
 Je Numset
 Cmp Dl,'+'             ;signed plus
 Je Nsplus

;----- illegal sign character

 Push Ax
 Mov Ax,2019h           ;error 25
 Call Error             ;error routine
 Pop Ax

 Mov Cx,S16bit Or S8bit ;16 or 8 bit size
 Ret

;----- no sign, just set size

Numsus
 Test Ax,Signed         ;test if operand already signed
 Jnz Numssi             ;jump if so

;----- set unsigned size

 Mov Cx,S16bit          ;can always be 16 bit
 Or Bh,Bh               ;check if high byte zero
 Jnz Nonumset           ;jump if not, cannot be 8 bit
 Or Cx,S8bit            ;set to also 8 bit

Nonumset Ret

;----- set signed size

Numssi
 Mov Cx,S16bit          ;can always be 16 bit
 Cmp Bx,-128            ;check if 0 to -128
 Jl Nonumset
 Cmp Bx,127             ;check if 0 to 127
 Jg Nonumset
 Or Cx,S8bit            ;make 8 bit
 Ret

;----- minus

Numset
 Or Ax,Signed           ;make signed operand
 Mov Dx,Bx
 Neg Bx                 ;negative value

 Mov Cx,S16bit          ;can always be 16 bit

 Cmp Dx,32768           ;check lower limit of signed numbers
 Ja Numsover            ;jump if too low
 Cmp Dx,128             ;check if 0 to -128 for 8 bit
 Ja Nonumset            ;jump if not

 Or Cx,S8bit
 Ret

;----- plus

Nsplus
 Or Ax,Signed           ;make signed operand
 Mov Cx,S16bit          ;can always be 16 bit

 Cmp Bx,32767           ;check upper limit of signed numbers
 Ja Numsover            ;jump if too high
 Cmp Bx,127             ;check if 0 to 127 for 8 bit
 Ja Nonumset            ;jump if not

 Or Cx,S8bit            ;can be 8 bit
 Ret

;----- overflow

Numsover
 Push Ax
 Mov Ax,0425h           ;error 37, overflow
 Call Error             ;error routine
 Pop Ax
 Ret
 Endp                   ;Num_Set

;===============================================;
;                   Char_Look                   ;
; Character look up. Checks if the operand at   ;
; SI is a single character in quotes. If so     ;
; the carry is set and AX returns the type, BX  ;
; returns the value, and CX returns the size.   ;
;===============================================;

Char_Look Proc Near
 Cmp Byte [Si],2        ;check length
 Jne Notchar            ;jump if too big
 Cmp Byte [Si+1],Str_Delim ;make sure is string
 Jne Notchar            ;jump if not single quote

;----- successful

 Mov Ah,[Si+2]          ;byte
 Mov Al,1               ;string length
 Mov Di,Str_Buff        ;string storage
 Mov [Di],Ax            ;store

 Mov Bl,Ah              ;value
 Sub Bh,Bh
 Mov Ax,Immed Or String Or True_Const ;type
 Mov Cx,S8bit Or S16bit ;size
 Stc                    ;set carry
 Ret

;----- unsuccessful

Notchar
 Clc                    ;clear carry
 Ret
 Endp                   ;Char_Look

;===============================================;
;                  Dchar_Look                   ;
; Double character look up. Checks if the       ;
; operand at SI is exacty two characters in     ;
; quotes. If so the carry is set and AX         ;
; returns the type, BX returns the (two byte)   ;
; value, and CX returns the size.               ;
;===============================================;

Dchar_Look Proc Near
 Cmp Byte [Si],3        ;check length
 Jne Notdchar           ;jump if too big
 Cmp Byte [Si+1],Str_Delim ;make sure is string
 Jne Notdchar           ;jump if not single quote

;----- successful

 Mov Di,Str_Buff        ;string storage
 Mov Al,2               ;string length
 Mov [Di],Al            ;store
 Mov Ax,[Si+2]          ;word of data
 Mov [Di+1],Ax          ;store

 Mov Bx,Ax              ;value
 Mov Ax,Immed Or String Or True_Const ;type
 Mov Cx,S16bit          ;size
 Stc                    ;set carry
 Ret

;----- unsuccessful

Notdchar
 Clc                    ;clear carry
 Ret
 Endp                   ;Dchar_Look

;===============================================;
;                   Str_Look                    ;
; String look up. Checks if the operand at SI   ;
; is a string of character(s) in quotes. If so  ;
; the carry is set and AX returns the type, BX  ;
; returns the value, and CX returns the size.   ;
; If found, the string itself (minus the        ;
; quotes) is placed in STR_BUFF.                ;
;===============================================;

Str_Look Proc Near
 Mov Al,[Si+1]          ;first character
 Cmp Al,39              ;first character must be '
 Jne Notstr             ;jump if not single quote

;----- save string

 Push Si
 Mov Di,Str_Buff        ;string storage
 Lodsb                  ;load length
 Dec Al                 ;remove single quote
 Stosb                  ;save length
 Sub Ch,Ch
 Mov Cl,Al
 Inc Si                 ;skip over quote
 Rep
 Movsb                  ;move into buffer
 Pop Si

;----- successful, only the type is relevant

 Mov Ax,String          ;type
 Sub Bx,Bx              ;value
 Mov Cx,Bx              ;size
 Stc                    ;set carry
 Ret

;----- unsuccessful

Notstr
 Clc                    ;clear carry
 Ret
 Endp                   ;Str_Look

;===============================================;
;                   Sym_Look                    ;
; Symbol table look up. Checks if the operand   ;
; at SI is in the symbol table. If so the       ;
; carry is set and AX returns the type, BX      ;
; returns the value, and CX returns the size.   ;
; Handles all types of symbols including        ;
; symbols within macro definitions and          ;
; expansions. Clears special type bits.         ;
;===============================================;

Sym_Look Proc Near
 Test Mac_Stat,Mac_Flag Or Mac_Def ;check if in macro definition or expansion
 Jnz Symlmac            ;jump if so

Symlookc
 Push Es
 Call Seek_Sym          ;find symbol
 Jc Foundsym            ;jump if found
 Pop Es
 Ret

;----- check for local symbols first

Symlmac
 Cmp Pass, 2            ;check if pass two
 Je Symlmac2            ;jump if so, actual lookup

;----- local symbol lookup, pass one, only return type

 Call Mac_Dlabel        ;make it a definition macro label
 Push Es
 Call Seek_Sym          ;find symbol
 Pushf
 Call Mac_Unlabel       ;fix symbol
 Popf
 Jc Foundsym            ;jump if found
 Pop Es
 Jmps Symlookc

;----- local symbol lookup, pass two

Symlmac2
 Call Mac_Label         ;make it a macro label
 Push Es
 Call Seek_Sym          ;find symbol
 Pushf
 Call Mac_Unlabel       ;fix symbol
 Popf
 Jc Foundsym            ;jump if found
 Pop Es
 Jmps Symlookc

;----- symbol found

Foundsym
 Seg Es                 ;in symbol seg
 Or Word [Di+4],Used_Sym ;set access bit
 Seg Es                 ;in symbol seg
 Mov Bx,[Di]            ;value
 Seg Es                 ;in symbol seg
 Mov Cx,[Di+2]          ;size
 Seg Es                 ;in symbol seg
 Mov Ax,[Di+4]          ;type
 Pop Es                 ;restore ES
 Stc                    ;set carry, found
 Ret
 Endp                   ;Sym_Look

;===============================================;
;                   Mem_Look                    ;
; Evaluates a memory reference operand (an      ;
; address inside of brackets). AX returns the   ;
; type, BX returns the value, and CX returns    ;
; the size. If an addressing offset is          ;
; required, OFF_FLAG is set and OFF_VAL         ;
; returns the value.                            ;
;===============================================;

Mem_Look Proc Near
 Or Oprnd_Stat,Mem_Flag ;set memory operand flag
 Mov Mem_Store,'+'      ;initalize operator
 Mov Off_Val,0          ;initialize offset
 Mov Dh, Im_Con         ;DH holds the addressing bits, init. constant bit

;----- loop for each operand in brackets

Memilop
 Push Dx
 Push Fe_Start
 Call Next_Oprnd        ;get next operand
 Mov Si,Fe_Start        ;save operand start, in case of error
 Pop Fe_Start
 Pop Dx
 Jc Membpop             ;jump if operator returned

 Test Ax,Reg            ;test if register
 Jnz Memreg
 Test Ax,Immed          ;test if number
 Jnz Memnum
 Test Ax,Addr           ;test if address
 Jnz Memaddr
 Test Ax,None           ;test if nothing
 Jnz Memeerr
 Test Ax,Undef          ;see if undefined
 Jnz Nomemilld

;----- illegal memory operand

Memill
 Push Fe_Start
 Mov Fe_Start,Si        ;set start of operand
 Mov Ax,0167h           ;error 103
 Call Error             ;error routine
 Pop Fe_Start

Nomemilld
 Or Dh,Im_Off           ;make offset
 Jmps Memcont           ;next expression

;----- operator returned preceding next operand

Membpop
 Push Dx
 Mov Dl,Al
 Mov Ax,2019h           ;error 25
 Call Error             ;error routine
 Mov Al,Dl
 Pop Dx

 Cmp Al,']'             ;check if closing bracket
 Jne Memilop            ;jump if not
 Jmps Memset            ;finished

;---- no operand returned, missing bracket

Memeerr
 Mov Ax,0066h           ;error 102
 Call Error             ;error routine
 Jmps Memset            ;evaluate address

;----- immediate data

Memnum
 Call Chk_Mnum          ;evaluate immediate data
 Jmps Memcont

;----- direct address

Memaddr
 Call Chk_Mnum          ;evaluate immediate data
 Jmps Memcont

;----- addressing register

Memreg
 Mov Cx,4               ;four possibilities
 Mov Dl,Cl
 Mov Di,Offset Memop    ;translation table
 Mov Al,Bl              ;register value
 Repne
 Scasb
 Jne Memill             ;illegal value (not address register)

 Sub Dl,Cl
 Dec Dl                 ;DL gets addressing index
 Call Chk_Mreg          ;evaluate memory register

;----- next part of operand, get operator

Memcont
 Push Dx
 Push Fe_Start
 Push Fe_Stop
 Call Get_Oprnd         ;get operand
 Pop Di
 Pop Fe_Start
 Pop Dx
 Jnc Membadop           ;jump if field returned

 Cmp Al,']'             ;check if closing bracket
 Je Memset              ;jump if so
 Cmp Al,','             ;check if end of operand
 Je Membacom            ;jump if so

 Mov Mem_Store,Al       ;save for future reference
 Jmp Memilop            ;loop back to evaluate

;----- comma before end of operand

Membacom
 Mov Fe_Stop,Di         ;field end before the comma
 Or Oprnd_Stat,Oprnd_End ;set flag
 Jmps Memeerr

;----- illegal or missing operator, cancel memory operand

Membadop
 Jz Memeerr             ;jump if nothing
 Mov Si,Tempbuff        ;source
 Mov Ax,0462h           ;error 98
 Call Error             ;error routine

;----- set values according to flags in DH

Memset
 Mov Al,Dh
 Mov Ah,Im_Off          ;immediate data offset
 And Al,Not Im_Con      ;turn off constant bit if set
 Cmp Al,Ah              ;if only immediate data then direct address
 Je Memdir              ;jump if direct address

;----- not direct addressing, get particular addressing mode

 And Al,Not Im_Off      ;turn off immediate data bit if set
 Mov Cx,8               ;8 possibilities
 Mov Di,Offset Memdata  ;location of possiblities

 Repne
 Scasb                  ;scan

 Jne Memerror           ;jump if not found

 Mov Ax,Mem             ;type
 Mov Bx,Cx              ;value
 Sub Cx,Cx              ;size

 Test Dh,Im_Off         ;test for immed. (immed. becomes offset)
 Jnz Offmem             ;jump if there

;----- normal addressing, no offset

 Cmp Bl,6               ;test for special case [BP]
 Je Offmem              ;jump if so, make [BP+00]
 Ret

;----- normal addressing with offset

Offmem
 Or Line_Stat,Off_Flag  ;set offset flag
 Test Dh,Im_Con         ;check if constant
 Jz Nooffmemc           ;jump if not
 Or Line_Stat,Off_Cons  ;set constant flag
Nooffmemc Ret

;----- direct addressing

Memdir
 Mov Ax,Addr Or Mem     ;types
 Mov Bx,Off_Val         ;value
 Sub Cx,Cx              ;size
 Ret

;----- addressing error

Memerror
 Mov Ax,0068h           ;error 104
 Call Error             ;error routine
 Mov Ax,Mem             ;type
 Sub Bx,Bx              ;value
 Mov Cx,Bx              ;size
 Ret

;----- data, register addressing

;Si_Reg Equ Bit1         ;
;Di_Reg Equ Bit2         ;
;Bp_Reg Equ Bit3         ;--- set by equation
;Bx_Reg Equ Bit4         ;
Im_Off Equ Bit5         ;immediate data
Im_Con Equ Bit6         ;constant immdiate data (a literal number)

Memop Db 48,56,40,24    ;addressing register values:SI,DI,BP,BX
Memdata Db 8,4,2,1,6,5,10,9 ;BX,BP,DI,SI,BP+DI,BP+SI,BX+DI,BX+SI

;===============================================;
;                  Chk_Mreg                     ;
; Local routine of MEM_LOOK to evaluate the     ;
; memory address register in BX. Updates        ;
; address bit pattern in DH.                    ;
;===============================================;

Chk_Mreg Proc Near
 Mov Cl,Dl              ;value returned by translation
 Mov Dl,1
 Shl Dl,Cl              ;DL gets register flag bit setting
 Or Dh,Dl               ;put in DH
 Cmp Mem_Store,'+'      ;make sure used correct operator
 Jne Mreger             ;jump if incorrect
 Ret

;----- invalid operator

Mreger
 Mov Dl,Mem_Store       ;operator
 Mov Ax,2019h           ;error 25
 Call Error             ;error routine
 Ret
 Endp                   ;Chk_Mreg

;===============================================;
;                   Chk_Mnum                    ;
; A local routine of MEM_LOOK to evaluate a     ;
; number in BX. Updates address bit pattern in  ;
; DH.                                           ;
;===============================================;

Chk_Mnum Proc Near
 Test Ax,True_Const     ;check if literal number
 Jnz Nocmcon            ;jump if set

 And Dh,Not Im_Con      ;not literal number, clear bit

Nocmcon Or Dh,Im_Off    ;set flag for immediate data
 Cmp Mem_Store,'+'      ;check if operator was plus
 Jne Mnumne             ;jump if not

;----- add value

 Add Off_Val,Bx         ;add new value
 Ret

;----- subtract value

Mnumne
 Cmp Mem_Store,'-'      ;check if operator was minus
 Jne Mnumerr            ;jump if not

 Sub Off_Val,Bx         ;subtract new value
 Ret

;----- illegal operator (add value)

Mnumerr
 Add Off_Val,Bx         ;add value
 Mov Dl,Mem_Store       ;operator
 Mov Ax,2019h           ;error 25
 Call Error             ;error routine
 Ret
 Endp                   ;Chk_Mnum

 Endp                   ;Mem_Look

