////////////////////////////////////////////////////////////////////////////////
//
//  I/O Dialogs - Core File
//
//  (c) Copyright 2003,2004 Point Mad, Lukas Lipka. All rights reserved.
//
////////////////////////////////////////////////////////////////////////////////
#include "kernel.h"
#include "iodlg.h"
#include "label.h"
#include "window.h"
#include "button.h"
#include "textbox.h"
#include "combobox.h"
#include "fms.h"
#include "clipboard.h"
#include "listbox.h"
#include "checkbox.h"
////////////////////////////////////////////////////////////////////////////////
l_ulong AppVersion = ULONG_ID(0,0,0,1);
char    AppName[] = "I/O Dialogs";
l_uid	nUID = "iodlg";
l_uid NeededLibs[] = { "widget","listview","fms","textbox","combobox","button","window","clipbrd","listbox","label","checkbox","" };

p_bitmap ICONParentDir32	= NULL;
p_bitmap ICONParentDir16	= NULL;

////////////////////////////////////////////////////////////////////////////////
void FreeFileTypes ( PFileTypes o ) {
	if ( o->Title ) free(o->Title);
	if ( o->Extentions ) FreeList(o->Extentions);
	if ( o->Next ) FreeFileTypes(o->Next);
	free(o);
}
////////////////////////////////////////////////////////////////////////////////
PFileTypes NewFileTypes ( l_text Title, l_text txtexts, PFileTypes Next ) {

	PFileTypes o = malloc(sizeof(TFileTypes));
	if ( !o ) return NULL;
	memset(o,0,sizeof(TFileTypes));

	o->Title = TextDup(Title);

	if ( txtexts ) {
		l_text s,e = txtexts;
		o->Extentions = NewList();
		do {
			s = TextChr(e,',');
			if ( s ) {
				l_text n = TextNDup(e,s-e);
				ListAdd(o->Extentions,n,NULL,NULL);
				free(n);
				e = s+1;
			} else
			  ListAdd(o->Extentions,e,NULL,NULL);
		} while ( s );
	}
	o->Next = Next;
	return o;
}
////////////////////////////////////////////////////////////////////////////////
PFileTypes FileTypesAt ( PFileTypes o, l_ulong index ) {
	if ( !o ) return NULL;
	if ( !index ) return o;
	return FileTypesAt(o->Next,index-1);
}
////////////////////////////////////////////////////////////////////////////////
void DirviewItemFree(PDirviewItem o) {

	free(LISTVIEWITEM(o)->Caption);
	FreeFileInfo(o->f);

	free(o);
}
////////////////////////////////////////////////////////////////////////////////
PDirviewItem DirviewNewItem ( PDirview o )
{
	PDirviewItem p = malloc(sizeof(TDirviewItem));
	if (!p) return;
	memset(p, 0, sizeof(TDirviewItem));
	ListAdd (LISTVIEW(o)->Items, NULL, p, (void*)&DirviewItemFree);
	return p;
}
////////////////////////////////////////////////////////////////////////////////
PDirviewItem _DirviewAddItem ( PDirview o, PFileInfo f )
{
	PDirviewItem p = malloc(sizeof(TDirviewItem));
	if (!p) return;
	memset(p, 0, sizeof(TDirviewItem));

	/*if (f->Attributes & FA_DIREC ) {
		LISTVIEWITEM(p)->OnSelMsg = MSG_IODLG_ITEMDIR_SEL;
		LISTVIEWITEM(p)->OnValMsg = MSG_IODLG_ITEMDIR_VAL;
		LISTVIEWITEM(p)->OnRClkMsg = MSG_IODLG_ITEMDIR_MENU;
	} else {
		LISTVIEWITEM(p)->OnSelMsg = MSG_IODLG_ITEMFILE_SEL;
		LISTVIEWITEM(p)->OnValMsg = MSG_IODLG_ITEMFILE_VAL;
		LISTVIEWITEM(p)->OnRClkMsg = MSG_IODLG_ITEMFILE_MENU;
	}	*/

	FileGetAllThingsEx(f,NULL,&LISTVIEWITEM(p)->Icon16,&LISTVIEWITEM(p)->Icon32,NULL,NULL);

	LISTVIEWITEM(p)->Caption = TextDup(f->Name);

	p->f = f;

	ListAdd (LISTVIEW(o)->Items, f->Name, p, (void*)&DirviewItemFree);

	return p;
}
////////////////////////////////////////////////////////////////////////////////
l_int DirviewItemOrder ( PDirviewItem a, PDirviewItem b ) {

	if ( (a->f->Attributes & FA_DIREC) == (b->f->Attributes & FA_DIREC) )
		return TextCaseCompare(a->f->Name,b->f->Name);
	else if ( !(a->f->Attributes & FA_DIREC) && (b->f->Attributes & FA_DIREC) )
		return 1;
	else
		return -1;
}
////////////////////////////////////////////////////////////////////////////////
PDirviewItem DirviewAddItem ( PDirview o, PFileInfo f )
{
	PDirviewItem p = malloc(sizeof(TDirviewItem));
	if (!p) return;
	memset(p, 0, sizeof(TDirviewItem));
	FileGetAllThingsEx(f,NULL,&LISTVIEWITEM(p)->Icon16,&LISTVIEWITEM(p)->Icon32,NULL,NULL);
	LISTVIEWITEM(p)->Caption = TextDup(f->Name);
	p->f = f;

	ListAdd (LISTVIEW(o)->Items, f->Name, p, (void*)&DirviewItemFree);

	ListOrderByData(LISTVIEW(o)->Items,(void*)&DirviewItemOrder);

	return p;
}

////////////////////////////////////////////////////////////////////////////////
void DirviewGenerateSubs ( PDirview o ) {
  p_bitmap i16;
  p_bitmap i32;
  l_ulong  fi;

	PListItem a, b;
	PList l;

	ListviewRemoveAllItems(LISTVIEW(o));

	if ( !(o->Flags & DVF_NOPARICON) )
		if ( o->Directory[1] ) {
			PFileInfo pf = FileGetInfo(GetPathParent(o->Directory));
			PDirviewItem pi = DirviewNewItem(o);

			pi->f = pf;
			LISTVIEWITEM(pi)->Caption = TextDup("..");
			LISTVIEWITEM(pi)->Icon16 = ICONParentDir16;
			LISTVIEWITEM(pi)->Icon32 = ICONParentDir32;

		}

	l = ListDir(o->Directory,FA_DIREC|FA_LABEL);
	if ( l ) {

	if ( l->Last )
	{
		a = b = l->Last->Next;

		do
		{
			if ( FILEINFO(a->Data)->Attributes & FA_DIREC )
				_DirviewAddItem(o,FILEINFO(a->Data));
			a = a->Next;
		}
		while ( a != b );


		do
		{
			if ( !(FILEINFO(a->Data)->Attributes & FA_DIREC ))
			if (	o->Extentions ) {
				l_text ext = TextRChr(FILEINFO(a->Data)->Name,'.');
				if ( ext )
					if ( ListKeyCaseItem(o->Extentions,ext+1) ) _DirviewAddItem(o,FILEINFO(a->Data));
			} else
				_DirviewAddItem(o,FILEINFO(a->Data));
			a = a->Next;
		}
		while ( a != b );
	}
	KickList(l);
}

}
////////////////////////////////////////////////////////////////////////////////
void DirviewOpenDirectory ( PDirview o, l_text dir ) {
	if ( o->Directory ) free(o->Directory);
	o->Directory = TextDup(dir);

	DirviewGenerateSubs(o);

	WidgetDrawAll(WIDGET(o));
	if ( SLLWDG(o)->Vertical ) WidgetDraw(WIDGET(SLLWDG(o)->Vertical), NULL);
}
////////////////////////////////////////////////////////////////////////////////
void DirviewOpenParentDirectory ( PDirview o ) {
	l_text dir = GetPathParent(o->Directory);
	DirviewOpenDirectory(o,dir);
	free(dir);
}
////////////////////////////////////////////////////////////////////////////////
void DirviewRefreshData ( PDirview o ) {
	DirviewGenerateSubs(o);
	WidgetDraw(WIDGET(o),NULL);
}
////////////////////////////////////////////////////////////////////////////////
l_bool DirviewEventHandler ( PWidget o, PEvent Ev ) {




	if (Ev->Type == EV_KEYBOARD) {

		if ( Ev->Keyb.Shift & KB_CTRL ) {
			if ( SCANCODE(Ev,KEY_C) ) {
				PListviewItem p = GetSelectedItem(LISTVIEW(o));
				if ( p ) {
					PList l = GenerateFileRefs(DIRVIEWITEM(p)->f);
					if ( l )
						ClipboardPut(TYPE_FILEREFLIST,l,0,false);
					else
						MessageBox(&Me, "I/O", "Unable to copy files to clipboard.", 0);
				}


			}	else if ( SCANCODE(Ev,KEY_V) ) {

				/*PList l = ClipboardGet(TYPE_FILEREFLIST,NULL);
				if ( l ) {
					CopyFilesFromList ( l, DIRVIEW(o)->Directory );
					DirviewRefreshData(DIRVIEW(o));
				}*/

				WidgetSendSEvent(o, EV_MESSAGE, MSG_IOMENU_PASTE, Ev->Extra);

			}
			return true;
		}	else if ( SCANCODE(Ev,KEY_F5) ) {
			DirviewRefreshData(DIRVIEW(o));
			return true;
		}
	}
	if ( Ev->Type == EV_MESSAGE ) {
		if ( Ev->Message == MSG_IODLG_ITEMFILE_VAL ) {
			if ( DIRVIEWITEM(Ev->Extra)->f->Attributes & FA_DIREC ) {
				DirviewOpenDirectory(DIRVIEW(o),DIRVIEWITEM(Ev->Extra)->f->VPathName);
				return true;
			}
		}
		if ( Ev->Message == MSG_IODLG_OPENPARENT ) {
			DirviewOpenParentDirectory(DIRVIEW(o));
			return true;
		}
		if ( Ev->Message == MSG_IODLG_ITEMFILE_MENU ) {
			//l_text file = FileNameToPath(DIRVIEW(o)->Directory,LISTVIEWITEM(Ev->Extra)->Caption);
			PMenu m = FileGetMenuEx(DIRVIEWITEM(Ev->Extra)->f,0);

			l_ulong msg = PopUpMenuWait(o->AppOwner,Mouse->State.p,m,0);

			if ( msg == MSG_IOMENU_COPY ) {
				PList l = GenerateFileRefs(DIRVIEWITEM(Ev->Extra)->f);

				if ( l )
					ClipboardPut(TYPE_FILEREFLIST,l,0,false);
				else // System was unable to collect files, so we put only file reference
					MessageBox(&Me, "I/O", "Unable to copy files to clipboard.", 0);
			} else if ( msg == MSG_IOMENU_DELETE ) {
				PList l = GenerateFileRefs(DIRVIEWITEM(Ev->Extra)->f);

				if ( l ) { DeleteFilesFromList(l); FreeList(l); DirviewRefreshData(DIRVIEW(o)); }

			} else if ( msg == MSG_IOMENU_PASTE ) {
				PList l = ClipboardGet(TYPE_FILEREFLIST,NULL);
				if ( l ) {
					CopyFilesFromList ( l, DIRVIEWITEM(Ev->Extra)->f->VPathName );
					DirviewRefreshData(DIRVIEW(o));
				}
			} else if ( msg == MSG_IOMENU_OPEN ) {
				WidgetSendSEvent(o, EV_MESSAGE, MSG_IODLG_ITEMFILE_VAL, Ev->Extra);
			} else if ( msg == MSG_IOMENU_RENAME ) {

				l_text NewName = ListviewRenameItemBox(LISTVIEW(o),LISTVIEWITEM(Ev->Extra));

				if ( NewName ) {
					l_text NewVPathName = FileNameToPath(DIRVIEW(o)->Directory,NewName);

					l_bool r = false;

					if ( DIRVIEWITEM(Ev->Extra)->f->Attributes & FA_DIREC )
						r = DirRename(DIRVIEWITEM(Ev->Extra)->f->VPathName,NewVPathName);
					else
						r = FileRename(DIRVIEWITEM(Ev->Extra)->f->VPathName,NewVPathName);

					if ( r ) {
						FreeFileInfo(DIRVIEWITEM(Ev->Extra)->f);
						DIRVIEWITEM(Ev->Extra)->f = FileGetInfo(NewVPathName);
						free(LISTVIEWITEM(Ev->Extra)->Caption);
						LISTVIEWITEM(Ev->Extra)->Caption = TextDup(NewName);
						WidgetDraw(o,NULL);
						ScrollBarRedraw(SLLWDG(o));
					}
					free(NewVPathName);
					free(NewName);
				}


			} else if ( msg )
				FileHandleMenuMsgEx(DIRVIEWITEM(Ev->Extra)->f,msg);


			//free(file);
			return true;
		}


		if ( Ev->Message == MSG_IOMENU_PASTE ) {

			PList l = ClipboardGet(TYPE_FILEREFLIST,NULL);
			if ( l ) {
				CopyFilesFromList ( l, DIRVIEW(o)->Directory );
				DirviewRefreshData(DIRVIEW(o));
			}

			return true;
		}

		if ( Ev->Message == MSG_IODLG_NEWDIR ) {
			l_ulong nb = 0;
			l_text NewName;
			l_text Out = FileNameToPath(DIRVIEW(o)->Directory,"New directory");
			PFileInfo of;
			while ( of = FileGetInfo(Out) ) {
				free(Out);
				FreeFileInfo(of);
				nb++;
				NewName = TextArgs("New directory (%d)",nb);
				Out = FileNameToPath(DIRVIEW(o)->Directory,NewName);
				free(NewName);
			}
			MakeDir(Out);
			of = FileGetInfo(Out);
			free(Out);
			if ( of ) {
				DirviewAddItem(DIRVIEW(o),of);
				WidgetDraw(o,NULL);
				ScrollBarRedraw(SLLWDG(o));
			}
			return true;
		}

	}
	if ( ListviewEventHandler(o,Ev) ) return true;

	if ( Ev->Message == WEvMouseRUp ) {
		PFileInfo f = FileGetInfo(DIRVIEW(o)->Directory);
		if ( f ) {
			PMenu m = FileGetMenuEx(f,1);
			l_ulong msg = PopUpMenuWait(o->AppOwner,Mouse->State.p,m,0);

			if ( msg == MSG_IOMENU_PASTE || msg == MSG_IODLG_NEWDIR ) {
				/*PList l = ClipboardGet(TYPE_FILEREFLIST,NULL);
				if ( l ) {
					CopyFilesFromList ( l, DIRVIEW(o)->Directory );
					DirviewRefreshData(DIRVIEW(o));
				}		*/

				WidgetSendSEvent(o, EV_MESSAGE, msg, Ev->Extra);



			} else
			FileHandleMenuMsgEx(f,msg);
		}
		return true;
	}

	return false;
}
////////////////////////////////////////////////////////////////////////////////
void DirviewSetExtentions ( PDirview o, PList Extentions ) {
	o->Extentions = Extentions;
	DirviewGenerateSubs(o);
	WidgetDraw(WIDGET(o),NULL);
	ScrollBarRedraw(SLLWDG(o));
}
////////////////////////////////////////////////////////////////////////////////
PDirview CreateDirview ( PApplication App, TRect r, l_text dir, PList Extentions )
{
	PDirview o = malloc(sizeof(TDirview));

	if (!o) return NULL;
	memset(o, 0, sizeof(TDirview));
	InitListview(LISTVIEW(o), App, r);

	o->Directory = TextDup(dir);
	o->Extentions = Extentions;
	WIDGET(o)->EventHandler = DirviewEventHandler;

	DirviewGenerateSubs(o);

	LISTVIEW(o)->OnSelMsg = MSG_IODLG_ITEMFILE_SEL;
	LISTVIEW(o)->OnValMsg = MSG_IODLG_ITEMFILE_VAL;
	LISTVIEW(o)->OnRClkMsg = MSG_IODLG_ITEMFILE_MENU;

	return o;
}
////////////////////////////////////////////////////////////////////////////////
PTreeItem TreeDirViewOpenDirEx ( PTreeView t, PTreeItem i, l_text dir ) {

	i->Flags |= TIF_EXPENDED;

	if ( !i->Last )
		if ( i->GenerateSubs )
			i->GenerateSubs(i,i->Args);

	if ( i->Last ) {
		l_text S;
		PTreeItem a = i->Last;
		PTreeItem b = a;

		if ( *dir=='\\' || *dir == '/' ) dir++;

		if ( !(*dir) ) return i;
		S = dir;
  	while ( *S ) {
	  	if ( *S=='\\' || *S == '/' ) {
		  	do {
		   	  if ( !TextSqNCaseCompare(dir,a->Caption,S-dir) )
		   	  	return TreeDirViewOpenDirEx( t,a,S+1);
		    	a = a->Next;
	    	} while ( a != b );
	  	}
	  	S++;
  	}
  	do {
	  	if ( !TextCaseCompare(dir,a->Caption) ) return TreeDirViewOpenDirEx( t,a,"");
		  a = a->Next;
	  } while ( a != b );
	}
	return i;
}
////////////////////////////////////////////////////////////////////////////////
void TreeDirViewOpenDir ( PTreeDirview o, l_text dir ) {
	PTreeItem i = TreeDirViewOpenDirEx(TREEVIEW(o),TREEVIEW(o)->Items->Last,dir);
	TreeviewScrollBarRecalculate(TREEVIEW(o));
	if ( i ) TreeViewSelectItem(TREEVIEW(o),i);
}
////////////////////////////////////////////////////////////////////////////////
l_bool DirTreeViewEventHandler ( PWidget o, PEvent Ev ) {

	if ( Ev->Type == EV_MESSAGE ) {
		if ( Ev->Message == MSG_TREEVIEW_ITEM ) {

			if ( !TREEVIEW(o)->Sel ) return true;

			if ( TREEDIRVIEW(o)->Linked ) {
				DirviewOpenDirectory(TREEDIRVIEW(o)->Linked,TREEVIEW(o)->Sel->Args);
			}

			if ( TREEDIRVIEW(o)->Msg ) {
				if ( TREEDIRVIEW(o)->Msg != MSG_TREEVIEW_ITEM ) // Prevent infinite loops
					WidgetSendSEvent(o, EV_MESSAGE, TREEDIRVIEW(o)->Msg, TREEVIEW(o)->Sel);
			}

			return true;
		}
		if ( (Ev->Message == MSG_TREEVIEW_ITEM_MENU) && TREEVIEW(o)->Sel ) {
			PFileInfo f = FileGetInfo((l_text)TREEVIEW(o)->Sel->Args);
			if ( f ) {
				PMenu m = FileGetMenuEx(f,1);
				if ( m ) {
					l_ulong msg = PopUpMenuWait(o->AppOwner,Mouse->State.p,m,0);
					FileHandleMenuMsgEx(f,msg);
				}
			}
			return true;
		}
	}

	return TreeViewEventHandler(o,Ev);
}
////////////////////////////////////////////////////////////////////////////////
void DirTreeViewGenerateSubs ( PTreeItem o , void* Args )
{
	PListItem a, b;
	PList l = ListDir(Args,FA_DIREC|FA_LABEL);
	EmptySubTreeItems(o);
	if ( !l ) return;
	if ( l->Last ) {
		a = b = l->Last->Next;
		do {
	  	if ( FILEINFO(a->Data)->Attributes & FA_DIREC )
	  		AddTreeItem(o,FILEINFO(a->Data)->Name,FileGetIconEx(FILEINFO(a->Data),16,NULL),&DirTreeViewGenerateSubs,TextDup(FILEINFO(a->Data)->VPathName));
	  	a = a->Next;
  	} while ( a != b );
	}
	FreeList(l);
}
////////////////////////////////////////////////////////////////////////////////
PTreeDirview CreateTreeDirview ( PApplication App, TRect r ) {

	PTreeDirview o = malloc(sizeof(TTreeDirview));

	if ( !o ) return NULL;
	memset(o,0,sizeof(TTreeDirview));

	InitTreeView(TREEVIEW(o),App,r,MSG_TREEVIEW_ITEM);

	WIDGET(o)->EventHandler = &DirTreeViewEventHandler;


	TREEVIEW(o)->OnMenuMsg = MSG_TREEVIEW_ITEM_MENU;

	AddTreeItem(TREEVIEW(o)->Items,"Computer",LoadImage("./SYSTEM/ICONS/pc16.bmp"),&DirTreeViewGenerateSubs,"/");


	return o;
}
////////////////////////////////////////////////////////////////////////////////
#define CB_FILTER	0xFF00FFB0
#define TB_ADD		0xFF00FFB1
#define TB_REMOVE	0xFF00FFB2
#define TB_GOUP		0xFF00FFB4
#define LB_CHANGE	0xFF00FFB8

l_text IOBox ( l_text Title, l_ulong Flags, l_text Dir, PFileTypes Types, l_bool FExists )
{
	TRect r;
	PList DefExt = NULL;

	PWindow w;
	PButton b;
	PButton BAdd;
	PButton BRemove;
	PButton BUp;
	PListbox l;
	PDirview dir;
	PCheckbox c;
	PLabel t;

	PTextbox  FileName;
	PCombobox tsel;
	l_ulong    msg = 0;
	l_text    OpenFile = NULL;
	p_bitmap Icon = LoadImage("./SYSTEM/ICONS/folder16.bmp");

	if ( !Dir ) Dir = "/";

	RectAssign(&r, 0, 0, 600, 440);
	w = CreateWindow(&Me, r, Title, WF_CAPTION|WF_FRAME|WF_CENTERED|WF_MINIMIZE);
	InsertWidget(DeskTop, WIDGET(w));

	RectAssign(&r, 10, 10, 70, 40);
	BAdd = CreateButton(&Me,r,"Add",TB_ADD);
	BAdd->Icon = LoadImage("SYSTEM/ICONS/add.bmp");
	InsertWidget(WIDGET(w), WIDGET(BAdd));

	RectAssign(&r, 75, 10, 160, 40);
	BRemove = CreateButton(&Me,r,"Remove",TB_REMOVE);
	BRemove->Icon = LoadImage("SYSTEM/ICONS/remove.bmp");
	BRemove->Flags |= BF_DISABLED;
	InsertWidget(WIDGET(w), WIDGET(BRemove));

	RectAssign(&r, 165, 10, 215, 40);
	BUp = CreateButton(&Me,r,"Up",TB_GOUP);
	BUp->Icon = LoadImage("SYSTEM/ICONS/up.bmp");
	InsertWidget(WIDGET(w), WIDGET(BUp));

	RectAssign(&r, 10, 50, 215, 295);
	l = CreateListbox(&Me, r);
	l->AddItem(l, "Home", "/", Icon);
	l->AddItem(l, "Desktop", "/system/DESKTOP/", Icon);
	l->AddItem(l, "Floppy", "/a/", Icon);
	l->AddItem(l, "CD-ROM", "/d/", Icon);
	l->AddSeperator(l);
	l->AddItem(l, "Documents", NULL, Icon);
	l->OnSelect = LB_CHANGE;
	InsertWidget(WIDGET(w), WIDGET(l));

	RectAssign(&r, 225, 50, 590, 295);
	dir = CreateDirview(&Me, r, Dir, DefExt);
	LISTVIEW(dir)->Style = LVS_LIST;
	dir->Flags |= DVF_NOPARICON;
	InsertWidget(WIDGET(w), WIDGET(dir));

	RectAssign(&r, 10, 315, 80, 340);
	t = CreateLabel(&Me,r,"Filename:");
	WIDGET(t)->BackgroundColor = COL_3DFACE;
	InsertWidget(WIDGET(w), WIDGET(t));

	RectAssign(&r, 90, 315, 430, 340);
	FileName = CreateTextbox(&Me,r,TBF_EDITABLE);
	InsertWidget(WIDGET(w), WIDGET(FileName));

	RectAssign(&r, 440, 315, 590, 340);
	tsel = NewComboBox(&Me, r, CB_FILTER);
	InsertWidget(WIDGET(w), WIDGET(tsel));

	if ( Types )
	{
		PFileTypes ft = Types;

		while ( ft )
		{
			ComboboxAddItem(tsel,ft->Title,NULL,NULL);
			ft = ft->Next;
		}

		DefExt = Types->Extentions;
	}
	else
	{
		ComboboxAddItem(tsel, "All files (*.*)", NULL, NULL);
	}

	ComboboxSelectIndex(tsel, 1);

	RectAssign(&r, 10, 360, 430, 385);
	c = CreateCheckbox(&Me, r, "Lart whoever asks about this button");
	InsertWidget(WIDGET(w), WIDGET(c));

	RectAssign(&r, 405, 400, 490, 430);
	b = CreateButton(&Me,r,"Cancel",WM_CLOSE);
	b->Icon = LoadImage("./SYSTEM/ICONS/cancel.bmp");
	InsertWidget(WIDGET(w), WIDGET(b));

	RectAssign(&r ,505, 400, 590, 430);
	if ( Flags & IOBOX_SAVE )
	{
		b = CreateButton(&Me, r, "Save", MSG_IODLG_BOX_OK);
		b->Icon = LoadImage("./SYSTEM/ICONS/save.bmp");
	}
	else
	{
		b = CreateButton(&Me, r, "Open", MSG_IODLG_BOX_OK);
		b->Icon = LoadImage("./SYSTEM/ICONS/open.bmp");
	}

	InsertWidget(WIDGET(w), WIDGET(b));
	WidgetDrawAll(WIDGET(w));

	while ( (msg = WidgetExecute(WIDGET(w))) != WM_CLOSE )
	{
		switch ( msg )
		{
			case WM_CLOSE:
				break;
			break;

			case TB_GOUP:
				DirviewOpenParentDirectory(dir);
				TextBoxSetText(FileName, "");
			break;

			case LB_CHANGE:
				if ( ((PListboxItem)(l->SelectedRow)) )
				{
					if ( BRemove->Flags & BF_DISABLED )
					{
						BRemove->Flags &= ~BF_DISABLED;
						WidgetDraw(WIDGET(BRemove), NULL);
					}

					if ( ((PListboxItem)(l->SelectedRow->ItemList->Last->Data))->Data )
						DirviewOpenDirectory(dir, ((PListboxItem)(l->SelectedRow->ItemList->Last->Data))->Data);

				}
				else
				{
					BRemove->Flags |= BF_DISABLED;
					WidgetDraw(WIDGET(BRemove), NULL);
				}
			break;

			case MSG_IODLG_BOX_OK:
			{
				if ( TextCompare(FileName->Text, "") )
					OpenFile = FileNameToPath(dir->Directory,FileName->Text);
			}
			break;

			case CB_FILTER:
				if ( Types && tsel->Selected )
				{
					PFileTypes ft = FileTypesAt(Types,ComboboxItemIndex(tsel,tsel->Selected)-1);
					if ( ft ) DirviewSetExtentions(dir,ft->Extentions);
				}
			break;

			case MSG_IODLG_ITEMFILE_SEL:
			{
				PListviewItem a = GetSelectedItem(LISTVIEW(dir));
				if ( a )
				{
					if ( DIRVIEWITEM(a)->f->Attributes & FA_DIREC )
						TextBoxSetTextEx(FileName,"");
					else
						TextBoxSetTextEx(FileName,a->Caption);
				}
			}
			break;

			case MSG_IODLG_ITEMFILE_VAL:
			{
				PListviewItem a = GetSelectedItem(LISTVIEW(dir));
				if ( a )
				{
					if ( !(DIRVIEWITEM(a)->f->Attributes & FA_DIREC) )
					{
						OpenFile = FileNameToPath(dir->Directory,a->Caption);
					}
				}
			}
			break;
		}

		/**
		 * TODO: Clean this crap
		 */
		if ( OpenFile )
		{
			PFileInfo fi = FileGetInfo(OpenFile);

			if ( fi )
			{
				if ( fi->Attributes & FA_DIREC )
				{
					DirviewOpenDirectory(dir,OpenFile);
					TextBoxSetText(FileName,"");
					free(OpenFile);
					OpenFile = NULL;
					FreeFileInfo(fi);
				}
				else if ( !FExists )
				{
					l_ulong msg = MessageBox(&Me,"Warning", "File already exists. Do you want to replace it?", MBB_YESNOCANCEL|MBI_WARNING);
					FreeFileInfo(fi);
					if ( msg == MSG_YES ) break;
					free(OpenFile);
					OpenFile = NULL;
					if ( msg == MSG_CANCEL ) break;
				}
				else
				{
					FreeFileInfo(fi);
					break;
				}
			}
			else
			{
				if ( FExists )
				{
					free(OpenFile);
					OpenFile = NULL;
				}
				else break;
			}
		}
	}

	WidgetDispose(WIDGET(w));
	DestroyImage(Icon);

	return OpenFile;
}
////////////////////////////////////////////////////////////////////////////////
l_bool CollectCallBack ( void* id, l_text curfile ) {

	SysPoll();

	if ( WIDGET(PROGRESSPOP(id)->W)->MessageBuffer == WM_CLOSE ) {
	 	ProgressPopSetInfo((PProgressPop)id,"Stopping...");
		return false;
 	} else
 		ProgressPopSetInfo((PProgressPop)id,"Adding %s",curfile);

	return true;
}
////////////////////////////////////////////////////////////////////////////////
PList GenerateFileRefs ( PFileInfo f ) {
	PList Lst = NewList();
	PProgressPop p;
	p = ProgressPopUp(&Me,1,"I/O","Please wait. Collecting files...",0,"");
 	if ( !_GenerateFileRefs( Lst, f, NULL, &CollectCallBack, p ) ) {
	 	FreeList(Lst);
 	 	ProgressPopKick(p);
	 	return NULL;
 	}
 	ProgressPopKick(p);
	return Lst;
}
////////////////////////////////////////////////////////////////////////////////
l_bool FileCopier ( PProgressPop p, PFile s, PFile d ) {
	l_char Buffer[4096];
	l_int r = 0;
	while ( !FileEOF(s) ) {
		r = FileRead(&Buffer,1,4096,s);
		SysPoll();
		FileWrite(&Buffer,1,r,d);
		ProgressPopStep(p,r/1024);
		SysPoll();
		if ( WIDGET(p->W)->MessageBuffer == WM_CLOSE ) ProgressPopSetInfo(p,"Stopping...");
	}
	if ( WIDGET(p->W)->MessageBuffer == WM_CLOSE ) return false;
	return true;
}
////////////////////////////////////////////////////////////////////////////////
void CopyFilesFromList ( PList l, l_text Dest ) {
if ( l->Last ) {
	l_ulong GlobalSize = 0;
	PProgressPop p;
	PListItem a = l->Last->Next, b = a;
	l_bool Ignore = false, ReplaceAll = false;
	l_ulong SamePlace = 0;

	do {
		PFileRef r = FILEREF(a->Data);

		if ( r->Attributes & FA_DIREC )
			GlobalSize += 1;
		else
			GlobalSize += r->Size/1024;

		a = a->Next;
	} while ( a != b );

	p = ProgressPopUp(&Me,7,"I/O","Please wait. Copying files...",GlobalSize,"...");

	do {
		PFileRef r = FILEREF(a->Data);
		l_text Out = SamePlace ? NULL : FileNameToPath(Dest,r->SName);
		PFileInfo of = ( ReplaceAll || SamePlace ) ? NULL : FileGetInfo(Out);
		ProgressPopSetInfo(p,"%s",r->SName);
		SysPoll();

		Ignore = false;

		if ( of ) {

			if ( of->Drive == r->Drive && !TextCompare(of->LocalName,r->LocalName) ) {
				l_text NewSName = TextArgs("Copy of %s",r->SName);
				free(Out);
				FreeFileInfo(of);
				SamePlace = 1;
				Out = FileNameToPath(Dest,NewSName);
				free(NewSName);
				while ( of = FileGetInfo(Out) ) {
					free(Out);
					FreeFileInfo(of);
					NewSName = TextArgs("Copy(%d) of %s",SamePlace-1,r->SName);
					Out = FileNameToPath(Dest,NewSName);
					free(NewSName);
					SamePlace++;
				}
			} else {

				l_text text = TextArgs("File %s (%d bytes) yet exists.\nDo you want to replace it with %s (%d bytes) ?",Out,of->Size,r->SName,r->Size);
				l_ulong msg = MessageBox(&Me,"I/O",text,MBB_YESYESALLNOCANCEL);

				if ( msg == MSG_CANCEL ) break;
				if ( msg == MSG_NO ) Ignore = true;
				if ( msg == MSG_YESALL ) ReplaceAll = true;

				free(text);
				FreeFileInfo(of);

			}
		}

		if ( !Ignore ) {

		if ( SamePlace ) {
			l_text NewSName;
			if ( SamePlace > 1 )
				NewSName = TextArgs("Copy(%d) of %s",SamePlace-1,r->SName);
			else
				NewSName = TextArgs("Copy of %s",r->SName);
			Out = FileNameToPath(Dest,NewSName);
		}


		if ( r->Attributes & FA_DIREC ) {
			MakeDir(Out);
			SysPoll();
			ProgressPopStep(p,1);
		} else if ( r->Drive->Driver->FileOpen ) {
			PFile s = r->Drive->Driver->FileOpen(r->Drive,r->LocalName,"rb");
			PFile d = FileOpen(Out,"wb");
			if ( d && s ) {
				FileCopier(p,s,d);
			}	else
				ProgressPopStep(p,r->Size/1024);
			if ( s ) FileClose(s);
			if ( d ) FileClose(d);
		}

		}
		free(Out);
		if ( WIDGET(p->W)->MessageBuffer == WM_CLOSE ) break;

		a = a->Next;
	} while ( a != b );

 	ProgressPopKick(p);

	}
}
////////////////////////////////////////////////////////////////////////////////
void DeleteFilesFromList ( PList l ) {

	l_ulong Nb = 0;
	PProgressPop p;
	PListItem a = l->Last, b = a;

	do {
		Nb++;
		a = a->Prev;
	} while ( a != b );

	p = ProgressPopUp(&Me,7,"I/O","Please wait. Removing files...",Nb,"...");

	do {
		PFileRef r = FILEREF(a->Data);

		ProgressPopSetInfo(p,"%s",r->SName);

		SysPoll();

		if ( r->Attributes & FA_DIREC ) {
			if ( r->Drive->Driver->DirDelete )
				r->Drive->Driver->DirDelete(r->Drive,r->LocalName);
		} else {
			if ( r->Drive->Driver->FileDelete )
				r->Drive->Driver->FileDelete(r->Drive,r->LocalName);
		}

		if ( WIDGET(p->W)->MessageBuffer == WM_CLOSE ) break;


		ProgressPopStep(p,1);
		a = a->Prev;
	} while ( a != b );

	ProgressPopKick(p);

}
////////////////////////////////////////////////////////////////////////////////
l_bool LibMain (l_text Args)
{
	ICONParentDir32 = LoadImage("SYSTEM/ICONS/folpar32.bmp");
	ICONParentDir16 = LoadImage("SYSTEM/ICONS/folpar16.bmp");

	APPEXPORT(IOBox);
	APPEXPORT(CreateDirview);
	APPEXPORT(DirviewOpenDirectory);
	APPEXPORT(TreeDirViewOpenDir);
	APPEXPORT(FreeFileTypes);
	APPEXPORT(NewFileTypes);
	APPEXPORT(DirviewOpenParentDirectory);//
	APPEXPORT(CreateTreeDirview);
	/*MouseShow();
	IOBox ( "Select a file", "Open", "/", NULL, true );*/

	return true;
}
////////////////////////////////////////////////////////////////////////////////
void Close (void)
{

}
////////////////////////////////////////////////////////////////////////////////
