/*
 * ipcfg - the configuration interface for the picotcp4dos environment
 * Copyright (C) 2015 Mateusz Viste
 *
 * http://picotcp4dos.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <stdio.h>
#include <stdlib.h>  /* getenv() */
#include <string.h>  /* memcpy() */
#include <stdint.h>  /* uint16_t, uint32_t... */

#include "parsecli.h"
#include "dhcp.h"
#include "picoconf.h"
#include "..\version.h"


/* the structure below is used to store the whole network configuration */
struct ipcfg_config {
  struct iplist {
    unsigned char ipaddr[16];
    unsigned char masklen;
    struct iplist *next;
  } *iplist;
  struct dnslist {
    unsigned char ipaddr[16];
    struct dnslist *next;
  } *dnslist;
  struct rtlist {
    unsigned char prefix[16];
    unsigned char gw[16];
    unsigned char prefixlen;
    struct rtlist *next;
  } *rtlist;
  int pktint;
  int changedflag;
};


/******** configuration-handling routines ***********************************/


static int addnode_iplist(struct ipcfg_config *cfg, unsigned char *addr, int masklen) {
  struct iplist *node, *parent;
  node = calloc(1, sizeof(struct iplist));
  if (node == NULL) return(-1);
  /* populate the new node with data and attach to the list */
  memcpy(node->ipaddr, addr, 16);
  node->masklen = masklen;
  /* find the last node in list and attach the new node */
  if (cfg->iplist != NULL) {
    for (parent = cfg->iplist; parent->next != NULL; parent = parent->next);
    parent->next = node;
  } else {
    cfg->iplist = node;
  }
  /* return */
  return(0);
}


static int addnode_rtlist(struct ipcfg_config *cfg, unsigned char *addr, int prefixlen, unsigned char *gw) {
  struct rtlist *node, *parent;
  node = calloc(1, sizeof(struct rtlist));
  if (node == NULL) return(-1);
  /* populate the new node with data and attach to the list */
  memcpy(node->prefix, addr, 16);
  memcpy(node->gw, gw, 16);
  node->prefixlen = prefixlen;
  /* find the last node in list and attach the new node */
  if (cfg->rtlist != NULL) {
    for (parent = cfg->rtlist; parent->next != NULL; parent = parent->next);
    parent->next = node;
  } else {
    cfg->rtlist = node;
  }
  /* return */
  return(0);
}


static int addnode_dnslist(struct ipcfg_config *cfg, unsigned char *addr) {
  struct dnslist *node, *parent;
  node = calloc(1, sizeof(struct dnslist));
  if (node == NULL) return(-1);
  /* populate the new node with data and attach to the list */
  memcpy(node->ipaddr, addr, 16);
  /* find the last node in list and attach the new node */
  if (cfg->dnslist != NULL) {
    for (parent = cfg->dnslist; parent->next != NULL; parent = parent->next);
    parent->next = node;
  } else {
    cfg->dnslist = node;
  }
  /* return */
  return(0);
}


static void pico_freeconf_iplist(struct ipcfg_config *cfg) {
  while (cfg->iplist != NULL) {
    struct iplist *victim = cfg->iplist;
    cfg->iplist = cfg->iplist->next;
    free(victim);
  }
}


static void pico_freeconf_dnslist(struct ipcfg_config *cfg) {
  while (cfg->dnslist != NULL) {
    struct dnslist *victim = cfg->dnslist;
    cfg->dnslist = cfg->dnslist->next;
    free(victim);
  }
}


static void pico_freeconf_rtlist(struct ipcfg_config *cfg) {
  while (cfg->rtlist != NULL) {
    struct rtlist *victim = cfg->rtlist;
    cfg->rtlist = cfg->rtlist->next;
    free(victim);
  }
}


/* free the ipcfg config structure, along with all its children */
static void pico_freeconf(struct ipcfg_config *cfg) {
  pico_freeconf_iplist(cfg);
  pico_freeconf_dnslist(cfg);
  pico_freeconf_rtlist(cfg);
  free(cfg);
}


/* checks the validity of the %PICOTCPCFG% variable and loads the configuration */
static struct ipcfg_config *pico_loadconf(void) {
  struct ipcfg_config *res;
  FILE *fd;
  char *e;
  int count, masklen;
  unsigned char addr1[16], addr2[16];

  e = getenv("PICOTCP");
  /* the env. variable must exist */
  if ((e == NULL) || (e[0] == 0)) {
    return(NULL);
  }
  /* open file */
  fd = picoconf_open(e);
  if (fd == NULL) return(NULL);
  /* allocate a result struct and populate it with the config */
  res = calloc(1, sizeof(struct ipcfg_config));
  if (res == NULL) {
    picoconf_close(fd);
    return(NULL);
  }
  /* fetch the number of interfaces (must be 1 for now) */
  count = picoconf_getint(fd, NULL);
  picoconf_getint(fd, &(res->pktint));
  /* fetch list of IP addresses (and masklen) */
  count = picoconf_getip(fd, NULL, NULL);
  if (count < 0) {
    pico_freeconf(res);
    picoconf_close(fd);
    return(NULL);
  }
  while (count-- > 0) {
    picoconf_getip(fd, addr1, &masklen);
    addnode_iplist(res, addr1, masklen);
  }
  /* fetch list of routes */
  count = picoconf_getrt(fd, NULL, NULL, NULL);
  if (count < 0) {
    pico_freeconf(res);
    picoconf_close(fd);
    return(NULL);
  }
  while (count-- > 0) {
    picoconf_getrt(fd, addr1, &masklen, addr2);
    addnode_rtlist(res, addr1, masklen, addr2);
  }
  /* fetch list of DNS servers */
  count = picoconf_getdns(fd, NULL);
  if (count < 0) {
    pico_freeconf(res);
    picoconf_close(fd);
    return(NULL);
  }
  while (count-- > 0) {
    picoconf_getdns(fd, addr1);
    addnode_dnslist(res, addr1);
  }
  /* close the config file and return result */
  picoconf_close(fd);
  return(res);
}


static void writeword(FILE *fd, unsigned int val) {
  fputc((val >> 8) & 0xff, fd);
  fputc(val & 0xff, fd);
}


static int pico_saveconf(struct ipcfg_config *cfg) {
  int count;
  unsigned int offset;
  char *e;
  FILE *fd;
  struct iplist *ipnode;
  struct rtlist *rtnode;
  struct dnslist *dnsnode;

  e = getenv("PICOTCP");
  fd = fopen(e, "wb");
  if (fd == NULL) return(-1);
  /* save header */
  offset = fwrite("pTCPxxxx\1xxxx", 1, 13, fd);

  /* save interface pktdrv, and offset for IP addresses */
  fseek(fd, 0x09, SEEK_SET);
  writeword(fd, cfg->pktint);
  writeword(fd, offset);
  fseek(fd, offset, SEEK_SET);

  /* write list of IP addresses (and masklen) */
  count = 0;
  fputc(0, fd);
  for (ipnode = cfg->iplist; ipnode != NULL; ipnode = ipnode->next) {
    count++;
    fwrite(ipnode->ipaddr, 1, 16, fd);
    fputc(ipnode->masklen, fd);
  }
  fseek(fd, offset, SEEK_SET);
  fputc(count, fd);

  /* write list of routes */
  fseek(fd, 0, SEEK_END);
  offset = ftell(fd);
  count = 0;
  fputc(0, fd);
  for (rtnode = cfg->rtlist; rtnode != NULL; rtnode = rtnode->next) {
    count++;
    fwrite(rtnode->prefix, 1, 16, fd);
    fputc(rtnode->prefixlen, fd);
    fwrite(rtnode->gw, 1, 16, fd);
  }
  fseek(fd, offset, SEEK_SET);
  fputc(count, fd);
  fseek(fd, 0x04, SEEK_SET);
  writeword(fd, offset);

  /* write list of DNS servers */
  fseek(fd, 0, SEEK_END);
  offset = ftell(fd);
  count = 0;
  fputc(0, fd);
  for (dnsnode = cfg->dnslist; dnsnode != NULL; dnsnode = dnsnode->next) {
    count++;
    fwrite(dnsnode->ipaddr, 1, 16, fd);
  }
  fseek(fd, offset, SEEK_SET);
  fputc(count, fd);
  fseek(fd, 0x06, SEEK_SET);
  writeword(fd, offset);

  /* close file and return */
  fclose(fd);
  return(0);
}


/* converts an IP address to humanly readable string */
static char *ip2string(char *buff, unsigned char *ip) {
  uint16_t *uip = (uint16_t *)ip;
  if ((uip[0] == 0) && (uip[1] == 0) && (uip[2] == 0) && (uip[3] == 0) && (uip[4] == 0) && (uip[5] == 0xffffu)) {
    /* IPv4 */
    sprintf(buff, "%d.%d.%d.%d", ip[12], ip[13], ip[14], ip[15]);
  } else { /* IPv6 */
    int x, offset = 0;
    for (x = 0; x < 16; x += 2) {
      offset += sprintf(buff + offset, "%x:", ((long)ip[x] << 8) | ip[x+1]);
    }
    buff[offset - 1] = 0;
  }
  return(buff);
}


/* compares two IP address for equality, zero if equal, non-zero otherwise */
static int ipcmp(unsigned char *ip1, unsigned char *ip2) {
  uint32_t *p1 = (uint32_t *)ip1;
  uint32_t *p2 = (uint32_t *)ip2;
  if ((p1[0] != p2[0]) || (p1[1] != p2[1]) || (p1[2] != p2[2]) || (p1[3] != p2[3])) return(-1);
  return(0);
}


/******** all cmd functions below *******************************************/


static int cmd_help(char *msg) {
  if (msg != NULL) {
    printf("ERROR: %s\n       Run ipcfg /? for help", msg);
    return(-1);
  }
  printf("picoTCP config tool [ver " PVER "] / Copyright (C) " PDATE " Mateusz Viste\n"
         PURL "\n"
         "\n"
         "Possible invocations are listed below. Wherever an IP address is expected,\n"
         "both IPv4 and IPv6 addresses are supported.\n"
         "\n"
         "ipcfg int                     shows the pkt driver interrupt used by picoTCP\n"
         "ipcfg int xx                  binds picotcp to pkt driver at INT xxh (eg. 60)\n"
         "ipcfg ip                      lists all configured IP addresses\n"
         "ipcfg ip add xxxx/xx          adds a new IPv4 or IPv6 address\n"
         "ipcfg ip del xxxx/xx          deconfigures an IPv4 or IPv6 address\n"
         "ipcfg dhcp                    autoconfigures the interface using DHCP\n"
         "ipcfg dns                     lists all configured DNS server\n"
         "ipcfg dns add xxxx            adds a new DNS server\n"
         "ipcfg dns del xxxx            removes the specified DNS server\n"
         "ipcfg route                   lists all currently configured static routes\n"
         "ipcfg route add xxxx/xx xxxx  adds a new static route\n"
         "ipcfg route del xxxx/xx       removes the route to a given destination\n"
         "\n"
         "A few common examples:   ipcfg ip add 192.168.0.5/24\n"
         "                         ipcfg ip route add 0.0.0.0/0 192.168.0.1\n"
         "                         ipcfg ip dns add 2001:0DB8:1985::8888\n"
         "\n");
  return(0);
}


static int cmd_int_show(struct ipcfg_config *cfg) {
  printf("Currently used packet driver interrupt: 0x%02X\n", cfg->pktint);
  return(0);
}


static int cmd_int_set(struct ipcfg_config *cfg, int pktint) {
  cfg->pktint = pktint;
  cfg->changedflag = 1;
  return(0);
}


static int cmd_ip_show(struct ipcfg_config *cfg) {
  char sbuff[64];
  struct iplist *ipnode;
  for (ipnode = cfg->iplist; ipnode != NULL; ipnode = ipnode->next) {
    printf("%s/%d\n", ip2string(sbuff, ipnode->ipaddr), ipnode->masklen);
  }
  return(0);
}


static int cmd_ip_add(struct ipcfg_config *cfg, unsigned char *ipaddr, int masklen) {
  struct iplist *node;
  /* look for duplicates first */
  for (node = cfg->iplist; node != NULL; node = node->next) {
    if (ipcmp(node->ipaddr, ipaddr) == 0) {
      printf("ERROR: address already exists\n");
      return(-1);
    }
  }
  /* add the new address */
  addnode_iplist(cfg, ipaddr, masklen);
  cfg->changedflag = 1;
  return(0);
}


static int cmd_ip_del(struct ipcfg_config *cfg, unsigned char *ipaddr, int masklen) {
  struct iplist *victim, *parent = NULL;
  /* find the record */
  for (victim = cfg->iplist; victim != NULL; victim = victim->next) {
    if ((ipcmp(victim->ipaddr, ipaddr) == 0) && (victim->masklen == masklen)) break;
    parent = victim;
  }
  if (victim == NULL) {
    printf("ERROR: address not found\n");
    return(-1);
  }
  /* remove the address */
  if (parent == NULL) {
    cfg->iplist = victim->next;
  } else {
    parent->next = victim->next;
  }
  free(victim);
  /* mark config as changed and return */
  cfg->changedflag = 1;
  return(0);
}


static int cmd_dns_show(struct ipcfg_config *cfg) {
  char sbuff[64];
  struct dnslist *node;
  for (node = cfg->dnslist; node != NULL; node = node->next) {
    printf("%s\n", ip2string(sbuff, node->ipaddr));
  }
  return(0);
}


static int cmd_dns_add(struct ipcfg_config *cfg, unsigned char *ipaddr) {
  struct dnslist *node;
  /* look for duplicates first */
  for (node = cfg->dnslist; node != NULL; node = node->next) {
    if (ipcmp(node->ipaddr, ipaddr) == 0) {
      printf("ERROR: address already exists\n");
      return(-1);
    }
  }
  /* add the new address */
  addnode_dnslist(cfg, ipaddr);
  cfg->changedflag = 1;
  return(0);
}


static int cmd_dns_del(struct ipcfg_config *cfg, unsigned char *ipaddr) {
  struct dnslist *victim, *parent = NULL;
  /* find the record */
  for (victim = cfg->dnslist; victim != NULL; victim = victim->next) {
    if (ipcmp(victim->ipaddr, ipaddr) == 0) break;
    parent = victim;
  }
  if (victim == NULL) {
    printf("ERROR: address not found\n");
    return(-1);
  }
  /* remove the address */
  if (parent == NULL) {
    cfg->dnslist = victim->next;
  } else {
    parent->next = victim->next;
  }
  free(victim);
  /* mark config as changed and return */
  cfg->changedflag = 1;
  return(0);
}


static int cmd_route_show(struct ipcfg_config *cfg) {
  char sbuff[128];
  struct rtlist *rtnode;
  for (rtnode = cfg->rtlist; rtnode != NULL; rtnode = rtnode->next) {
    printf("%s/%d -> %s\n", ip2string(sbuff, rtnode->prefix), rtnode->prefixlen, ip2string(sbuff + 64, rtnode->gw));
  }
  return(0);
}


static int cmd_route_add(struct ipcfg_config *cfg, unsigned char *prefix, int prefixlen, unsigned char *gw) {
  struct rtlist *node;
  /* look for duplicates first */
  for (node = cfg->rtlist; node != NULL; node = node->next) {
    if ((ipcmp(node->prefix, prefix) == 0) && (node->prefixlen == prefixlen)) {
      printf("ERROR: route already exists\n");
      return(-1);
    }
  }
  /* add the new route */
  addnode_rtlist(cfg, prefix, prefixlen, gw);
  cfg->changedflag = 1;
  return(0);
}


static int cmd_route_del(struct ipcfg_config *cfg, unsigned char *prefix, int prefixlen) {
  struct rtlist *victim, *parent = NULL;
  /* find the record */
  for (victim = cfg->rtlist; victim != NULL; victim = victim->next) {
    if ((ipcmp(victim->prefix, prefix) == 0) && (victim->prefixlen == prefixlen)) break;
    parent = victim;
  }
  if (victim == NULL) {
    printf("ERROR: route not found\n");
    return(-1);
  }
  /* remove the address */
  if (parent == NULL) {
    cfg->rtlist = victim->next;
  } else {
    parent->next = victim->next;
  }
  free(victim);
  /* mark config as changed and return */
  cfg->changedflag = 1;
  return(0);
}


/******** end of cmd functions **********************************************/


int main(int argc, char **argv) {
  struct membuffer {
    unsigned char arg1[16];
    unsigned char arg2[16];
    unsigned char dhcp_ip[16];
    unsigned char dhcp_gw[16];
    unsigned char dhcp_dns1[16];
    unsigned char dhcp_dns2[16];
    int dhcp_masklen;
    char tmpbuff[64];
  } *buff;
  char *errmsg;
  int argint;
  int rescode;
  enum command_t command;
  struct ipcfg_config *pcfg;

  /* verify that a %PICOTCP% variable is present and points to a real directory */
  pcfg = pico_loadconf();
  if (pcfg == NULL) {
    printf("ERROR: %%PICOTCP%% is either not set or points to an invalid location.\n"
           "       Please set %%PICOTCP%% first. Example: SET PICOTCP=C:\\PTCP.DAT\n");
    return(1);
  }

  /* allocate memory buffer */
  buff = malloc(sizeof(struct membuffer));
  if (buff == NULL) {
    printf("ERROR: out of memory\n");
    return(1);
  }

  /* parse and validate command line arguments */
  command = parsecli(buff->arg1, buff->arg2, &argint, &errmsg, argc, argv);

  /* execute proper action */
  switch (command) {
    case CMD_HELP:
      rescode = cmd_help(errmsg);
      break;
    case CMD_INT_SHOW:
      rescode = cmd_int_show(pcfg);
      break;
    case CMD_INT_SET:
      rescode = cmd_int_set(pcfg, argint);
      break;
    case CMD_IP_SHOW:
      rescode = cmd_ip_show(pcfg);
      break;
    case CMD_IP_ADD:
      rescode = cmd_ip_add(pcfg, buff->arg1, argint);
      break;
    case CMD_IP_DEL:
      rescode = cmd_ip_del(pcfg, buff->arg1, argint);
      break;
    case CMD_DHCP:
      buff->dhcp_masklen = dhcp_get(buff->dhcp_ip, buff->dhcp_gw, buff->dhcp_dns1, buff->dhcp_dns2);
      if (buff->dhcp_masklen == -3) { /* dhcp timeout */
        printf("ERROR: DHCP failure (timeout)\n", buff->dhcp_masklen);
      } else if (buff->dhcp_masklen < 0) { /* other dhcp error */
        printf("ERROR: DHCP failure (%d)\n", buff->dhcp_masklen);
      } else {
        /* clear config */
        pico_freeconf_iplist(pcfg);
        pico_freeconf_dnslist(pcfg);
        pico_freeconf_rtlist(pcfg);
        /* add ip/mask */
        cmd_ip_add(pcfg, buff->dhcp_ip, buff->dhcp_masklen);
        /* add gw */
        if (buff->dhcp_gw[12] != 0) {
          cmd_route_add(pcfg, (unsigned char *)"\0\0\0\0\0\0\0\0\0\0\xff\xff\0\0\0\0", 0, buff->dhcp_gw);
        }
        /* add dns servers */
        if (buff->dhcp_dns1[12] != 0) cmd_dns_add(pcfg, buff->dhcp_dns1);
        if (buff->dhcp_dns2[12] != 0) cmd_dns_add(pcfg, buff->dhcp_dns2);
        printf("DHCP configuration fetched: %s/%d", ip2string(buff->tmpbuff, buff->dhcp_ip), buff->dhcp_masklen);
      }
      break;
    case CMD_DNS_SHOW:
      rescode = cmd_dns_show(pcfg);
      break;
    case CMD_DNS_ADD:
      rescode = cmd_dns_add(pcfg, buff->arg1);
      break;
    case CMD_DNS_DEL:
      rescode = cmd_dns_del(pcfg, buff->arg1);
      break;
    case CMD_ROUTE_SHOW:
      rescode = cmd_route_show(pcfg);
      break;
    case CMD_ROUTE_ADD:
      rescode = cmd_route_add(pcfg, buff->arg1, argint, buff->arg2);
      break;
    case CMD_ROUTE_DEL:
      rescode = cmd_route_del(pcfg, buff->arg1, argint);
      break;
    default:
      printf("CODEFLOW ERROR\n");
      rescode = -1;
      break;
  }

  /* if configuration has been changed, save it */
  if (pcfg->changedflag != 0) pico_saveconf(pcfg);

  /* free memory and quit */
  pico_freeconf(pcfg);
  free(buff);

  if (rescode < 0) return(0 - rescode);
  return(rescode);
}
