/*
 * Library to access OPL2/OPL3 hardware (YM3812 / YMF262)
 *
 * Copyright (C) 2015-2024 Mateusz Viste
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef OPL

#include <conio.h> /* inp(), out() */
#include <stdlib.h> /* calloc() */
#include <string.h> /* strdup() */

#include "opl.h"

#include "fio.h"
#include "opl-gm.h"
#include "timer.h"

struct voicealloc_t {
  unsigned short priority;
  signed short timbreid;
  signed char channel;
  signed char note;
};

struct oplstate_t {
  signed char notes2voices[16][128];    /* keeps the map of channel:notes -> voice allocations */
  signed short channelpitch[16];        /* per-channel pitch level (-128..127) */
  unsigned short channelbendrange[16];  /* Pitch Bend Range (Pitch Bend Sensitivity), in semitones */
  unsigned short channelvol[16];        /* per-channel volume level */
  struct voicealloc_t voices2notes[18]; /* keeps the map of what voice is playing what note/channel currently */
  unsigned char channelprog[16];        /* programs (patches) assigned to channels */
  int opl3; /* flag indicating whether or not the sound module is OPL3-compatible or only OPL2 */
};

struct oplstate_t *oplmem = NULL; /* memory area holding all OPL's current states */

const unsigned short freqtable[128] = {                          /* note # */
        345, 365, 387, 410, 435, 460, 488, 517, 547, 580, 615, 651,  /*  0 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 12 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 24 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 36 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 48 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 60 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 72 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 84 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,  /* 96 */
        690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651, /* 108 */
        690, 731, 774, 820, 869, 921, 975, 517};                    /* 120 */

const unsigned char octavetable[128] = {                         /* note # */
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                          /*  0 */
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,                          /* 12 */
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1,                          /* 24 */
        1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2,                          /* 36 */
        2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3,                          /* 48 */
        3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4,                          /* 60 */
        4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5,                          /* 72 */
        5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6,                          /* 84 */
        6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7,                          /* 96 */
        7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,                         /* 108 */
        7, 7, 7, 7, 7, 7, 7, 7};                                    /* 120 */


/* tables below provide register offsets for each voice. note, that these are
 * NOT the registers IDs, but their direct offsets instead - this for simpler
 * and faster computations. */
const unsigned short op1offsets[18] = {0x00,0x01,0x02,0x08,0x09,0x0a,0x10,0x11,0x12,0x100,0x101,0x102,0x108,0x109,0x10a,0x110,0x111,0x112};
const unsigned short op2offsets[18] = {0x03,0x04,0x05,0x0b,0x0c,0x0d,0x13,0x14,0x15,0x103,0x104,0x105,0x10b,0x10c,0x10d,0x113,0x114,0x115};

/* number of melodic voices: 9 by default (OPL2), can go up to 18 (OPL3) */
static unsigned short voicescount = 9;


static void OPL2writeReg(unsigned short port, unsigned short reg, unsigned char data);
#pragma aux OPL2writeReg = \
        "out    dx,al"      \
        "mov    cx,6"       \
"loop1:  in     al,dx"      \
        "loop   loop1"      \
        "inc    dx"         \
        "mov    al,bl"      \
        "out    dx,al"      \
        "dec    dx"         \
        "mov    cx,36"      \
"loop2:	 in     al,dx"      \
        "loop   loop2"      \
        parm [DX][AX][BL]   \
        modify exact [AL CX DX] nomemory;

static void OPL3writeReg(unsigned short port, unsigned short reg, unsigned char data);
#pragma aux OPL3writeReg = \
        "test   ah,ah"      \
        "jz     bank0"      \
        "inc    dx"         \
        "inc    dx"         \
"bank0:  out    dx,al"      \
        "in     al,dx"      \
        "mov    ah,al"      \
        "inc    dx"         \
        "mov    al,bl"      \
        "out    dx,al"      \
        parm [DX][AX][BL]   \
        modify exact [AX DX] nomemory;


/* function used to write into a register 'reg' of the OPL chip located at
 * port 'port', writing byte 'data' into it. this function supports also OPL3.
 * to write into the secondary address of an OPL3, just OR your register with
 * the 0x100 value (the 0x100 flag will be ignored then and data will be
 * written into port+3). */
static void oplregwr(unsigned short port, unsigned short reg, unsigned char data) {
  if (oplmem->opl3) {
    OPL3writeReg(port, reg, data);
  } else {
    /* slower because OPL2 requires ugly delays between register writes, which
     * may sometimes lead to hearable cracks and glitches */
    OPL2writeReg(port, reg, data);
  }
}


/* 'volume' is in range 0..127 - take care to change only the 'attenuation'
 * part of the register, and never touch the KSL bits */
static void calc_vol(unsigned char *regbyte, int volume) {
  int level;
  /* invert bits and strip out the KSL header */
  level = (~(*regbyte)) & 0x3f;

  /* adjust volume */
  level = (level * volume) / 127;

  /* boundaries check */
  if (level > 0x3f) {
    level = 0x3f;
  } else if (level < 0) {
    level = 0;
  }

  /* invert the value (but keep it 6-bit), as expected by the OPL registers */
  level ^= 0x3F;

  /* final result computation */
  *regbyte &= 0xC0;  /* zero out all attenuation bits */
  *regbyte |= level; /* fill in the new attentuation value */
}


/* Initialize hardware upon startup - positive on success, negative otherwise
 * Returns 0 for OPL2 initialization, or 1 if OPL3 has been detected */
int opl_init(unsigned short port) {
  unsigned short x, y;

  /* make sure we're not inited yet */
  if (oplmem != NULL) return(-1);

  /* detect the hardware and return error if not found */
  oplregwr(port, 0x04, 0x60); /* reset both timers by writing 60h to register 4 */
  oplregwr(port, 0x04, 0x80); /* enable interrupts by writing 80h to register 4 (must be a separate write from the 1st one) */
  x = inp(port) & 0xE0; /* read the status register (port 388h) and store the result */
  oplregwr(port, 0x02, 0xff); /* write FFh to register 2 (Timer 1) */
  oplregwr(port, 0x04, 0x21); /* start timer 1 by writing 21h to register 4 */
  udelay(500); /* Creative Labs recommends a delay of at least 80 microseconds
                  I delay for 500us just to be sure. DO NOT perform inp()
                  calls for delay here, some cards do not initialize well then
                  (reported for CT2760) */
  y = inp(port) & 0xE0;  /* read the upper bits of the status register */
  /* reset both timers and interrupts (see steps 1 and 2) */
  oplregwr(port, 0x04, 0x60);
  oplregwr(port, 0x04, 0x80);

  /* test the results of steps 3 and 7 by ANDing them with E0h. The reesult of
   * step 3 should be 00h, and the result of step 7 should be C0h. If both are
   * good then we know that this PC has an AdLib-compatible board installed. */
  if (x != 0) return(-1);
  if (y != 0xC0) return(-2);

  /* init memory */
  oplmem = calloc(1, sizeof(struct oplstate_t));
  if (oplmem == NULL) return(-3);

  voicescount = 9; /* OPL2 provides 9 melodic voices */

  /* init the hardware */

  /* is it an OPL3? */
  if ((inp(port) & 0x06) == 0) {
    oplmem->opl3 = 1;
    oplregwr(port, 0x105, 1);  /* enable OPL3 mode (36 operators) */
    oplregwr(port, 0x104, 0);  /* disable four-operator voices */
    voicescount = 18;          /* OPL3 provides 18 melodic channels */

    /* Init the secondary OPL chip
     * NOTE: this I don't do anymore, it turns my Aztech Waverider mute! */
    /* oplregwr(port, 0x101, 0x20); */ /* enable Waveform Select */
    /* oplregwr(port, 0x108, 0x40); */ /* turn off CSW mode and activate FM synth mode */
    /* oplregwr(port, 0x1BD, 0x00); */ /* set vibrato/tremolo depth to low, set melodic mode */
  }

  oplregwr(port, 0x01, 0x20);  /* enable Waveform Select */
  oplregwr(port, 0x04, 0x00);  /* turn off timers IRQs */
  oplregwr(port, 0x08, 0x40);  /* turn off CSW mode and activate FM synth mode */
  oplregwr(port, 0xBD, 0x00);  /* set vibrato/tremolo depth to low, set melodic mode */

  for (x = 0; x < voicescount; x++) {
    /* set the modulator's multiple to 1 */
    oplregwr(port, 0x20 + op1offsets[x], 0x1);
    oplregwr(port, 0x20 + op2offsets[x], 0x1);

    /* set volume of all channels to about 40 dB */
    oplregwr(port, 0x40 + op1offsets[x], 0x10);
    oplregwr(port, 0x40 + op2offsets[x], 0x10);
  }

  opl_clear(port);

  /* all done */
  return(oplmem->opl3);
}


/* close OPL device */
void opl_close(unsigned short port) {
  int x;

  /* turns all notes 'off' */
  opl_clear(port);

  /* set volume to lowest level on all voices */
  for (x = 0; x < voicescount; x++) {
    oplregwr(port, 0x40 + op1offsets[x], 0x1f);
    oplregwr(port, 0x40 + op2offsets[x], 0x1f);
  }

  /* if OPL3, switch the chip back into its default OPL2 mode */
  if (oplmem->opl3 != 0) oplregwr(port, 0x105, 0);

  /* free state memory */
  free(oplmem);
  oplmem = NULL;
}


static void opl_noteoff(unsigned short port, unsigned short voice) {
  /* if voice is one of the OPL3 set, adjust it and route over secondary OPL port */
  if (voice >= 9) {
    oplregwr(port, 0x1B0 + voice - 9, 0);
  } else {
    oplregwr(port, 0xB0 + voice, 0);
  }
}


static void opl_noteon(unsigned short port, unsigned short voice, unsigned short note, int pitch) {
  long freq = freqtable[note];
  unsigned int octave = octavetable[note];

  /* The pitch wheel is meant to adjust the frequency of notes occurring on a
   * channel by a number of semitones. A semitone is a ratio of the 12th root
   * of 2 to 1, ie. roughly 1.05946:1, so 1000 Hz and 1059.46 Hz are a semitone
   * apart. In other words, to increase a frequency by one semitone, multiply
   * it by 1.05946. To decrease it by one semitone, divide it by 1.05946.
   * Note: A nice rational approximation of 2^(1/12) is 1009/1069. */
  if (pitch < 0) {
    /* pitch is the amount of 1/128th of a semi tone to add or subtract from
     * the base freq */
    freq += freq * pitch * 11 / 196 / 128; /* 11/196 is the difference of subtracting one half step */
  } else if (pitch > 0) {
    freq += freq * pitch * 49 / 824 / 128; /* 49/824 is the difference of adding one half step */

    /* Correct freq and octave if freq no longer fits in 10 bits. How? Musical
     * intervals are defined by a ratio of frequencies. An octave is a ratio of
     * 2:1, so from 100 Hz to 200 Hz, and from 200 Hz to 400 Hz, are both
     * octaves. */
    if (freq >= 1024) {
      freq >>= 1;
      if (octave < 7) octave++;
    }
  }

  /* if voice is one of the OPL3 set, adjust it and route over secondary OPL port */
  if (voice >= 9) {
    voice -= 9;
    voice |= 0x100;
  }

  /* A0-A8: Frequency Number register: determines the pitch of the note.
   *        (lowest 8 bits). Highest 2 bits are stored in the register B0-B8,
   *        hence the F-Number value must be in range 0..1023.
   *
   * B0-B8: Key On / Block Number / F-Number(hi bits):
   *
   *    +-7-+-6-+-5-+-4-+-3-+-2-+-1-+-0-+
   *    |       |KEY| Block Num.| Freq  |
   *    +---+---+---+---+---+---+---+---+
   *
   * bit 5: KEY-ON. When 1, channel output is enabled.
   * bits 2-4: Block Number. Roughly determines the octave (0..7).
   * bits 0-1: Frequency Number. 2 highest bits of the A0-A8 register.
   *
   * The following formula is used to determine F-Number and Block:
   *
   * F-Number = Music Frequency * 2^(20 - BlockNum) / 49716 Hz */

  oplregwr(port, 0xA0 + voice, freq & 0xff); /* set lowfreq */
  oplregwr(port, 0xB0 + voice, (freq >> 8) | (octave << 2) | 32); /* KEY ON + hifreq + octave */
}


/* turns off all notes */
void opl_clear(unsigned short port) {
  int x, y;
  for (x = 0; x < voicescount; x++) opl_noteoff(port, x);

  /* reset the percussion bits at the 0xBD register */
  oplregwr(port, 0xBD, 0);

  /* mark all voices as unused */
  for (x = 0; x < voicescount; x++) {
    oplmem->voices2notes[x].channel = -1;
    oplmem->voices2notes[x].note = -1;
    oplmem->voices2notes[x].timbreid = -1;
  }

  /* mark all notes as unallocated */
  for (x = 0; x < 16; x++) {
    for (y = 0; y < 128; y++) oplmem->notes2voices[x][y] = -1;
  }

  /* pre-set emulated channel patches to default GM ids and reset all
   * per-channel volumes and pitch bends */
  for (x = 0; x < 16; x++) {
    opl_midi_changeprog(x, x);
    oplmem->channelvol[x] = 127;
    oplmem->channelbendrange[x] = 2; /* GM recommends a default pitch wheel range of +/- 2 semitones */
  }
}


/* The pitch wheel value range is in 0..0x3FFF. A value of 0x2000 is meant to
 * indicate that the Pitch Wheel is centered (ie, the sounding notes aren't
 * being transposed up or down). Higher values transpose pitch up, and lower
 * values transpose pitch down.
 *
 * The Pitch Wheel range is usually adjustable by the musician on each MIDI
 * device. For example, although 0x2000 is always center position, on one MIDI
 * device, a 0x3000 could transpose the pitch up a whole step, whereas on
 * another device that may result in only a half step up. The GM spec
 * recommends that MIDI devices default to using the entire range of possible
 * Pitch Wheel message values (ie, 0x0000 to 0x3FFF) as +/- 2 half steps
 * transposition (ie, 4 half-steps total range). The Pitch Wheel Range (or
 * Sensitivity) is adjusted via an RPN controller message. */
void opl_midi_pitchwheel(unsigned short oplport, int channel, int pitchwheel) {
  int voice;

  /* Some words about the "Pitch Wheel Change" and the related Pitch Bend Range
   * used in MIDI files for note bending.
   *
   * Pitch Bend Range (aka Pitch Bend Sensitivity) is set through an RPN-0
   * message.
   * A Pitch Bend Range of 1 means that you can bend +/- 1 semitone.
   * A Pitch Bend Range of 12 means that you can bend +/- 12 semitones.
   *
   * The MIDI Specification supports pitch bend ranges up to 24 max, and the
   * current MPE proposal extends this to 96 (meaning +/- 96 semitones).
   *
   * The MPE proposal calls out 48 as the default Pitch Bend Range, while the
   * GM spec recommends that MIDI devices default to using the entire range of
   * possible Pitch Wheel message values (ie, 0x0000 to 0x3FFF) as +/- 2 half
   * steps transposition (ie, 4 half-steps total range).
   *
   * Pitch Wheel Change (aka pitch bend) is a 14 bit controller, with values
   * ranging from 0 to 16383, where a value of 8192 means "no pitch". So it is
   * essentially treated as a signed integer with a range of -8192..8191.
   */

  /* convert the pitchwheel value from its MIDI range of 0..3FFFh to a signed
   * value (-2000h..1999h) and translate it into a number of 1/128th of a
   * seminotes according to the channel bend range (ie. val 256 means "two
   * seminotes") */
  oplmem->channelpitch[channel] = (long)(oplmem->channelbendrange[channel]) * (pitchwheel - 0x2000) / 64;

  /* check all active voices to see who is playing on given channel now and
   * recompute playing notes for this channel with the updated pitch */
  for (voice = 0; voice < voicescount; voice++) {
    /* skip voices that are set to a different channel */
    if (oplmem->voices2notes[voice].channel != channel) continue;

    /* skip voices that do not play any note at the moment */
    if (oplmem->voices2notes[voice].note < 0) continue;

    //printf("PITCH WHEEL CHAN=%d VAL=%d (%d)\n", channel, pitchwheel, oplmem->channelpitch[channel]);

    /* reload the note to force the pitch change */
    opl_noteon(oplport, voice, oplmem->voices2notes[voice].note, oplmem->channelpitch[channel] + gmtimbres[oplmem->voices2notes[voice].timbreid].finetune);
  }
}


void opl_midi_controller(unsigned short oplport, int channel, int id, int value) {
  int x;
  static unsigned short selectedrpn[16];

  switch (id) {
    case 6:  /* Data Entry MSB (coarse) */
      /* this value relates to the RPN that has been selected on this channel */
      if (selectedrpn[channel] == 0) { /* Pitch Bend Sensitivity (Bend Range) */
      /* The high order adjustment uses Controller 6 (Data Entry MSB) to set
       * the range in semitones. The low order adjustment uses Controller 38
       * (Data Entry LSB) to set the range in cents (hundredths of a semitone). */
        oplmem->channelbendrange[channel] = value;
      }
      selectedrpn[channel] = 0xffff;
      break;
    case 11: /* "Expression" (meaning "channel volume") */
      oplmem->channelvol[channel] = value;
      break;
    case 38: /* Data Entry LSB (fine tuning) */
      /* ignored, my OPL driver is too primitive for fine tuning to be meaningful */
      break;
    case 100: /* RPNs (Registered Parameter Numbers) fine tuning */
      /* ignored, my OPL driver is too primitive for fine tuning to be meaningful */
      break;
    case 101: /* RPNs (Registered Parameter Numbers) coarse tuning */
      /* Remember what RPN is being selected. In a moment I should receive a
       * value for this RPN through controller 6 (Data Entry MSB) */
      selectedrpn[channel] = value;
      break;
    case 123: /* 'all notes off' */
    case 120: /* 'all sound off' - I map it to 'all notes off' for now, not perfect but better than not handling it at all */
      for (x = 0; x < voicescount; x++) {
        if (oplmem->voices2notes[x].channel != channel) continue;
        opl_midi_noteoff(oplport, channel, oplmem->voices2notes[x].note);
      }
      break;
  }
}


/* assign a new instrument to emulated MIDI channel */
void opl_midi_changeprog(int channel, int program) {
  if (channel == 9) return; /* do not allow to change channel 9, it is for percussions only */
  oplmem->channelprog[channel] = program;
}


void opl_loadinstrument(unsigned short oplport, unsigned short voice, struct timbre_t *timbre) {
  /* KSL (key level scaling) / attenuation */
  oplregwr(oplport, 0x40 + op2offsets[voice], timbre->carrier_40 | 0x3f); /* immediately force volume to 0 to avoid possible glitches. it will be set to
  a proper value later during 'note on' */
  oplregwr(oplport, 0x40 + op1offsets[voice], timbre->modulator_40);

  /* select waveform on both operators */
  oplregwr(oplport, 0xE0 + op1offsets[voice], timbre->modulator_E862 >> 24);
  oplregwr(oplport, 0xE0 + op2offsets[voice], timbre->carrier_E862 >> 24);

  /* sustain / release */
  oplregwr(oplport, 0x80 + op1offsets[voice], (timbre->modulator_E862 >> 16) & 0xff);
  oplregwr(oplport, 0x80 + op2offsets[voice], (timbre->carrier_E862 >> 16) & 0xff);

  /* attack rate / decay */
  oplregwr(oplport, 0x60 + op1offsets[voice], (timbre->modulator_E862 >> 8) & 0xff);
  oplregwr(oplport, 0x60 + op2offsets[voice], (timbre->carrier_E862 >> 8) & 0xff);

  /* AM / vibrato / envelope */
  oplregwr(oplport, 0x20 + op1offsets[voice], timbre->modulator_E862 & 0xff);
  oplregwr(oplport, 0x20 + op2offsets[voice], timbre->carrier_E862 & 0xff);

  /* feedback / connection */
  if (voice >= 9) {
    voice -= 9;
    voice |= 0x100;
  }
  if (oplmem->opl3 != 0) { /* on OPL3 make sure to enable LEFT/RIGHT unmute bits */
    oplregwr(oplport, 0xC0 + voice, timbre->feedconn | 0x30);
  } else {
    oplregwr(oplport, 0xC0 + voice, timbre->feedconn);
  }

}


/* adjust the volume of the voice (in the usual MIDI range of 0..127) */
static void voicevolume(unsigned short port, unsigned short voice, int program, int volume) {
  unsigned char carrierval = gmtimbres[program].carrier_40;
  if (volume == 0) {
    carrierval |= 0x3f;
  } else {
    calc_vol(&carrierval, volume);
  }
  oplregwr(port, 0x40 + op2offsets[voice], carrierval);
}


/* get the id of the instrument that relates to channel/note pair */
static int getinstrument(int channel, int note) {
  if ((note < 0) || (note > 127) || (channel > 15)) return(-1);
  if (channel == 9) { /* the percussion channel requires special handling */
    return(128 | note);
  }
  return(oplmem->channelprog[channel]);
}


void opl_midi_noteon(unsigned short port, int channel, int note, int velocity) {
  int x, voice = -1;
  int lowestpriorityvoice = 0;
  int instrument;

  /* get the instrument to play */
  instrument = getinstrument(channel, note);
  if (instrument < 0) return;

  /* if note already playing, then reuse its voice to avoid leaving a stuck voice */
  if (oplmem->notes2voices[channel][note] >= 0) {
    voice = oplmem->notes2voices[channel][note];
  } else {
    /* else find a free voice, possibly with the right timbre, or at least locate the oldest note */
    for (x = 0; x < voicescount; x++) {
      if (oplmem->voices2notes[x].channel < 0) {
        voice = x; /* preselect this voice, but continue looking */
        /* if the instrument is right, do not look further */
        if (oplmem->voices2notes[x].timbreid == instrument) {
          break;
        }
      }
      if (oplmem->voices2notes[x].priority < oplmem->voices2notes[lowestpriorityvoice].priority) lowestpriorityvoice = x;
    }
    /* if no free voice available, then abort the oldest one */
    if (voice < 0) {
      voice = lowestpriorityvoice;
      opl_midi_noteoff(port, oplmem->voices2notes[voice].channel, oplmem->voices2notes[voice].note);
    }
  }

  /* load the proper instrument, if not already good */
  if (oplmem->voices2notes[voice].timbreid != instrument) {
    oplmem->voices2notes[voice].timbreid = instrument;
    opl_loadinstrument(port, voice, &(gmtimbres[instrument]));
  }

  /* update states */
  oplmem->voices2notes[voice].channel = channel;
  oplmem->voices2notes[voice].note = note;
  oplmem->voices2notes[voice].priority = ((16 - channel) << 8) | 0xff; /* lower channels must have priority */
  oplmem->notes2voices[channel][note] = voice;

  /* set the requested velocity on the voice */
  voicevolume(port, voice, oplmem->voices2notes[voice].timbreid, velocity * oplmem->channelvol[channel] / 127);

  /* percussion channel doesn't provide a real note, so I use a static one
   * MUSPLAYER uses C-5 (60), why not using this as a default */
  if (channel == 9) {
    /* the IBK timbre might have a suggestion */
    note = gmtimbres[instrument].percnote;
    if (note == 0) note = 60; /* if not then fall back to 60 */
  }

  /* trigger NOTE_ON on the OPL, take care to apply the 'finetune' pitch
   * correction, too */
  opl_noteon(port, voice, note, oplmem->channelpitch[channel] + gmtimbres[instrument].finetune);

  /* readjust all priorities */
  for (x = 0; x < voicescount; x++) {
    if (oplmem->voices2notes[x].priority > 0) oplmem->voices2notes[x].priority -= 1;
  }
}


void opl_midi_noteoff(unsigned short port, int channel, int note) {
  int voice = oplmem->notes2voices[channel][note];

  if (voice >= 0) {
    opl_noteoff(port, voice);
    oplmem->voices2notes[voice].channel = -1;
    oplmem->voices2notes[voice].note = -1;
    oplmem->voices2notes[voice].priority = -1;
    oplmem->notes2voices[channel][note] = -1;
  }
}


static int opl_loadbank_internal(char *file, int offset) {
  unsigned char buff[16];
  int i;
  struct fiofile_t f;
  /* open the IBK file */
  if (fio_open(file, FIO_OPEN_RD, &f) != 0) return(-1);
  /* file must be exactly 3204 bytes long */
  if (fio_seek(&f, FIO_SEEK_END, 0) != 3204) {
    fio_close(&f);
    return(-2);
  }
  fio_seek(&f, FIO_SEEK_START, 0);
  /* file must start with an IBK header */
  if ((fio_read(&f, buff, 4) != 4) || (buff[0] != 'I') || (buff[1] != 'B') || (buff[2] != 'K') || (buff[3] != 0x1A)) {
    fio_close(&f);
    return(-3);
  }
  /* load 128 instruments from the IBK file */
  for (i = offset; i < 128 + offset; i++) {
    /* read the 16-bytes timbre definition */
    if (fio_read(&f, buff, 16) != 16) {
      fio_close(&f);
      return(-4);
    }

    /* first of all check the 12th byte: it indicates percussion voice mode
     * (0=melodic 6=bassdrum 7=snare 8=tomtom 9=cymbal 10=hihat), DOSMid drives
     * OPL in melodic mode (no rhytm voices) so I skip non-melodic timbres. */
    if (buff[11] != 0) continue;

    /* load modulator */
    gmtimbres[i].modulator_E862 = buff[8]; /* wave select */
    gmtimbres[i].modulator_E862 <<= 8;
    gmtimbres[i].modulator_E862 |= buff[6]; /* sust/release */
    gmtimbres[i].modulator_E862 <<= 8;
    gmtimbres[i].modulator_E862 |= buff[4]; /* attack/decay */
    gmtimbres[i].modulator_E862 <<= 8;
    gmtimbres[i].modulator_E862 |= buff[0]; /* AM/VIB... flags */
    /* load carrier */
    gmtimbres[i].carrier_E862 = buff[9]; /* wave select */
    gmtimbres[i].carrier_E862 <<= 8;
    gmtimbres[i].carrier_E862 |= buff[7]; /* sust/release */
    gmtimbres[i].carrier_E862 <<= 8;
    gmtimbres[i].carrier_E862 |= buff[5]; /* attack/decay */
    gmtimbres[i].carrier_E862 <<= 8;
    gmtimbres[i].carrier_E862 |= buff[1]; /* AM/VIB... flags */
    /* load KSL */
    gmtimbres[i].modulator_40 = buff[2];
    gmtimbres[i].carrier_40 = buff[3];
    /* feedconn & finetune */
    gmtimbres[i].feedconn = buff[10];
    gmtimbres[i].finetune = buff[12] * 128; /* used only in some IBK files */
    /* percussion pitch (note) */
    gmtimbres[i].percnote = buff[13];
  }
  /* close file and return success */
  fio_close(&f);
  return(0);
}

/*
static void dump2file(void) {
  FILE *fd;
  int i;
  fd = fopen("dump.txt", "wb");
  if (fd == NULL) return;
  for (i = 0; i < 256; i++) {
    char *comma = "";
    if (i < 255) comma = ",";
    fprintf(fd, "{0x%07lX,0x%07lX,0x%02X,0x%02X,0x%02X,%d}%s\r\n", gmtimbres[i].modulator_E862, gmtimbres[i].carrier_E862, gmtimbres[i].modulator_40, gmtimbres[i].carrier_40, gmtimbres[i].feedconn, gmtimbres[i].finetune, comma);
  }
  fclose(fd);
}*/

int opl_loadbank(char *file) {
  char *instruments = NULL, *percussion = NULL;
  int i, res;
  instruments = strdup(file); /* duplicate the string so we can modify it */
  if (instruments == NULL) return(-64); /* out of mem */
  /* if a second file is provided, it's for percussion */
  for (i = 0; instruments[i] != 0; i++) {
    if (instruments[i] == ',') {
      instruments[i] = 0;
      percussion = instruments + i + 1;
      break;
    }
  }
  /* load the file(s) */
  res = opl_loadbank_internal(instruments, 0);
  if ((res == 0) && (percussion != NULL)) {
    res = opl_loadbank_internal(percussion, 128);
  }
  free(instruments);
  /*dump2file();*/ /* dump instruments to a 'dump.txt' file */
  return(res);
}

#endif /* #ifdef OPL */
