-- gif.e - GIF files reading/writing - HIP 2.1
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file has routines for reading the headers of GIF files,
-- loading them into the memory, and saving GIF files.

constant include_name = "gif.e"

include machine.e
include fileutil.e
include constant.e


constant FLAG_GLOBALCOLORTABLE = #80,
	 FLAG_GLOBALCOLORTABLESIZE = #7,
--         FLAG_COLORRESOLUTION = #70,
	 resolution_divide = #10,
--         FLAG_SORTED = #8,

	 FLAG_TRANSPARENTCOLOR = #1,

	 FLAG_LOCALCOLORTABLE = #80,
	 FLAG_INTERLACED = #40,
	 FLAG_LOCALCOLORTABLESIZE = #7

function read_color_table (atom fn, atom colors)
    -- Reads the palette from a GIF file.

    sequence pal, pal_entry

    pal = repeat (-1, colors)
    pal_entry = {0,0,0}

    for n = 1 to colors do
	pal_entry [PAL_RED] = getc (fn)
	pal_entry [PAL_GREEN] = getc (fn)
	pal_entry [PAL_BLUE] = getc (fn)

	if pal_entry [PAL_BLUE] = -1 then   -- unexpected eof
	    return ERROR_CORRUPTED
	end if

	pal [n] = pal_entry
    end for

    return pal
end function

constant interlace_step = {8, 8, 4, 2}, interlace_start = {0, 4, 2, 1}

procedure un_interlace (atom address, atom width, atom height)
    -- Orders the lines of an interlaced picture.

    atom new_address, line_address

    -- allocate temporary space
    new_address = allocate (width * height)
    line_address = address

    -- move the lines from the original data to the temporary space,
    -- in the correct order
    for step = 1 to 4 do

	for line = interlace_start [step] to height - 1 by interlace_step [step] do
	    mem_copy ( new_address + line * width, line_address, width )
	    line_address += width
	end for
    end for

    -- move the data in the temporary space back into the original space
    mem_copy (address, new_address, width * height)

    free (new_address)  -- free temporary space
end procedure


function uncompress_gif(atom fn, atom width, atom height, atom interlaced)
    -- Uncompresses a LZW-compressed GIF file.

    atom address, cur_address, max_address
    integer code_size, bits_per_code, clear_code, end_code,
	 bytes_left_in_block, buffer, bits_in_buffer, code, mask,
	 divisor

    sequence codes, code_string, old


    -- allocate space for storing the image
    address = allocate (width * height)

    if address = 0 then
	return ERROR_OUTOFMEMORY
    end if



    code_size = getc (fn)   -- get code size
    if code_size < 2 or code_size > 8 then
	return ERROR_CORRUPTED    -- invalid value
    end if

    bits_per_code = code_size + 1
    divisor = power (2, bits_per_code)
    mask = divisor - 1

    -- Initialize prefix table
    codes = repeat (0, power (2, code_size) + 1)
    for n = 1 to length (codes) do  -- init code
	codes [n] = {n - 1}
    end for

    -- Note: initially there are (2^code_size) + 2 codes, so you can
    -- see that the code table is one code shorter than it should be.
    -- This is because after the first code is processed, the decompressor
    -- should not add it to the table (as it is always already there).
    -- In order to save code size and improve speed, this decompressor
    -- does not treat the first code separately, but rather adds it to
    -- the table, making it the correct size.

    clear_code = power (2, code_size)
    end_code = power (2, code_size) + 1

    -- Initialize reading buffer
    buffer = 0
    bits_in_buffer = 0
    bytes_left_in_block = 0
    old = {}

    cur_address = address
    max_address = address + width*height

    while cur_address <= max_address do

	-- Are there enough bits in the buffer for another code?
	while bits_in_buffer < bits_per_code do
	    -- add another byte to the buffer

	    if bytes_left_in_block = 0 then     -- start of new block?
		bytes_left_in_block = getc (fn) -- get block length
		if bytes_left_in_block <= 0 then -- end of data or of file
		    return ERROR_CORRUPTED  -- decompression hasn't finished yet
		end if
	    end if

	    -- read another byte and add it to the buffer
	    buffer = buffer + getc (fn) * power (2, bits_in_buffer)
	    bits_in_buffer = bits_in_buffer + 8
	    bytes_left_in_block = bytes_left_in_block - 1

	end while

	-- read code
	code = and_bits (buffer, mask)

	-- remove used bits from buffer
	buffer = floor (buffer / divisor)
	bits_in_buffer = bits_in_buffer - bits_per_code

	-- handle code
	if cur_address >= max_address then      -- end of decompressed data
	    if code != end_code then
		return ERROR_CORRUPTED
	    else
		exit    -- success!
	    end if

	elsif code = clear_code then

	    -- clear code table

	    bits_per_code = code_size + 1
	    divisor = power (2, bits_per_code)
	    mask = divisor - 1
	    codes = repeat (0, power (2, code_size) + 1)
	    for n = 1 to length (codes) do  -- init code
		codes [n] = {n - 1}
	    end for

	elsif code = end_code then
	    -- error, this should not happen
	    return ERROR_CORRUPTED

	elsif code <= length (codes) - 1 then

	    code_string = codes [code+1]        -- get translation for code

	    if cur_address + length (code_string) > max_address then
		return ERROR_CORRUPTED  -- data goes over maximum size
	    end if

	    poke (cur_address, code_string)     -- write to memory

	    cur_address += length (code_string)
	    codes = append (codes, old & code_string [1])-- add code to table
	    old = code_string

	    if length (codes) = divisor then
		bits_per_code = bits_per_code + 1
		if bits_per_code > 12 then bits_per_code = 12 end if
		divisor = power (2, bits_per_code)
		mask = divisor - 1
	    end if


	elsif code = length (codes) then

	    code_string = old & old [1]

	    if cur_address + length (code_string) > max_address then
		return ERROR_CORRUPTED  -- data goes over maximum size
	    end if

	    poke (cur_address, code_string)

	    cur_address += length (code_string)

	    codes = append (codes, code_string)-- add code to table
	    old = code_string

	    if length (codes) = divisor then
		bits_per_code = bits_per_code + 1
		if bits_per_code > 12 then bits_per_code = 12 end if
		divisor = power (2, bits_per_code)
		mask = divisor - 1
	    end if


	else
	    return address --ERROR_CORRUPTED
	end if

    end while

    if interlaced then
	un_interlace (address, width, height)
    end if

    return address
end function




global function gif_read (atom fn)
    -- Reads the gif file header.

    atom width, height, colors, -- version,
	 flags, logical_screen_height,
	 logical_screen_width, pixel_aspect_ratio, background_color,
	 field_type, extension_type, block_size,
	 interlaced, address, transparent

    sequence pal, img_info
    object temp

    transparent = -1
    colors = 0
    pal = {}


    -- Move to the beginning of the file
    if seek(fn, 0) then
	return ERROR_FILEERROR
    end if

    -- Bitmap ID - 2 bytes

    if compare (read_bytes(fn, 3), "GIF") then
	return ERROR_INVALIDBMP
    end if

    temp = read_bytes (fn, 3)

--    if compare (temp, "87a") = 0 then
--        version = GIF_87a
--    elsif compare (temp, "89a") = 0 then
--        version = GIF_89a
--    else
    if compare (temp, "87a") and compare (temp, "89a") then
	-- Not a known bitmap file header
	return ERROR_UNSUPPORTED
    end if


    logical_screen_width = read_word (fn, LITTLE_ENDIAN)    -- logical screen width
    logical_screen_height = read_word (fn, LITTLE_ENDIAN)    -- logical screen height

    flags = getc (fn)

    background_color = getc (fn)
    pixel_aspect_ratio = getc (fn)

    if and_bits (flags, FLAG_GLOBALCOLORTABLE) then
	colors = power (2, and_bits (flags, FLAG_GLOBALCOLORTABLESIZE)+1)

	temp = read_color_table (fn, colors)
	if atom (temp) then
	    return temp     -- error
	end if

	pal = temp
    end if

    while 1 do      -- read each block until an image is found


	field_type = getc (fn)      -- get field type

	if field_type = #21 then    -- extension block

	    extension_type = getc (fn)  -- extension code

	    if extension_type = #F9 then   -- Graphic Control Extension, process

		block_size = getc (fn)  -- block size
		if block_size != 4 then -- error
		    return ERROR_CORRUPTED
		end if

		flags = getc (fn)   -- flags byte

		temp = read_word (fn, LITTLE_ENDIAN)    -- delay time
		temp = getc (fn)    -- transparent color index

		if and_bits (flags, FLAG_TRANSPARENTCOLOR) then
		    -- transparent color present
		    transparent = temp
		else
		    -- no transparent color
		    transparent = -1
		end if

		block_size = getc (fn)  -- block size
		if block_size != 0 then -- error
		    return ERROR_CORRUPTED
		end if



	    else        -- skip unknown extension block

		while 1 do
		    block_size = getc (fn)
		    if block_size = 0 then exit end if

		    for n = 1 to block_size do
			temp = getc (fn)
		    end for
		end while

	    end if


	elsif field_type = #2C then
	    -- image
	    temp = read_word (fn, LITTLE_ENDIAN)    -- image left
	    temp = read_word (fn, LITTLE_ENDIAN)    -- image top
	    width = read_word (fn, LITTLE_ENDIAN)    -- image width
	    height = read_word (fn, LITTLE_ENDIAN)    -- image height

	    flags = getc (fn)       -- flags byte

	    if and_bits (flags, FLAG_LOCALCOLORTABLE) then
		colors = power (2, and_bits (flags, FLAG_LOCALCOLORTABLESIZE)+1)

		temp = read_color_table (fn, colors)
		if atom (temp) then
		    return temp     -- error
		end if

		pal = temp
	    end if

	    -- 1 if image is interlaced, 0 otherwise
	    interlaced = ( and_bits (flags, FLAG_INTERLACED) != 0)

	    address = uncompress_gif (fn, width, height, interlaced)

	    if address < 0 then
		return address      -- error
	    end if

	    exit

	elsif field_type = #3B then -- end of file
	    return ERROR_CORRUPTED      -- no image in the file!

	else
	    return ERROR_CORRUPTED      -- unknown field type code

	end if
    end while

    -- Write information in img_info structure
    img_info = repeat(-1, IMG_INFO_SIZE)

    img_info [IMG_WIDTH] = width
    img_info [IMG_HEIGHT] = height
    img_info [IMG_BACKGROUNDCOLOR] = background_color
    img_info [IMG_BPP] = 8      -- always in GIFs
    img_info [IMG_HRES] = 0 --hres
    img_info [IMG_VRES] = 0 --vres
    img_info [IMG_PAL] = pal
    img_info [IMG_COMPRESSION] = 0 --compression
    img_info [IMG_INTERLACED] = interlaced
--    img_info [IMG_TYPE] = version
    img_info [IMG_ADDRESS] = address
--    img_info [IMG_FORMAT] = FORMAT_GIF

    img_info [IMG_TRANSPARENT] = transparent

    return img_info

end function

function interlace (atom address, atom width, atom height)
    -- Interlaces a picture before compression

    atom new_address, line_address

    new_address = allocate (width * height)
    line_address = new_address

    for step = 1 to 4 do

	for line = interlace_start [step] to height - 1 by interlace_step [step] do
	    mem_copy (line_address, address + line * width, width )
	    line_address += width
	end for
    end for

    return new_address

end function




procedure compress_gif (atom fn, atom address, atom width, atom height, atom interlaced, atom code_size)
    -- Compresses a picture using the LZW algorithm used in the GIF format,
    -- and writes the output to fn.

    -- When the table is full, this routine will use at most about
    -- 4096 * 256 * 4 bytes = 4 MB of memory.
    atom bits_per_code, prefix, char, new_prefix,
	 chars, max_code, clear_code, end_code,
	 buffer, bits_in_buffer
    sequence compressed, table, empty

    -- Initialize buffer
    buffer = 0
    bits_in_buffer=0
    compressed={}

    -- minimum code size is 3 bits
    if code_size < 2 then code_size = 2 end if
    code_size = 8

    puts (fn, code_size)        -- write code size

    if interlaced then          -- create interlaced image
	address = interlace (address, width, height)
    end if

    chars = power (2, code_size)    -- how many chars to add to code table?
    clear_code = chars              -- code number for clear_code
    end_code = clear_code + 1       -- code number for end_code

    -- The data structure used is something like this:
    -- for each possible code, there is a sequence of <chars> elements,
    -- and each element points to the code corresponding to the current
    -- prefix plus the corresponding char, or contains -1 if the code is
    -- not present.

    empty = repeat (-1, chars)  -- the initial sequence for any code in the table

    bits_per_code = code_size + 1
    max_code = power (2, bits_per_code)
    table = repeat (empty, chars + 2)   -- the initial code table

    -- Add clear code to start of data stream
    buffer = buffer + clear_code * power(2,bits_in_buffer)
    bits_in_buffer = bits_in_buffer + bits_per_code

    prefix = peek (address) -- prefix starts with the first char of the data

    for pos = address + 1 to address + width*height - 1 do
	-- read each pixel

	char = peek (pos)

	new_prefix = table [prefix+1][char+1]   -- get code corresponding to prefix+char

	if new_prefix = -1 then    -- this prefix is not on the table
	    -- output prefix

	    buffer = buffer + prefix * power(2,bits_in_buffer)
	    bits_in_buffer = bits_in_buffer + bits_per_code


	    -- append code to table
	    table = append (table, empty)
	    table [prefix+1][char+1] = length (table)-1

	    prefix = char       -- set new prefix

	    if length (table) >= max_code then
		-- time to increase code size

		-- already maximum?
		if bits_per_code = 12 then

		    -- output clear code
		    buffer = buffer + clear_code * power(2,bits_in_buffer)
		    bits_in_buffer = bits_in_buffer + bits_per_code

		    -- re-initialize table
		    bits_per_code = code_size + 1
		    max_code = power (2, bits_per_code)
		    table = repeat (empty, chars + 2)

		elsif length(table) > max_code then

		    -- increase code length
		    bits_per_code = bits_per_code + 1
		    max_code = power (2, bits_per_code)

		end if
	    end if

	    -- flush buffer if necessary
	    while bits_in_buffer >= 8 do
		compressed = compressed & and_bits (buffer, #FF)
		buffer = floor (buffer / #100)
		bits_in_buffer = bits_in_buffer - 8
	    end while

	    if length(compressed) >= 255 then
		puts (fn, 255) -- sub-block length
		puts (fn, compressed [1..255])  -- sub-block
		compressed = compressed [256..length(compressed)]
	    end if


	else    -- code is already in table

	    prefix = new_prefix

	end if

    end for

    -- end of data, output current prefix
    buffer = buffer + prefix * power(2,bits_in_buffer)
    bits_in_buffer = bits_in_buffer + bits_per_code

    -- output end_code
    buffer = buffer + end_code * power(2,bits_in_buffer)
    bits_in_buffer = bits_in_buffer + bits_per_code

    -- flush buffer
    while bits_in_buffer >= 8 do
	compressed = compressed & and_bits (buffer, #FF)
	buffer = floor (buffer / #100)
	bits_in_buffer = bits_in_buffer - 8
    end while

    if bits_in_buffer then  -- remaining bits in buffer?
	compressed = compressed & buffer
    end if


    -- output code
    if length(compressed) >= 255 then
	puts (fn, 255) -- sub-block length
	puts (fn, compressed [1..255])  -- sub-block
	compressed = compressed [256..length(compressed)]
    end if


    if length(compressed) then  -- not a full block
	puts (fn, length (compressed)) -- sub-block length
	puts (fn, compressed)  -- sub-block
    end if

    puts (fn, 0)  --  end of data


    if interlaced then
	free (address)  -- free interlaced picture
    end if
end procedure


global function gif_save (atom fn, sequence img_info)
    atom flags, bits_in_color_table
    sequence pal

--    puts(1,"\nsave_gif:\n")
    if img_info [IMG_BPP] != 8 then
	return ERROR_UNSUPPORTEDBPP
    end if


    puts (fn, "GIF")

    if img_info [IMG_TRANSPARENT] != -1 then    -- transparent color,
	puts(fn, "89a")                         -- requires 89a
    else
	puts(fn, "87a")
    end if

    write_word (fn, img_info [IMG_WIDTH], LITTLE_ENDIAN)   -- logical screen width
    write_word (fn, img_info [IMG_HEIGHT], LITTLE_ENDIAN)   -- logical screen height

    -- how many entries in color table are necessary?
    bits_in_color_table = - floor ( -log(length(img_info [IMG_PAL]))/log(2))
    if bits_in_color_table < 3 then
	bits_in_color_table = 3
    end if


--    ?bits_in_color_table

    flags = FLAG_GLOBALCOLORTABLE +     -- global color table present
	    7 * resolution_divide +     -- 8 bits color resolution
	    bits_in_color_table - 1     -- n bits color table

    puts (fn, flags)        -- write flags

    if img_info [IMG_BACKGROUNDCOLOR] >= 0 then     -- background color index
	puts (fn, img_info [IMG_BACKGROUNDCOLOR])
    else        -- no background color specified
	puts (fn, 0)
    end if

    puts (fn, 0)    -- pixel aspect ratio - 0: no information


    pal = img_info [IMG_PAL]
--    ?length(pal)
    -- Fill palette with black until desired length
    pal = pal & repeat ( {0,0,0}, power(2, bits_in_color_table)-length(pal))
--    ?length(pal)
--    ?power(2,bits_in_color_table)

    for n = 1 to power(2, bits_in_color_table) do -- write color table
	puts (fn, pal[n][PAL_RED])
	puts (fn, pal[n][PAL_GREEN])
	puts (fn, pal[n][PAL_BLUE])
    end for

    if img_info [IMG_TRANSPARENT] != -1 then    -- transparent color,
				   -- write Graphic Control Extension

	puts (fn, {#21, #F9})   -- block label
	puts (fn, 4)            -- block size

	flags = FLAG_TRANSPARENTCOLOR   -- transparent color present

	puts (fn, flags)        -- flags
	write_word (fn, 0, LITTLE_ENDIAN)   -- delay time
	puts (fn, img_info [IMG_TRANSPARENT]) -- transparent color index
	puts (fn, 0)        -- block terminator

    end if

    -- Image descriptor
    puts (fn, #2C)      -- block label
    write_word (fn, 0, LITTLE_ENDIAN)   -- image left pos
    write_word (fn, 0, LITTLE_ENDIAN)   -- image top pos
    write_word (fn, img_info [IMG_WIDTH], LITTLE_ENDIAN)   -- image width
    write_word (fn, img_info [IMG_HEIGHT], LITTLE_ENDIAN)   -- image height

    flags = 0
    if img_info [IMG_INTERLACED] = -1 then img_info [IMG_INTERLACED] = 0 end if

    if img_info [IMG_INTERLACED] then
	flags = flags + FLAG_INTERLACED
    end if

    puts (fn, flags)        -- flags

    -- write image
    compress_gif (fn, img_info [IMG_ADDRESS], img_info [IMG_WIDTH],
	img_info [IMG_HEIGHT], img_info [IMG_INTERLACED], bits_in_color_table)

    puts (fn, #3B)  -- end of file

    return 0    -- no errors

end function

constant FORMAT_GIF = register_img_format ("GIF","gif","gif",{"GIF"},{8})

