-- fileutil.e - File utilities - HIP 2.0
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file contains routines related to file handling.

include file.e

global function file_exists (sequence filename)
    -- Returns 1 if file exists, 0 otherwise

    atom fn
    fn = open (filename, "rb")
    if fn != -1 then
	close(fn)
	return 1    -- Yes, file exists

    else
	return 0
    end if

end function

global function file_size(atom file_number)
    -- Gets the file size

    atom lof,current_position

    -- Get old position
    current_position=where(file_number)

    -- Go to end of file
    if seek(file_number,-1) then
	return -1
    end if

    -- Get position
    lof=where(file_number)

    -- Go back to old position
    if seek(file_number,current_position) then
	return -1
    end if
    return lof

end function



constant read_block_size=256
global function read_bytes(atom file_number,atom bytes)
    -- Reads a certain number of bytes from the file.
    -- It is faster than get_bytes when the end of the file is reached.

    integer full_blocks
    sequence data

    full_blocks=bytes-remainder (bytes,read_block_size)
    data=repeat(-1,bytes)


    for temp=1 to full_blocks by read_block_size do

	for byte_number=temp to temp+read_block_size-1 do
	    -- Read from file
	    data[byte_number]=getc(file_number)
	end for

	if data[temp+read_block_size-1]=-1 then
	    -- Unexpected end of file

	    -- Find last valid character
	    full_blocks=temp+read_block_size-2  -- Using full_blocks simply to avoid declaring another variable
	    while full_blocks >= 1 and data[full_blocks] = -1 do
		full_blocks=full_blocks-1
	    end while

	    return data[1..full_blocks]
	end if
    end for

    -- Read last block
    for byte_number=full_blocks+1 to bytes do
	data[byte_number]=getc(file_number)
    end for

    if data[bytes]=-1 then
	-- Unexpected end of file

	-- Find last valid character
	full_blocks=bytes-1
	while full_blocks >= 1 and data[full_blocks] = -1 do
	    full_blocks=full_blocks-1
	end while
	return data[1..full_blocks]
    end if


    return data
end function


global function get_name(sequence file)
   -- Removes the path, leaving only the filename.

    for temp=length(file) to 1 by -1 do
	if file[temp]='\\' or file[temp]='/' or file[temp]=':' then
	    return file[temp+1..length(file)]
	end if
    end for
    return file

end function


global constant LITTLE_ENDIAN = 1, BIG_ENDIAN = 2

-- Reading/writing words and dwords in little-endian or big-endian format.

global function read_word(atom fn, atom format)
    atom byte1,byte2
    byte1 = getc(fn)
    byte2 = getc(fn)
    if byte2=-1 then return -1 end if -- Past end of file

    if format = LITTLE_ENDIAN then
	return byte1 + byte2*#100
    else
	return byte1*#100 + byte2
    end if
end function

global function read_dword(atom fn, atom format)
    atom byte1,byte2, byte3, byte4
    byte1 = getc(fn)
    byte2 = getc(fn)
    byte3 = getc(fn)
    byte4 = getc(fn)


    if byte4=-1 then return -1 end if -- Past end of file

    if format = LITTLE_ENDIAN then
	return byte1 + byte2*#100 + byte3*#10000 + byte4*#1000000
    else
	return byte1*#1000000 + byte2*#10000 + byte3*#100 + byte4
    end if
end function

global procedure write_word(atom fn, atom word, atom format)
    atom a, b
    a = and_bits(word, #FF)
    b = floor(word/#100)

    if format = LITTLE_ENDIAN then
	puts(fn, {a, b} )
    else
	puts(fn, {b, a} )
    end if

end procedure

global procedure write_dword(atom fn, atom word, atom format)
    atom a, b, c, d
    a = and_bits(word, #FF)
    b = and_bits(floor(word/#100), #FF)
    c = and_bits(floor(word/#10000), #FF)
    d = floor(word/#1000000)

    if format = LITTLE_ENDIAN then
	puts(fn, {a, b, c, d} )
    else
	puts(fn, {d, c, b, a} )
    end if

end procedure

